/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: ShmCommentIterator.cpp 1774 2006-10-09 04:13:54Z svn $
 *****************************************************************************/

#include "Environment.h"

#include "ShmCommentIterator.h"
#include "BBSThread.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: ShmCommentIterator.cpp 1774 2006-10-09 04:13:54Z svn $");

/******************************************************************************
 * public メソッド
 *****************************************************************************/
ShmCommentIterator::ShmCommentIterator(BBSThread *bbs_thread,
                                       apr_atomic_t *lock_count,
                                       BBSCommentIterator::range_t *ranges,
                                       apr_size_t range_count)
  : BBSCommentIterator(bbs_thread, ranges, range_count),
    locker_(lock_count)
{
    bbs_thread->increment_view_count();
}


/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_ShmCommentIterator
#include <memory>

#include "TestRunner.h"
#include "BBSThreadReader.h"
#include "TemporaryPool.h"
#include "File.h"
#include "Message.h"

void show_usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <DATA_DIR_PATH> <THREAD_ID>" << endl;
}

static void run_read(apr_pool_t *pool, const char *data_dir_path,
                     apr_size_t thread_id, apr_atomic_t *lock_count,
                     BBSCommentIterator **bcomment_iter)
{
    TemporaryPool temp_pool(pool);
    BBSThreadReader reader(temp_pool.get(), data_dir_path);
    BBSThread *bbs_thread;

    show_test_name("read");

    bbs_thread = BBSThread::get_instance(pool);
    reader.read(thread_id, bbs_thread);

    *bcomment_iter = new ShmCommentIterator(bbs_thread, lock_count);

    show_spacer();
}

static void run_iter(BBSCommentIterator *bcomment_iter)
{
    apr_size_t comment_count;
    BBSThread::bbs_comment_p_t bcomment_p;

    show_test_name("iter");

    comment_count = 0;
    do {
        comment_count++;
        bcomment_iter->get_comment(&bcomment_p);
    } while (bcomment_iter->go_next());

    show_item("comment count", comment_count);
    show_spacer();
}

void run_all(apr_pool_t *pool, int argc, const char * const *argv)
{
    const char *data_dir_path;
    apr_size_t thread_id;
    apr_atomic_t lock_count;
    BBSCommentIterator *bcomment_iter;

    if (argc != 3) {
        THROW(MESSAGE_ARGUMENT_INVALID);
    }

    data_dir_path = argv[1];
    thread_id = atoi(argv[2]);

    if (!is_exist(pool, data_dir_path)) {
        THROW(MESSAGE_DAT_DIR_NOT_FOUND);
    }

    show_item("data dir", data_dir_path);
    show_item("thread id", thread_id);
    show_line();

    apr_atomic_set(&lock_count, 0);
    {
        run_read(pool, data_dir_path, thread_id, &lock_count, &bcomment_iter);
        auto_ptr<BBSCommentIterator> bcomment_iter_ap(bcomment_iter);
        run_iter(bcomment_iter);
    }

    // チェック
    if (lock_count != 0) {
        THROW(MESSAGE_BBS_THREAD_SHOULD_NOT_LOCKED);
    }
}

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
