/******************************************************************************
 * mod_uploader / UploaderConfig.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: UploaderConfig.cpp 426 2005-07-03 19:24:03Z svn $
 *****************************************************************************/

#include "UploaderConfig.h"

#ifdef WIN32
#include <Windows.h>
#else
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_UTIME_H
#include <utime.h>
#endif
#endif

#ifdef DEBUG
#include <iostream>
#endif

#undef DELETE
#define DELETE(ptr) if (ptr != NULL) { delete ptr; ptr = NULL;}
#define CLEAR(pool) if (pool != NULL) { apr_pool_clear(pool); pool = NULL;}

// const char UploaderConfig::MUTEX_TEMPLATE[]             = "mutex.XXXXXX";
const apr_uint64_t UploaderConfig::TOTAL_FILE_SIZE_LIMIT= 10*1024*1024;
const apr_size_t UploaderConfig::TOTAL_FILE_NUMBER_LIMIT= 1000;
const apr_size_t UploaderConfig::MAX_FORM_FILE_SIZE     = 1*1024*1024;
const apr_size_t UploaderConfig::MAX_FORM_TEXT_SIZE     = 1024;
const apr_size_t UploaderConfig::MAX_FORM_ITEM_NUMBER   = 5;
const apr_size_t UploaderConfig::PER_PAGE_ITEM_NUMBER   = 50;
const apr_size_t UploaderConfig::ITEM_RELOAD_THRESHOLD  = 100;

static apr_status_t finalize_config(void *data);
#if APR_HAS_THREADS
static apr_status_t finalize_mutex(void *data);
#endif

/******************************************************************************
 * public ᥽å
 *****************************************************************************/
UploaderConfig::UploaderConfig()
{
    url                     = "";
    file_dir                = NULL;
    tmp_dir                 = NULL;
    lock_dir                = NULL;
    view_tmpl_path          = NULL;
    download_tmpl_path      = NULL;
    thumb_tmpl_path         = NULL;
    error_tmpl_path         = NULL;
    max_file_size           = MAX_FORM_FILE_SIZE;
    total_file_size_limit   = TOTAL_FILE_SIZE_LIMIT;
    total_file_number_limit = TOTAL_FILE_NUMBER_LIMIT;
    per_page_item_number    = PER_PAGE_ITEM_NUMBER;

    item_list               = NULL;
    list_pool               = NULL;

    view_tmpl               = NULL;
    view_tmpl_pool          = NULL;
    download_tmpl           = NULL;
    download_tmpl_pool      = NULL;
    thumb_tmpl              = NULL;
    thumb_tmpl_pool         = NULL;
    error_tmpl              = NULL;
    error_tmpl_pool         = NULL;

    texecutor               = NULL;
    uparser                 = NULL;

#if APR_HAS_THREADS
    mutex                   = NULL;
#endif

    remove_item_nmber       = 0;

    is_init                 = false;
}

UploaderConfig::~UploaderConfig()
{
    finalize();
}

void UploaderConfig::init(apr_pool_t *pool, apr_pool_t *tmp_pool)
{
    if (is_init) {
        return update_list(pool, tmp_pool);
    }

    validate();

#if APR_HAS_THREADS
    init_mutex(pool, tmp_pool);
#endif
    load_list(pool);

    view_tmpl = new page_template();
    download_tmpl = new page_template();
#ifdef MAKE_THUMBNAIL
    thumb_tmpl = new page_template();
#endif
    error_tmpl = new page_template();

    texecutor = new TemplateExecutor();
    uparser = new UploadParser(tmp_dir, MAX_FORM_TEXT_SIZE,
                               max_file_size, MAX_FORM_ITEM_NUMBER,
                               UploadItem::ITEM_HEADER_SIZE);

    apr_pool_cleanup_register(tmp_pool, this, apr_pool_cleanup_null, finalize_config);

    is_init = true;
}

void UploaderConfig::finalize()
{
    DELETE(uparser);
    DELETE(texecutor);
    DELETE(item_list);
    DELETE(view_tmpl);
    DELETE(download_tmpl);
#ifdef MAKE_THUMBNAIL
    DELETE(thumb_tmpl);
#endif
    DELETE(error_tmpl);

    CLEAR(list_pool);
    CLEAR(view_tmpl_pool);
    CLEAR(download_tmpl_pool);
#ifdef MAKE_THUMBNAIL
    CLEAR(thumb_tmpl_pool);
#endif
    CLEAR(error_tmpl_pool);
}

UploaderConfig::page_template *UploaderConfig::get_view_template(apr_pool_t *pool)
{
    if (get_mtime(pool, view_tmpl_path) > view_tmpl->mtime) {
        load_template(pool, &view_tmpl_pool, view_tmpl_path, view_tmpl);
    }

    return view_tmpl;
}

UploaderConfig::page_template *UploaderConfig::get_download_template(apr_pool_t *pool)
{
    if (get_mtime(pool, download_tmpl_path) > download_tmpl->mtime) {
        load_template(pool, &download_tmpl_pool, download_tmpl_path, download_tmpl);
    }

    return download_tmpl;
}

UploaderConfig::page_template *UploaderConfig::get_thumb_template(apr_pool_t *pool)
{
    if (get_mtime(pool, thumb_tmpl_path) > thumb_tmpl->mtime) {
        load_template(pool, &thumb_tmpl_pool, thumb_tmpl_path, thumb_tmpl);
    }

    return thumb_tmpl;
}

UploaderConfig::page_template *UploaderConfig::get_error_template(apr_pool_t *pool)
{
    if (get_mtime(pool, error_tmpl_path) > error_tmpl->mtime) {
        load_template(pool, &error_tmpl_pool, error_tmpl_path, error_tmpl);
    }

    return error_tmpl;
}

UploadItemReader *UploaderConfig::get_ureader(apr_pool_t *pool)
{
    return new UploadItemReader(pool, file_dir);
}

UploadItemWriter *UploaderConfig::get_uwriter(apr_pool_t *pool)
{
    return new UploadItemWriter(pool, file_dir);
}

apr_status_t UploaderConfig::lock()
{
#if APR_HAS_THREADS
    return apr_thread_mutex_lock(mutex);
#else
    return APR_SUCCESS;
#endif
}

apr_status_t UploaderConfig::unlock()
{
#if APR_HAS_THREADS
    return apr_thread_mutex_unlock(mutex);
#else
    return APR_SUCCESS;
#endif
}


/******************************************************************************
 * private ᥽å
 *****************************************************************************/
void UploaderConfig::validate()
{
    if ((file_dir == NULL) ||
#ifdef MAKE_THUMBNAIL
        (thumb_dir == NULL) ||
#endif
        (tmp_dir == NULL) ||
        (lock_dir == NULL) ||
        (view_tmpl_path == NULL) || (download_tmpl_path == NULL) ||
        (thumb_tmpl_path == NULL) || (error_tmpl_path == NULL)) {
        throw "ϳ줬ޤ";
    }
}

#if APR_HAS_THREADS
void UploaderConfig::init_mutex(apr_pool_t *pool, apr_pool_t *tmp_pool)
{
    mutex = create_mutex(pool, tmp_pool, lock_dir);
}
#endif

void UploaderConfig::load_list(apr_pool_t *pool)
{
    DELETE(item_list);
    CLEAR(list_pool);

    if (apr_pool_create(&list_pool, pool) != APR_SUCCESS) {
        throw "åץɥƥΥꥹѤΥݤ˼Ԥޤ";
    }

    item_list = UploadItemList::load(list_pool, file_dir, thumb_dir,
                                     total_file_number_limit, total_file_size_limit);
}

void UploaderConfig::update_list(apr_pool_t *pool, apr_pool_t *tmp_pool)
{
    try {
        // äȴɡΨͤƤΰ֤ǥå(äѤᡩ)
        lock();
        if (get_mtime(tmp_pool, file_dir) > item_list->get_mtime()) {

            load_list(pool);

            apr_time_t mtime = item_list->get_mtime();

#ifdef WIN32
            FILETIME time;
            HANDLE file = CreateFile(file_dir,
                                     GENERIC_WRITE, 0, 0, OPEN_EXISTING,
                                     FILE_ATTRIBUTE_NORMAL, NULL);

            mtime += APR_TIME_C(11644473600000000);
            mtime *= 10;
            time.dwLowDateTime = static_cast<DWORD>(mtime & 0xFFFFFFFF);
            time.dwHighDateTime = static_cast<DWORD>(mtime >> 32);

            if (file != INVALID_HANDLE_VALUE) {
                SetFileTime(file, NULL, &time, &time);
                CloseHandle(file);
            }
#elif HAVE_UTIME
            struct utimbuf time;
            time.modtime = time.actime = mtime/1000000;
            utime(file_dir, &time);
#endif
        } else if (remove_item_nmber++ == ITEM_RELOAD_THRESHOLD) {
            load_list(pool);
            remove_item_nmber = 0;
        }
        unlock();

        return;
    } catch(const char *) {
        unlock();
        throw;
    }
}

void UploaderConfig::load_template(apr_pool_t *pool, apr_pool_t **tmpl_pool,
                                   const char *template_path, page_template *tmpl)
{
    apr_file_t *file;
    apr_mmap_t *file_map = NULL;
    apr_finfo_t info;

    if (apr_file_open(&file, template_path, APR_READ,
                      APR_OS_DEFAULT, pool) != APR_SUCCESS) {
        throw "ƥץ졼Ȥ open ˼Ԥޤ";
    }

    try {
        if (apr_file_info_get(&info, APR_FINFO_SIZE|APR_FINFO_MTIME, file) != APR_SUCCESS) {
            throw "ƥץ졼ȤΥǤޤǤ";
        }

        if (apr_mmap_create(&file_map, file, 0, static_cast<apr_size_t>(info.size), APR_MMAP_READ,
                            pool) != APR_SUCCESS) {
            throw "ƥץ졼Ȥ mmap ˼Ԥޤ";
        }

        DELETE(tmpl->imap);
        CLEAR(*tmpl_pool);

        if (apr_pool_create(tmpl_pool, pool) != APR_SUCCESS) {
            throw "ƥץ졼ѤΥݤ˼Ԥޤ";
        }

        TemplateLexer lexer(*tmpl_pool, reinterpret_cast<const char *>(file_map->mm),
                            static_cast<apr_size_t>(info.size));
        TemplateParser parser(*tmpl_pool);

        tmpl->imap = new TemplateLexer::ident_map(*(lexer.get_ident_map()));
        tmpl->node = parser.parse(lexer.get_token_list(), lexer.get_ident_map());
        tmpl->mtime = info.mtime;

        apr_mmap_delete(file_map);
        apr_file_close(file);

    } catch(const char *) {
        if (file_map != NULL) {
            apr_mmap_delete(file_map);
        }
        apr_file_close(file);

        throw;
    }
}

apr_time_t UploaderConfig::get_mtime(apr_pool_t *pool, const char *file_path)
{
    apr_finfo_t info;

    if (apr_stat(&info, file_path, APR_FINFO_MTIME, pool) != APR_SUCCESS) {
        throw "եޤϥǥ쥯ȥιǤޤǤ";
    }

    return info.mtime;
}

#if APR_HAS_THREADS
apr_thread_mutex_t *UploaderConfig::create_mutex(apr_pool_t *pool, apr_pool_t *tmp_pool,
                                                 const char *lock_dir)
{
    apr_thread_mutex_t *mutex;
//     char *lock_name;
//     apr_file_t *file;

//     if (apr_filepath_merge(&lock_name, lock_dir, name,
//                            APR_FILEPATH_NOTABOVEROOT, tmp_pool) != APR_SUCCESS) {
//         throw "mutex ̾ǤޤǤ(1)";
//     }

//     // Ӥߤ硼
//     if (apr_file_mktemp(&file, lock_name,
//                         APR_CREATE|APR_WRITE|APR_EXCL|APR_DELONCLOSE,
//                         pool) != APR_SUCCESS) {
//         throw "mutex ̾ǤޤǤ(2)";
//     }
//     apr_file_close(file);

    if (apr_thread_mutex_create(&mutex, APR_THREAD_MUTEX_DEFAULT, pool) != APR_SUCCESS) {
        throw "mutex ǤޤǤ";
    }

    apr_pool_cleanup_register(pool, mutex, apr_pool_cleanup_null, finalize_mutex);

    return mutex;
}
#endif


/******************************************************************************
 * Хåؿ
 *****************************************************************************/
apr_status_t finalize_config(void *data)
{
    UploaderConfig *config;

    config = reinterpret_cast<UploaderConfig *>(data);
    config->finalize();

    return APR_SUCCESS;
}

#if APR_HAS_THREADS
apr_status_t finalize_mutex(void *data)
{
    apr_thread_mutex_t *mutex;

    mutex = reinterpret_cast<apr_thread_mutex_t *>(data);
    apr_thread_mutex_destroy(mutex);

    return APR_SUCCESS;
}
#endif

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
