 /******************************************************************************
 * mod_uploader / mod_uploader.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: mod_uploader.cpp 431 2005-07-05 03:36:31Z svn $
 *****************************************************************************/

#include "mod_uploader.h"

#include "UploadItemCreator.h"
#include "TemplateVariableCreator.h"
#include "RssView.h"
#include "CharCodeConverter.h"
#ifdef MAKE_THUMBNAIL
#include "ThumbnailWriter.h"
#endif
#include "Auxiliary.h"

#include "http_core.h"

#include <stdlib.h>
#include <string.h>

#define MATCH(str, pat) (strstr(str, pat) != 0)

#ifdef WIN32
#define UCHAR(c) static_cast<const unsigned char>(c)
#else
#define UCHAR(c) c
#endif

#undef ERROR // for Visual C++
#ifdef DEBUG
#define ERROR(format, str) fprintf(stderr, format, str); fputc('\n', stderr)
#else
#define ERROR(format, str) ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, format, str)
#endif

static int command_handler(request_rec *r, uconfig *config, const char *arg);
static int view(request_rec *r, uconfig *config, const char *arg);
static int rss(request_rec *r, uconfig *config, const char *arg);
static int upload(request_rec *r, uconfig *config, const char *arg);
static int download(request_rec *r, uconfig *config, const char *arg);
static int remove(request_rec *r, uconfig *config, const char *arg);
static int error(request_rec *r, uconfig *config, const char *message);

static int input_download_pass(request_rec *r, uconfig *config, const char *arg,
                               UploadItem::header *header);
static int redirect_top(request_rec *r);
static uconfig *get_config(request_rec *r);
static apr_table_t *get_command_table(request_rec *r);
static variable *create_variable(apr_pool_t *pool, const char *str);
static variable *create_variable(apr_pool_t *pool, int i);
static variable_map *get_view_vmap(apr_pool_t *pool, UploaderConfig *config,
                                   page_template *tmpl, apr_size_t page);
static variable_map *get_download_vmap(apr_pool_t *pool, UploaderConfig *config,
                                       page_template *tmpl, UploadItem::header *header);
#ifdef MAKE_THUMBNAIL
static int thumbnail(request_rec *r, uconfig *config, const char *arg);
static variable_map *get_thumb_vmap(apr_pool_t *pool, UploaderConfig *config,
                                    page_template *tmpl, apr_size_t page);
#endif
static apr_size_t get_page(request_rec *r);
// static bool is_accept_gzip(request_rec *r);
static const char *get_name_encoding(request_rec *r);

int command_handler(request_rec *r, uconfig *config, const char *arg)
{
    if (*arg == '\0') {
        apr_table_set(r->headers_out, "Location", apr_pstrcat(r->pool, r->uri, "/", NULL));

        return HTTP_TEMPORARY_REDIRECT;
    }
    arg++;

    const char *command = ap_getword(r->pool, &arg, '/');

    if (*command == '\0') {
        return view(r, config, arg);
    } else if (strcmp(command, RSS_COMMAND) == 0) {
        return rss(r, config, arg);
    } else if (strcmp(command, UPLOAD_COMMAND) == 0) {
        return upload(r, config, arg);
    } else if (strcmp(command, DOWNLOAD_COMMAND) == 0) {
        return download(r, config, arg);
    } else if (strcmp(command, REMOVE_COMMAND) == 0) {
        return remove(r, config, arg);
#ifdef MAKE_THUMBNAIL
    } else if (strcmp(command, THUMBNAIL_COMMAND) == 0) {
        return thumbnail(r, config, arg);
#endif
    } else {
        return error(r, config, "ݡȤƤʤޥɤǤ");
    }
}

int view(request_rec *r, uconfig *config, const char *arg)
{
    variable_map *vmap = NULL;
    apr_status_t status;

    config->lock();

    ap_update_mtime(r, config->item_list->get_mtime());
    ap_set_last_modified(r);
    ap_set_etag(r);

    if ((status = ap_meets_conditions(r)) != OK) {
        config->unlock();
        return status;
    }

    ap_set_content_type(r, "text/html");
    if (r->header_only) {
        config->unlock();
        return OK;
    }

    try {
        page_template *tmpl = config->get_view_template(r->server->process->pool);

        vmap = get_view_vmap(r->pool, config, tmpl, get_page(r));

        config->texecutor->exec(r->pool, r, tmpl->node, tmpl->imap, vmap);

        config->unlock();

        return OK;
    } catch(const char *message) {
        config->unlock();

        if (vmap != NULL) {
            delete vmap;
        }

        ap_rputs("<div class=\"warning\">", r);
        ap_rputs(message, r);
        ap_rputs("</div>", r);

        return OK;
    }
}

#ifdef MAKE_THUMBNAIL
int thumbnail(request_rec *r, uconfig *config, const char *arg)
{
    variable_map *vmap = NULL;
    apr_status_t status;

    config->lock();

    ap_update_mtime(r, config->item_list->get_mtime());
    ap_set_last_modified(r);
    ap_set_etag(r);

    if ((status = ap_meets_conditions(r)) != OK) {
        config->unlock();
        return status;
    }

    ap_set_content_type(r, "text/html");
    if (r->header_only) {
        config->unlock();
        return OK;
    }

    try {
        page_template *tmpl = config->get_thumb_template(r->server->process->pool);

        vmap = get_thumb_vmap(r->pool, config, tmpl, get_page(r));

        config->texecutor->exec(r->pool, r, tmpl->node, tmpl->imap, vmap);

        config->unlock();

        return OK;
    } catch(const char *message) {
        config->unlock();

        if (vmap != NULL) {
            delete vmap;
        }

        ap_rputs("<div class=\"warning\">", r);
        ap_rputs(message, r);
        ap_rputs("</div>", r);

        return OK;
    }
}
#endif

int rss(request_rec *r, uconfig *config, const char *arg)
{
    apr_status_t status;

    config->lock();

    ap_update_mtime(r, config->item_list->get_mtime());
    ap_set_last_modified(r);
    ap_set_etag(r);

    if ((status = ap_meets_conditions(r)) != OK) {
        config->unlock();
        return status;
    }

    ap_set_content_type(r, RssView::content_type);
    if (r->header_only) {
        config->unlock();
        return OK;
    }

    try {
        RssView view;

        view.exec(r->pool, r, config->url, config->item_list,
                  config->per_page_item_number);

        config->unlock();

        return OK;
    } catch(const char *message) {
        config->unlock();

        ap_rputs("<div class=\"warning\">", r);
        ap_rputs(message, r);
        ap_rputs("</div>", r);

        return OK;
    }
}

int upload(request_rec *r, uconfig *config, const char *arg)
{
    apr_status_t status;
    UploadParser::query_map *qmap = NULL;
    UploadItemWriter *writer = NULL;

    if (r->method_number != M_POST) {
        return HTTP_BAD_REQUEST;
    }
    if ((status = ap_setup_client_block(r, REQUEST_CHUNKED_ERROR)) != OK) {
        return status;
    }
    if (!ap_should_client_block(r)) {
        return HTTP_NO_CONTENT;
    }

    try {
        UploadItem::header *header;
        const char *tmp_path;
        const char *file_name;

        qmap = config->uparser->parse(r->pool, r, apr_table_get(r->headers_in, "Content-Type"));
        header = UploadItemCreator::create_header(r->pool, qmap, &tmp_path);

        writer = config->get_uwriter(r->pool);
        file_name = writer->write(header, tmp_path);

        try {
            config->lock();
            config->item_list->add(file_name, true);
            config->unlock();
        } catch(const char *) {
            config->unlock();
            throw;
        }

        UploadParser::clear_map(qmap);

        delete qmap;
        delete writer;

        return redirect_top(r);
    } catch(const char *message) {
        if (qmap != NULL) {
            UploadParser::clear_map(qmap);
            delete qmap;
        }
        if (writer != NULL) {
            delete writer;
        }

        return error(r, config, message);
    }
}

int download(request_rec *r, uconfig *config, const char *arg)
{
    const char *name;
    UploadItemReader *reader = NULL;

    name = ap_getword(r->pool, &arg, '/');
    if (name == NULL) {
        return HTTP_BAD_REQUEST;
    }

    {
        // ե̾Υå
        const char *pos = name;

        while (isalnum(UCHAR(*pos)) || (*pos == '.')) {
            pos++;
        }

        if ((static_cast<size_t>(pos-name) != strlen(name)) ||
            (strlen(name) > APR_PATH_MAX)) {
            return HTTP_FORBIDDEN;
        }
    }

    try {
        apr_file_t *file;
        UploadItem::header *header;
        const char *file_name;
        const char *password;
        const char *open_mode;
        apr_size_t size;
        apr_status_t status;

        reader = config->get_ureader(r->pool);
        header = reader->read(name, &file);

        password = apr_table_get(get_command_table(r), UploadItem::DOWNLOAD_PASS_PARAM);

        // DL pass Υå
        if ((strlen(header->download_pass) > 0) &&
            ((password == NULL) ||
             (strncmp(header->download_pass, password, strlen(header->download_pass)) != 0))) {
            int result = input_download_pass(r, config, arg, header);

            delete reader;

            return result;
        }

        ap_update_mtime(r, header->time);
        ap_set_last_modified(r);
        ap_set_etag(r);

        if ((status = ap_meets_conditions(r)) != OK) {
            return status;
        }

        ap_set_content_type(r, header->file_mime);

        if (r->header_only) {
            return OK;
        }

        file_name = CharCodeConverter::convert(r->pool,
                                               header->file_name,
                                               CharCodeConverter::DEFAULT_CODE,
                                               get_name_encoding(r));

        // ɽ⡼ or ɥ⡼
        open_mode = (r->args == NULL) ? "inline" : "attachment";

        ap_set_content_length(r, header->file_size);
        apr_table_setn(r->headers_out, "Content-Disposition",
                       apr_psprintf(r->pool,
                                    "%s; filename=\"%s\"", open_mode, file_name));

        ap_send_fd(file, r, UploadItem::ITEM_HEADER_SIZE, header->file_size, &size);

        delete reader;

        return OK;
    } catch(const char *message) {
        if (reader != NULL) {
            delete reader;
        }

        return error(r, config, message);
    }
}

int input_download_pass(request_rec *r, uconfig *config, const char *arg,
                        UploadItem::header *header)
{
    variable_map *vmap = NULL;

    try {
        page_template *tmpl = config->get_download_template(r->server->process->pool);

        vmap = get_download_vmap(r->pool, config, tmpl, header);

        ap_set_content_type(r, "text/html");
        config->texecutor->exec(r->pool, r, tmpl->node, tmpl->imap, vmap);

        delete vmap;

        return OK;
    } catch(const char *message) {
        if (vmap != NULL) {
            delete vmap;
        }

        ap_rputs("<div class=\"warning\">", r);
        ap_rputs(message, r);
        ap_rputs("</div>", r);

        return OK;
    }
}

int remove(request_rec *r, uconfig *config, const char *arg)
{
    apr_table_t *ctable;

    if (r->method_number != M_POST) {
        return HTTP_BAD_REQUEST;
    }

    try {
        const char *file;
        const char *password;

        ctable = get_command_table(r);

        file = apr_table_get(ctable, UploadItem::FILE_PARAM);
        password = apr_table_get(ctable, UploadItem::REMOVE_PASS_PARAM);

        if ((file == NULL) || (password == NULL)) {
            return error(r, config, "եޤϥѥɤꤵƤޤ");
        }

        try {
            config->lock();

            config->item_list->remove(file, password);
            config->remove_item_nmber++;

            config->unlock();
        } catch(const char *) {
            config->unlock();
            throw;
        }

        return redirect_top(r);
    } catch(const char *message) {
        return error(r, config, message);
    }
}

int error(request_rec *r, uconfig *config, const char *message)
{
    try {
        page_template *tmpl = config->get_error_template(r->server->process->pool);
        apr_size_t ident_id;

        TemplateExecutor::variable_map vmap(tmpl->imap->size(), NULL);

        ident_id = TemplateExecutor::get_ident_id(tmpl->imap, ERROR_MESSAGE_KEY);
        vmap.at(ident_id) = TemplateVariableCreator::create(r->pool, message);

        ap_set_content_type(r, "text/html");

        config->texecutor->exec(r->pool, r, tmpl->node, tmpl->imap, &vmap);

        return OK;
    } catch(const char *message) {
        ERROR("Exception: %s", message);

        return HTTP_INTERNAL_SERVER_ERROR;
    }
}

int redirect_top(request_rec *r)
{
    ap_set_content_type(r, "text/html");

    ap_rputs("<?xml version=\"1.0\" encoding=\"EUC-JP\"?>", r);
    ap_rputs("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">", r);
    ap_rputs("<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"ja\" >", r);
    ap_rputs(" <head>", r);
    ap_rputs("  <meta http-equiv=\"content-type\" content=\"application/xhtml+xml\" />", r);
    ap_rputs("  <meta http-equiv=\"refresh\" content=\"0;url=../\" />", r);
    ap_rputs("  <title>mod_uploader</title>", r);
    ap_rputs(" </head>", r);
    ap_rputs(" <body />", r);
    ap_rputs("</html>", r);

    return OK;
}

uconfig *get_config(request_rec *r)
{
    return reinterpret_cast<uconfig *>(ap_get_module_config(r->per_dir_config,
                                                            &uploader_module));
}

apr_table_t *get_command_table(request_rec *r)
{
    const char *command;
    const char *token;
    apr_table_t *table;

    table = apr_table_make(r->pool, MAX_COMMAND_NUMBER);
    command = reinterpret_cast<char *>(apr_pcalloc(r->pool, MAX_COMMAND_LENGTH));

    if ((ap_setup_client_block(r, REQUEST_CHUNKED_ERROR) != OK) ||
        !ap_should_client_block(r)) {
        return table;
    }

    ap_get_client_block(r, const_cast<char *>(command), MAX_COMMAND_LENGTH-1);
    while ((*command != '\0') && (token = ap_getword(r->pool, &command, '&'))) {
        char *key;
        char *value;

        key = ap_getword(r->pool, &token, '=');
        value = ap_getword(r->pool, &token, '=');
        ap_unescape_url(key);
        ap_unescape_url(value);

        apr_table_set(table, key, value);
    }

    return table;
}

variable *create_variable(apr_pool_t *pool, const char *str)
{
    variable *var = reinterpret_cast<variable *>(apr_palloc(pool,
                                                            sizeof(variable)));
    var->type = TemplateExecutor::SCALAR;
    var->s_val = reinterpret_cast<scalar *>(apr_palloc(pool, sizeof(scalar)));
    var->s_val->type = TemplateExecutor::STRING;
    var->s_val->s_val = str;

    return var;
}

variable *create_variable(apr_pool_t *pool, int i)
{
    variable *var = reinterpret_cast<variable *>(apr_palloc(pool,
                                                            sizeof(variable)));
    var->type = TemplateExecutor::SCALAR;
    var->s_val = reinterpret_cast<scalar *>(apr_palloc(pool, sizeof(scalar)));
    var->s_val->type = TemplateExecutor::INTEGER;
    var->s_val->i_val = i;

    return var;
}

variable_map *get_view_vmap(apr_pool_t *pool, UploaderConfig *config,
                            page_template *tmpl, apr_size_t page)
{
    apr_size_t ident_id;
    variable_map *vmap;
    apr_size_t page_number;

    vmap = new variable_map(tmpl->imap->size()+SYSTEM_KEY_NUMBER, NULL);

    ident_id = TemplateExecutor::get_ident_id(tmpl->imap, FILE_LIST_KEY);
    vmap->at(ident_id) = config->item_list->to_varray(pool,
                                                      (page-1)*config->per_page_item_number,
                                                      config->per_page_item_number);

    ident_id = TemplateExecutor::get_ident_id(tmpl->imap, CURRENT_PAGE_KEY);
    vmap->at(ident_id) = create_variable(pool, static_cast<int>(page));

    page_number = (config->item_list->size() == 0)
        ? 1
        : (config->item_list->size()-1)/config->per_page_item_number + 1;
    ident_id = TemplateExecutor::get_ident_id(tmpl->imap, PAGE_NUMBER_KEY);
    vmap->at(ident_id) = create_variable(pool, static_cast<int>(page_number));

    ident_id = TemplateExecutor::get_ident_id(tmpl->imap, TOTAL_FILE_SIZE_KEY);
    vmap->at(ident_id) = create_variable(pool,
                                         size_str(pool, config->item_list->total_file_size()));

    ident_id = TemplateExecutor::get_ident_id(tmpl->imap, TOTAL_FILE_SIZE_KEY);
    vmap->at(ident_id) = create_variable(pool,
                                         size_str(pool, config->item_list->total_file_size()));

     ident_id = TemplateExecutor::get_ident_id(tmpl->imap, TOTAL_FILE_NUMBER_KEY);
     vmap->at(ident_id) = create_variable(pool,
                                          comma_str(pool, config->item_list->size()));

     ident_id = TemplateExecutor::get_ident_id(tmpl->imap, MAX_FILE_SIZE_KEY);
     vmap->at(ident_id) = create_variable(pool,
                                          size_str(pool, config->max_file_size));

     ident_id = TemplateExecutor::get_ident_id(tmpl->imap, TOTAL_FILE_SIZE_LIMIT_KEY);
     vmap->at(ident_id) = create_variable(pool,
                                          size_str(pool, config->total_file_size_limit));

    return vmap;
}

variable_map *get_download_vmap(apr_pool_t *pool, UploaderConfig *config,
                                page_template *tmpl, UploadItem::header *header)
{
    apr_size_t ident_id;
    variable_map *vmap;
    UploadItem item(pool);

    vmap = new variable_map(tmpl->imap->size()+SYSTEM_KEY_NUMBER, NULL);
    ident_id = TemplateExecutor::get_ident_id(tmpl->imap, FILE_KEY);
    item.set_data("", header); // ʤ
    vmap->at(ident_id) = TemplateVariableCreator::create(pool, &item);

    return vmap;
}

#ifdef MAKE_THUMBNAIL
variable_map *get_thumb_vmap(apr_pool_t *pool, UploaderConfig *config,
                            page_template *tmpl, apr_size_t page)
{
    apr_size_t ident_id;
    variable_map *vmap;
    apr_size_t page_number;

    vmap = new variable_map(tmpl->imap->size()+SYSTEM_KEY_NUMBER, NULL);

    ident_id = TemplateExecutor::get_ident_id(tmpl->imap, THUMB_LIST_KEY);
    vmap->at(ident_id) = config->item_list->to_thumb_varray(pool,
                                                            (page-1)*config->per_page_item_number,
                                                            config->per_page_item_number);

    ident_id = TemplateExecutor::get_ident_id(tmpl->imap, CURRENT_PAGE_KEY);
    vmap->at(ident_id) = create_variable(pool, static_cast<int>(page));

    page_number = (config->item_list->thumb_size() == 0)
        ? 1
        : (config->item_list->thumb_size()-1)/config->per_page_item_number + 1;
    ident_id = TemplateExecutor::get_ident_id(tmpl->imap, PAGE_NUMBER_KEY);
    vmap->at(ident_id) = create_variable(pool, static_cast<int>(page_number));

    ident_id = TemplateExecutor::get_ident_id(tmpl->imap, THUMB_NUMBER_KEY);
    vmap->at(ident_id) = create_variable(pool,
                                         comma_str(pool, config->item_list->thumb_size()));

    return vmap;
}
#endif

apr_size_t get_page(request_rec *r)
{
    int page = 1;
    const char *args = r->args;

    if (args != NULL) {
        page = atoi(args);

        if (page < 0) {
            page = 0;
        }
    }

    return static_cast<apr_size_t>(page);
}

// bool is_accept_gzip(request_rec *r)
// {
//     const char *accept_enc;

//     accept_enc = apr_table_get(r->headers_in, ACCEPT_ENCODING);

//     return strstr(accept_enc, GZIP) != NULL;
// }

const char *get_name_encoding(request_rec *r)
{
    const char *agent = apr_table_get(r->headers_in, "User-Agent");

    // 㤷Ū

    if (MATCH(agent, "compatible; MSIE") || MATCH(agent, "Sleipnir")) {
        return IE_NAME_CODE;
    } else if (MATCH(agent, "Firefox") || MATCH(agent, "Mozilla")) {
        return FIREFOX_NAME_CODE;
    } else {
        return CharCodeConverter::DEFAULT_CODE;
    }
}

extern "C" {
static int uploader_handler(request_rec *r)
{
    uconfig *config;

    if (strcmp(r->handler, "uploader")) {
        return DECLINED;
    }
    config = get_config(r);

    try {
        config->init(r->server->process->pool, r->pool);
    } catch(const char *message) {
        ap_rputs("<div class=\"warning\">", r);
        ap_rputs(message, r);
        ap_rputs("</div>", r);

        ERROR("Exception: %s", message);

        return HTTP_INTERNAL_SERVER_ERROR;
    }

    return command_handler(r, config, r->path_info);
}

static apr_status_t finalize_dir_config(void *data)
{
    uconfig *config;

    config = reinterpret_cast<uconfig *>(data);
    config->finalize();

    return APR_SUCCESS;
}

static void *create_dir_config(apr_pool_t *p, char *dirspec)
{
    uconfig *config;

    config = reinterpret_cast<uconfig *>(apr_pcalloc(p, sizeof(uconfig)));
    new(config) uconfig;

    apr_pool_cleanup_register(p, config, finalize_dir_config, apr_pool_cleanup_null);

    return reinterpret_cast<uconfig *>(config);
}

static int post_config(apr_pool_t *pconf, apr_pool_t *plog,
                       apr_pool_t *ptemp, server_rec *s)
{
    ap_add_version_component(pconf, PACKAGE_NAME "/" PACKAGE_VERSION);

    return OK;
}

static const char *set_url(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->url = apr_pstrdup(parms->pool, arg);

    return NULL;
}

static const char *set_file_dir(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->file_dir = apr_pstrdup(parms->pool, arg);

    return NULL;
}

static const char *set_thumb_dir(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->thumb_dir = apr_pstrdup(parms->pool, arg);

    return NULL;
}

static const char *set_tmp_dir(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->tmp_dir = apr_pstrdup(parms->pool, arg);

    return NULL;
}

static const char *set_lock_dir(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->lock_dir = apr_pstrdup(parms->pool, arg);

    return NULL;
}

static const char *set_view_tmpl_path(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->view_tmpl_path = apr_pstrdup(parms->pool, arg);

    return NULL;
}

static const char *set_download_tmpl_path(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->download_tmpl_path = apr_pstrdup(parms->pool, arg);

    return NULL;
}

static const char *set_thumb_tmpl_path(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->thumb_tmpl_path = apr_pstrdup(parms->pool, arg);

    return NULL;
}

static const char *set_error_tmpl_path(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->error_tmpl_path = apr_pstrdup(parms->pool, arg);

    return NULL;
}

static const char *set_max_file_size(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->max_file_size = atoi(arg) * 1024;

    return NULL;
}

static const char *set_total_file_size_limit(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->total_file_size_limit = atoi(arg) * 1024;

    return NULL;
}

static const char *set_total_file_number_limit(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->total_file_number_limit = atoi(arg);

    return NULL;
}

static const char *set_per_page_item_number(cmd_parms *parms, void *mconfig, char *arg)
{
    uconfig *config = reinterpret_cast<uconfig *>(mconfig);

    config->per_page_item_number = atoi(arg);

    return NULL;
}

static const command_rec uploader_cmds[] = {
    AP_INIT_TAKE1(URL_PARAM,
                  reinterpret_cast<const char*(*)()>(set_url), NULL,
                  ACCESS_CONF, "Uploader URL"),
    AP_INIT_TAKE1(FILE_DIRECTORY_PARAM,
                  reinterpret_cast<const char*(*)()>(set_file_dir), NULL,
                  ACCESS_CONF, "Upload File Directory"),
    AP_INIT_TAKE1(THUMB_DIRECTORY_PARAM,
                  reinterpret_cast<const char*(*)()>(set_thumb_dir), NULL,
                  ACCESS_CONF, "Thumbnail File Directory"),
    AP_INIT_TAKE1(TMP_DIRECTORY_PARAM,
                  reinterpret_cast<const char*(*)()>(set_tmp_dir), NULL,
                  ACCESS_CONF, "Temporary File Directory"),
    AP_INIT_TAKE1(LOCK_DIRECTORY_PARAM,
                  reinterpret_cast<const char*(*)()>(set_lock_dir), NULL,
                  ACCESS_CONF, "Lock File Directory"),
    AP_INIT_TAKE1(VIEW_TEMPLATE_FILE_PARAM,
                  reinterpret_cast<const char*(*)()>(set_view_tmpl_path), NULL,
                  ACCESS_CONF, "View Template File"),
    AP_INIT_TAKE1(DOWNLOAD_TEMPLATE_FILE_PARAM,
                  reinterpret_cast<const char*(*)()>(set_download_tmpl_path), NULL,
                  ACCESS_CONF, "Download Template File"),
    AP_INIT_TAKE1(THUMB_TEMPLATE_FILE_PARAM,
                  reinterpret_cast<const char*(*)()>(set_thumb_tmpl_path), NULL,
                  ACCESS_CONF, "Thumbnail Template File"),
    AP_INIT_TAKE1(ERROR_TEMPLATE_FILE_PARAM,
                  reinterpret_cast<const char*(*)()>(set_error_tmpl_path), NULL,
                  ACCESS_CONF, "Error Template File"),
    AP_INIT_TAKE1(MAX_FILE_SIZE_PARAM,
                  reinterpret_cast<const char*(*)()>(set_max_file_size), NULL,
                  ACCESS_CONF, "Max File Size (KB)"),
    AP_INIT_TAKE1(TOTAL_FILE_SIZE_LIMIT_PARAM,
                  reinterpret_cast<const char*(*)()>(set_total_file_size_limit), NULL,
                  ACCESS_CONF, "Total File Size Limit (KB)"),
    AP_INIT_TAKE1(TOTAL_FILE_NUMBER_LIMIT_PARAM,
                  reinterpret_cast<const char*(*)()>(set_total_file_number_limit), NULL,
                  ACCESS_CONF, "Total File Number Limit"),
    AP_INIT_TAKE1(PER_PAGE_ITEM_NUMBER_PARAM,
                  reinterpret_cast<const char*(*)()>(set_per_page_item_number), NULL,
                  ACCESS_CONF, "Per Page Item Number"),
    { NULL }
};

static void uploader_register_hooks(apr_pool_t *p)
{
    ap_hook_post_config(post_config, NULL, NULL, APR_HOOK_REALLY_FIRST);
    ap_hook_handler(uploader_handler, NULL, NULL, APR_HOOK_MIDDLE);
}

module AP_MODULE_DECLARE_DATA uploader_module = {
    STANDARD20_MODULE_STUFF,
    create_dir_config,
    NULL,
    NULL,
    NULL,
    uploader_cmds,
    uploader_register_hooks
};
}

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
