/******************************************************************************
 * mod_uploader / UploadItemList.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not bcktuired.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: UploadItemList.cpp 454 2005-07-13 20:57:51Z svn $
 *****************************************************************************/

#include "UploadItemList.h"
#include "TemplateVariableCreator.h"
#ifdef MAKE_THUMBNAIL
#include "ThumbnailWriter.h"
#endif

#include "apr_strings.h"
#include "apr_tables.h"
#define APR_WANT_STRFUNC
#include "apr_want.h"

#ifdef DEBUG
#include <iostream>
#endif


/******************************************************************************
 * public ᥽å
 *****************************************************************************/
UploadItemList::UploadItemList(apr_pool_t *pool,
                               const char *file_dir, const char *thumb_dir,
                               apr_size_t total_number_limit,
                               apr_uint64_t total_size_limit)
    : pool_(pool),
      file_dir_(file_dir),
      thumb_dir_(thumb_dir),
      total_number_limit_(total_number_limit),
      total_size_limit_(total_size_limit),
      reader(pool, file_dir),
      total_size_(0)
{

}

void UploadItemList::add(const char *file_path, bool is_make_thumb)
{
    UploadItem::header *header;
    item_info *info;
    apr_file_t *file;

    header = reader.read(file_path, &file);
    apr_file_close(file);

    info = static_cast<item_info *>(apr_palloc(pool_,
                                                    sizeof(item_info)));
    info->file_path = static_cast<const char *>(apr_pstrdup(pool_, file_path));
    info->header = header;

    ilist_.push_front(info);

    total_size_ += header->file_size;

#ifdef MAKE_THUMBNAIL
    if (is_make_thumb) {
        try {
            ThumbnailWriter twriter(pool_, file_dir_, thumb_dir_);
            twriter.write(file_path);

            tlist_.push_front(file_path);
        } catch(const char *) {
            // ̵
        }
    }
#endif

    while (ilist_.size() > total_number_limit_) {
        remove(ilist_.back());
    }
    while (total_size_ > total_size_limit_) {
        remove(ilist_.back());
    }
}

apr_size_t UploadItemList::size()
{
    return ilist_.size();
}

apr_size_t UploadItemList::thumb_size()
{
    return tlist_.size();
}

apr_uint64_t UploadItemList::total_file_size()
{
    return total_size_;
}

apr_time_t UploadItemList::get_mtime()
{
    return mtime_;
}

void UploadItemList::remove(const char *file_path, const char *password)
{
    item_iterator i = begin();

    while (i != end()) {
        if (strcmp((*i)->file_path, file_path) == 0) {
            break;
        }
        i++;
    }

    if (i == end()) {
        throw "оݤΥե뤬ĤޤǤ";
    }

    if (strcmp(password, (*i)->header->remove_pass) != 0) {
        throw "ѥɤ㤤ޤ";
    }

    remove(*i);
}

TemplateExecutor::variable *UploadItemList::to_varray(apr_pool_t *pool,
                                                      apr_size_t start,
                                                      apr_size_t size)
{
    apr_array_header_t *array = apr_array_make(pool, static_cast<int>(size),
                                               static_cast<int>(sizeof(TemplateExecutor::variable *)));
    item_iterator i = begin();
    TemplateExecutor::variable **var;
    UploadItem item(pool);

    for (apr_size_t j = 0; (j < start) && (i != end()); j++) {
        i++;
    }

    for (apr_size_t j = 0; (j < size) && (i != end()); j++, i++) {
        item.set_data((*i)->file_path, (*i)->header);

        var = static_cast<TemplateExecutor::variable **>(apr_array_push(array));
        *var = TemplateVariableCreator::create(pool, &item);
    }

    return TemplateVariableCreator::create(pool, array);
}

TemplateExecutor::variable *UploadItemList::to_thumb_varray(apr_pool_t *pool,
                                                            apr_size_t start,
                                                            apr_size_t size)
{
    apr_array_header_t *array = apr_array_make(pool, static_cast<int>(size),
                                               static_cast<int>(sizeof(TemplateExecutor::variable *)));

    thumb_iterator i = tlist_.begin();
    TemplateExecutor::variable **var;

    for (apr_size_t j = 0; (j < start) && (i != tlist_.end()); j++) {
        i++;
    }

    for (apr_size_t j = 0; (j < size) && (i != tlist_.end()); j++, i++) {
         var = static_cast<TemplateExecutor::variable **>(apr_array_push(array));
         *var = TemplateVariableCreator::create(pool, (*i).c_str());
    }

    return TemplateVariableCreator::create(pool, array);
}

UploadItemList::item_info **UploadItemList::to_array(apr_pool_t *pool,
                                                     apr_size_t start,
                                                     apr_size_t size)
{
    item_iterator i = begin();
    item_info **item_list;
    item_info **item;

    for (apr_size_t j = 0; (j < start) && (i != end()); j++) {
        i++;
    }

    item_list = static_cast<item_info **>(apr_pcalloc(pool,
                                                           sizeof(item_info *)*(size+1)));
    item = item_list;
    for (apr_size_t j = 0; (j < size) && (i != end()); j++, i++) {
        *item = *i;

        item++;
    }

    return item_list;
}

UploadItemList *UploadItemList::load(apr_pool_t *pool,
                                     const char *file_dir, const char *thumb_dir,
                                     apr_size_t total_number_limit,
                                     apr_uint64_t total_size_limit)
{
    UploadItemList *item_list = NULL;
    apr_dir_t *dir;
    apr_finfo_t info;

    if (apr_dir_open(&dir, file_dir, pool) != APR_SUCCESS) {
        throw "åץɥեΥꥹȤɤ߹ޤǤ";
    }

    try {
        item_list = new UploadItemList(pool, file_dir, thumb_dir,
                                       total_number_limit, total_size_limit);

        while (apr_dir_read(&info, APR_FINFO_NAME, dir) == APR_SUCCESS) {
            if (info.name[0] == '.') {
                continue;
            }
            item_list->add(info.name);
        }

        apr_dir_close(dir);

        item_list->init();

        return item_list;
    } catch(const char *) {
        apr_dir_close(dir);

        if (item_list != NULL) {
            delete item_list;
        }

        throw;
    }
}

#ifdef DEBUG
void UploadItemList::dump_list(UploadItemList *item_list)
{
    item_iterator i;

    for (i = item_list->begin(); i != item_list->end(); i++) {
        UploadItem::dump_header((*i)->header);
    }
}
#endif


/******************************************************************************
 * private ᥽å
 *****************************************************************************/
void UploadItemList::init()
{
#ifdef MAKE_THUMBNAIL
    apr_dir_t *dir;
    apr_finfo_t info;
    char *name;

    if (apr_dir_open(&dir, thumb_dir_, pool_) != APR_SUCCESS) {
        throw "ͥΥꥹȤɤ߹ޤǤ";
    }

    while (apr_dir_read(&info, APR_FINFO_NAME, dir) == APR_SUCCESS) {
        if ((info.name[0] == '.') ||
            (strlen(info.name) <= strlen(ThumbnailWriter::FILE_SUFFIX))) {
            continue;
        }

        if (strcmp(info.name+strlen(info.name)-strlen(ThumbnailWriter::FILE_SUFFIX),
                   ThumbnailWriter::FILE_SUFFIX) != 0) {
            continue;
        }

        name = apr_pstrndup(pool_, info.name,
                            strlen(info.name)-strlen(ThumbnailWriter::FILE_SUFFIX));

        tlist_.push_front(name);
    }

    apr_dir_close(dir);

    if (!tlist_.empty()) {
        tlist_.sort(ThumbnailCompare());
    }
#endif

    if (!ilist_.empty()) {
        ilist_.sort(ItemCompare());
    }
    mtime_ = apr_time_now();
}

inline UploadItemList::item_iterator UploadItemList::begin()
{
    return ilist_.begin();
}

inline UploadItemList::item_iterator UploadItemList::end()
{
    return ilist_.end();
}

void UploadItemList::remove(item_info *info)
{
    char *file_path;

    if (apr_filepath_merge(&file_path, file_dir_, info->file_path,
                           APR_FILEPATH_NOTABOVEROOT, pool_) != APR_SUCCESS) {
        throw "åץɥե̾ǤޤǤ";
    }

    if (apr_file_remove(file_path, pool_) != APR_SUCCESS) {
        throw "åץɥեǤޤǤ";
    }

#ifdef MAKE_THUMBNAIL
    tlist_.remove(info->file_path);
    apr_file_remove(ThumbnailWriter::create_thumb_path(pool_, thumb_dir_, info->file_path),
                    pool_);
#endif

    total_size_ -= info->header->file_size;
    ilist_.remove(info);
    mtime_ = apr_time_now();
}


/******************************************************************************
 * ƥ
 *****************************************************************************/
#ifdef DEBUG_UploadItemList
#include "apr_general.h"

const apr_size_t TOTAL_NUMBER_LIMIT = 100;
const apr_size_t TOTAL_SIZE_LIMIT   = 1024*1024*1024;

void usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <DIRECTORY>" << endl;
}

int main(int argc, const char * const *argv)
{
    apr_pool_t *pool;

    apr_app_initialize(&argc, &argv, NULL);
    apr_pool_create(&pool, NULL);

    try {
        if (argc != 2) {
            throw "եѥǥ쥯ȥ꤬ꤵƤޤ";
        }

        auto_ptr<UploadItemList> item_list(UploadItemList::load(pool, argv[1], argv[1],
                                                                TOTAL_NUMBER_LIMIT, TOTAL_SIZE_LIMIT));

        UploadItemList::dump_list(item_list.get());
        item_list->to_array(pool, 0, item_list->size());
    } catch(const char *message) {
        cerr << "Error: " << message << endl;
        usage(argv[0]);

        return EXIT_FAILURE;
    }

    apr_terminate();

    return EXIT_SUCCESS;
}
#endif

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
