/******************************************************************************
 * mod_uploader / ApacheUploaderConfig.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: ApacheUploaderConfig.cpp 869 2005-10-21 06:39:52Z svn $
 *****************************************************************************/

#include "ApacheUploaderConfig.h"
#include "Misc.h"

#include "apr_strings.h"

#undef DELETE
#define DELETE(ptr) if (ptr != NULL) { delete ptr; ptr = NULL;}
#define DESTROY(pool) if (pool != NULL) { apr_pool_destroy(pool); pool = NULL;}

#ifdef DEBUG
#include <iostream>
#endif

const char ApacheUploaderConfig::URL_PARAM[]                    = "Url";
const char ApacheUploaderConfig::FILE_DIRECTORY_PARAM[]         = "FileDirectory";
const char ApacheUploaderConfig::THUMB_DIRECTORY_PARAM[]        = "ThumbDirectory";
const char ApacheUploaderConfig::TMP_DIRECTORY_PARAM[]          = "TmpDirectory";
const char ApacheUploaderConfig::VIEW_TEMPLATE_FILE_PARAM[]     = "ViewTemplateFile";
const char ApacheUploaderConfig::PROGRESS_TEMPLATE_FILE_PARAM[] = "ProgressTemplateFile";
const char ApacheUploaderConfig::DOWNLOAD_TEMPLATE_FILE_PARAM[] = "DownloadTemplateFile";
const char ApacheUploaderConfig::THUMB_TEMPLATE_FILE_PARAM[]    = "ThumbTemplateFile";
const char ApacheUploaderConfig::ERROR_TEMPLATE_FILE_PARAM[]    = "ErrorTemplateFile";
const char ApacheUploaderConfig::MAX_FILE_SIZE_PARAM[]          = "MaxFileSize";
const char ApacheUploaderConfig::TOTAL_FILE_SIZE_LIMIT_PARAM[]  = "TotalFileSizeLimit";
const char ApacheUploaderConfig::TOTAL_FILE_NUMBER_LIMIT_PARAM[]= "TotalFileNumberLimit";
const char ApacheUploaderConfig::PER_PAGE_ITEM_NUMBER_PARAM[]   = "PerPageItemNumber";

static apr_status_t finalize_config(void *data);
#if APR_HAS_THREADS
static apr_status_t finalize_lock(void *data);
#endif


/******************************************************************************
 * public ᥽å
 *****************************************************************************/
ApacheUploaderConfig::ApacheUploaderConfig()
    : UploaderConfig(),
      uparser(NULL),
      glock_(NULL)
#if APR_HAS_THREADS
    , lock_(NULL)
#endif
{

}

ApacheUploaderConfig::~ApacheUploaderConfig()
{
    finalize();
}

void ApacheUploaderConfig::init(request_rec *r, sconfig *sconfig)
{
    if (LIKELY(is_init_)) {
        return update_list(*(sconfig->revision));
    }

    pool_ = r->server->process->pool;
    glock_ = sconfig->upload_lock;

    global_lock();
    try {
        if (!is_init_) {
            exec_init(*(sconfig->revision));
        }
        global_unlock();
    } catch(const char *) {
        global_unlock();
        throw;
    }
}

void ApacheUploaderConfig::finalize()
{
    DELETE(uparser);

    UploaderConfig::finalize();
}

void ApacheUploaderConfig::read_lock()
{
    apr_status_t status = APR_SUCCESS;

#if APR_HAS_THREADS
#ifdef LOCK_TYPE_MUTEX
    status = apr_thread_mutex_lock(lock_);
#else
    status = apr_thread_rwlock_rdlock(lock_);
#endif
#endif

    if (UNLIKELY(status != APR_SUCCESS)) {
        throw "read_lock() failed..";
    }
}

void ApacheUploaderConfig::read_unlock()
{
    apr_status_t status = APR_SUCCESS;

#if APR_HAS_THREADS
#ifdef LOCK_TYPE_MUTEX
    status = apr_thread_mutex_unlock(lock_);
#else
    status = apr_thread_rwlock_unlock(lock_);
#endif
#endif

    if (UNLIKELY(status != APR_SUCCESS)) {
        throw "read_unlock() failed. ";
    }
}

void ApacheUploaderConfig::gwrite_lock()
{
    write_lock();
    global_lock();
}


void ApacheUploaderConfig::gwrite_unlock()
{
    write_unlock();
    global_unlock();
}


/******************************************************************************
 * private ᥽å
 *****************************************************************************/
void ApacheUploaderConfig::exec_init(apr_size_t revision)
{
    validate();

#if APR_HAS_THREADS
    init_lock();
#endif
    load_list(revision);

    init_template();

    uparser = new ApacheRFC1867Parser(tmp_dir, MAX_FORM_TEXT_SIZE,
                                      max_file_size, MAX_FORM_ITEM_NUMBER,
                                      sizeof(UploadItem::header));

    apr_pool_cleanup_register(pool_, this, apr_pool_cleanup_null, finalize_config);

    is_init_ = true;
}

#if APR_HAS_THREADS
void ApacheUploaderConfig::init_lock()
{
#ifdef LOCK_TYPE_MUTEX
    if (apr_thread_mutex_create(&lock_, APR_THREAD_MUTEX_DEFAULT, pool_) != APR_SUCCESS) {
        throw "Can not create thread mutex.";
    }
#else
    if (apr_thread_rwlock_create(&lock_, pool_) != APR_SUCCESS) {
        throw "Can not create rwlock.";
    }
#endif

    apr_pool_cleanup_register(pool_, lock_, apr_pool_cleanup_null, finalize_lock);
}
#endif

void ApacheUploaderConfig::update_list(apr_size_t revision)
{
    if (LIKELY((item_list->get_revision() == revision) &&
               (remove_item_number != ITEM_RELOAD_THRESHOLD))) {
        return;
    }

    try {
        gwrite_lock();

        load_list(revision);

        gwrite_unlock();
    } catch(const char *) {
        gwrite_unlock();
        throw;
    }
}

void ApacheUploaderConfig::write_lock()
{
    apr_status_t status = APR_SUCCESS;

#if APR_HAS_THREADS
#ifdef LOCK_TYPE_MUTEX
    status = apr_thread_mutex_lock(lock_);
#else
    status = apr_thread_rwlock_wrlock(lock_);
#endif
#endif

    if (UNLIKELY(status != APR_SUCCESS)) {
        throw "write_lock() failed.";
    }
}

void ApacheUploaderConfig::write_unlock()
{
    read_unlock();
}

void ApacheUploaderConfig::global_lock()
{
    if (UNLIKELY(apr_global_mutex_lock(glock_) != APR_SUCCESS)) {
        throw "global_lock() failed.";
    }
}

void ApacheUploaderConfig::global_unlock()
{
    if (UNLIKELY(apr_global_mutex_unlock(glock_) != APR_SUCCESS)) {
        throw "global_unlock() failed.";
    }
}


/******************************************************************************
 * Хåؿ
 *****************************************************************************/
apr_status_t finalize_config(void *data)
{
    ApacheUploaderConfig *config;

    config = static_cast<ApacheUploaderConfig *>(data);
    config->finalize();

    return APR_SUCCESS;
}

#if APR_HAS_THREADS
apr_status_t finalize_lock(void *data)
{
#ifdef LOCK_TYPE_MUTEX
    apr_thread_mutex_t *lock;

    lock = static_cast<apr_thread_mutex_t *>(data);
    apr_thread_mutex_destroy(lock);
#else
    apr_thread_rwlock_t *lock;

    lock = static_cast<apr_thread_rwlock_t *>(data);
    apr_thread_rwlock_destroy(lock);
#endif

    return APR_SUCCESS;
}
#endif

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
