/******************************************************************************
 * mod_uploader / Auxiliary.cpp
 ******************************************************************************
 * Copyright (C) 2004 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: Auxiliary.cpp 919 2005-11-22 06:43:05Z svn $
 *****************************************************************************/

#include "Auxiliary.h"
#include "Misc.h"

#include "apr_file_info.h"
#include "apr_strings.h"

#include <string.h>
#include <ctype.h>

static const char *MON_STR[]                = {
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec",
    NULL
};

static const apr_size_t TIME_YEAR_OFFSET    = 1900;

static const char ALPHA_CHAR_FORMAT         = '@';
static const char DIGIT_CHAR_FORMAT         = '#';
static const char RFC822_FORMAT[]           = "@@@, ## @@@ #### ##:##:## GMT";
static const apr_size_t RFC822_DATE_INDEX   = 5;
static const apr_size_t RFC822_MON_INDEX    = 8;
static const apr_size_t RFC822_YEAR_INDEX   = 12;
static const apr_size_t RFC822_HOUR_INDEX   = 17;
static const apr_size_t RFC822_MIN_INDEX    = 20;
static const apr_size_t RFC822_SEC_INDEX    = 23;

static bool check_rfc822_format(const char *str);
static void parse_rfc822_Init(char *str);


const char *escape_html(apr_pool_t *pool, const char* str)
{
    static const char LESS_THAN[]       = "&lt;";
    static const char GREATER_THAN[]    = "&gt;";
    static const char AMPERSAND[]       = "&amp;";

    const char *plain;
    char *escaped;
    char *buffer;
    apr_size_t escaped_length;

    escaped_length = 0;
    plain = str;
    do {
        if ((*plain == '<') || (*plain == '>')) {
            escaped_length += 3;
        } else if (*plain == '&') {
            escaped_length += 4;
        }
    } while (*(++plain) != '\0');

    if (escaped_length == 0) {
        return apr_pstrdup(pool, str);
    }
    escaped_length += strlen(str);

    APR_PCALLOC(escaped, char *, pool, escaped_length + 1);
    buffer = escaped;

    do {
        if (*str == '<') {
            memcpy(buffer, LESS_THAN, strlen(LESS_THAN));
            buffer += strlen(LESS_THAN);
        } else if (*str == '>') {
            memcpy(buffer, GREATER_THAN, strlen(GREATER_THAN));
            buffer += strlen(GREATER_THAN);
        } else if (*str == '&') {
            memcpy(buffer, AMPERSAND, strlen(AMPERSAND));
            buffer += strlen(AMPERSAND);
        } else {
            *buffer++ = *str;
        }
    } while (*(++str) != '\0');

    return escaped;
}

const char *rfc1738_encode(apr_pool_t *pool, const char* str)
{
    static char hex[] = "0123456789ABCDEF";

    const char *plain;
    apr_size_t not_alnum;
    char *encoded;
    char *buffer;
    unsigned char c;

    not_alnum = 0;
    plain = str;
    do {
        if (!isalnum(*plain)) {
            not_alnum++;
        }
    } while (*(++plain) != '\0');

    if (not_alnum == 0) {
        return apr_pstrdup(pool, str);
    }

    APR_PCALLOC(encoded, char *, pool, strlen(str) + (not_alnum*2) + 1);
    buffer = encoded;

    do {
        *buffer = *str;

        if (!isalnum(*buffer)) {
            c = (unsigned char)(*buffer);

            *buffer++ = '%';
            *buffer++ = hex[c / 0x10];
            *buffer++ = hex[c % 0x10];
        } else {
            buffer++;
        }
    } while (*(++str) != '\0');

    return encoded;
}

static char ctoi(const char c)
{
    return ((c >= 'A') ? ((c & 0xff) - 'A') + 10 : (c - '0'));
}

const char *rfc1738_decode(apr_pool_t *pool, const char* str)
{
    char *plain;
    char *pos;

    APR_PCALLOC(plain, char *, pool, strlen(str) + 1);

    pos = plain;
    while (*str != '\0') {
        if (*str != '%') {
            *pos++ = *str++;
            continue;
        }

        if ((*(str+1) != '\0') && (*(str+2) != '\0') &&
            isxdigit(*(str+1) & 0xff) && isxdigit(*(str+2) & 0xff)) {

            *pos++ = (ctoi(*(str+1)) << 4) + ctoi(*(str+2));
            str += 3;
        } else {
            *pos++ = *str++;
        }
    }

    return plain;
}

const char *size_str(apr_pool_t *pool, apr_uint64_t size)
{
#ifdef NO_DOUBLE
    if (size < 1024) {
        return apr_psprintf(pool, "%" APR_UINT64_T_FMT " %s", size, BYTE_UNIT);
    } else if (size < 1024*1024) {
        return apr_psprintf(pool, "%" APR_UINT64_T_FMT " %s", size/1024, KBYTE_UNIT);
    } else if (size < 1024*1024*1024) {
        return apr_psprintf(pool, "%" APR_UINT64_T_FMT " %s", size/(1024*1024), MBYTE_UNIT);
    } else {
        return apr_psprintf(pool, "%" APR_UINT64_T_FMT " %s", size/(1024*1024*1024), GBYTE_UNIT);
    }
#else
    if (size < 1024) {
        return apr_psprintf(pool, "%" APR_UINT64_T_FMT " %s", size, BYTE_UNIT);
    } else if (size < 1024*1024) {
        return apr_psprintf(pool, "%.1f %s",
                            static_cast<double>(size)/1024, KBYTE_UNIT);
    } else if (size < 1024*1024*1024) {
        return apr_psprintf(pool, "%.1f %s",
                            static_cast<double>(size/1024)/1024, MBYTE_UNIT);
    } else {
        return apr_psprintf(pool, "%.1f %s",
                            static_cast<double>(size/(1024*1024))/1024, GBYTE_UNIT);
    }
#endif
}

const char *comma_str(apr_pool_t *pool, apr_size_t number)
{
    apr_size_t figure = 1;
    char *str;

    {
        apr_size_t tmp = number;
        while ((tmp /= 10) != 0) {
            figure++;
        }
    }
    {
        apr_size_t size = figure + (figure-1)/3 + 1;
        APR_PCALLOC(str, char *, pool, sizeof(char)*size);
        str += size - 1;
    }
    *str-- = '\0';
    {
        apr_size_t count = 0;
        do {
            *str-- = (char)(number%10) + '0';
            number /= 10;

            if ((number != 0) && ((++count % 3) == 0)) {
                *str-- = ',';
            }
        } while (number != 0);
    }

    return ++str;
}

bool check_rfc822_format(const char *str)
{
    if (strlen(str) != strlen(RFC822_FORMAT)) {
        return false;
    }

    for (apr_size_t i = 0; i < strlen(RFC822_FORMAT); i++) {
        if (RFC822_FORMAT[i] == ALPHA_CHAR_FORMAT) {
            if (!isalpha(str[i])) {
                return false;
            }
        } else if (RFC822_FORMAT[i] == DIGIT_CHAR_FORMAT) {
            if (!isdigit(str[i])) {
                return false;
            }
        } else {
            if (str[i] != RFC822_FORMAT[i]) {
                return false;
            }
        }
    }

    return true;
}

void parse_rfc822_Init(char *str)
{
    bool is_prev_digit_char;

    is_prev_digit_char = false;
    for (apr_size_t i = 0; i < strlen(RFC822_FORMAT); i++) {
        if (RFC822_FORMAT[i] == DIGIT_CHAR_FORMAT) {
            is_prev_digit_char = true;
            continue;
        }

        if (is_prev_digit_char) {
            str[i] = '\0';
        }
        is_prev_digit_char = false;
    }
}

apr_time_t parse_rfc822(apr_pool_t *pool, const char *str)
{
    char *parse_str;
    apr_time_exp_t time_exp;
    apr_time_t time;
    apr_size_t i;

    if (str == NULL) {
        return 0;
    }

    if (!check_rfc822_format(str)) {
        return 0;
    }

    parse_str = apr_pstrdup(pool, str);
    parse_rfc822_Init(parse_str);

    i = 0;
    while (true){
        if (MON_STR[i] == NULL) {
            return 0;
        }
        if (strncmp(MON_STR[i], parse_str+RFC822_MON_INDEX, strlen(MON_STR[i])) == 0) {
            time_exp.tm_mon = i;
            break;
        }

        i++;
    }

    time_exp.tm_year = atoi(parse_str + RFC822_YEAR_INDEX) - TIME_YEAR_OFFSET;
    time_exp.tm_mday = atoi(parse_str + RFC822_DATE_INDEX);
    time_exp.tm_hour = atoi(parse_str + RFC822_HOUR_INDEX);
    time_exp.tm_min  = atoi(parse_str + RFC822_MIN_INDEX);
    time_exp.tm_sec  = atoi(parse_str + RFC822_SEC_INDEX);
    time_exp.tm_usec = 0;

    if (apr_time_exp_get(&time, &time_exp) != APR_SUCCESS) {
        return 0;
    }

    return time;
}

const char *basename_ex(const char *path)
{
    const char *i = path + strlen(path) - 2;

    while ((*i != '/') && (*i != '\\') && (*i != ':') &&
           (*i != '\r') && (*i != '\n')) { /* ǰΤ */
        if (i == path) {
            return path;
        }

        i--;
    }

    return i + 1;
}

const char *dirname_ex(apr_pool_t *pool, const char *path)
{
    char *dirname;
    char *i;

    dirname = apr_pstrdup(pool, path);

    i = dirname + strlen(dirname) - 2;
    while ((*i != '/') && (*i != '\\') && (*i != ':') &&
           (*i != '\r') && (*i != '\n')) { /* ǰΤ */
        if (i == path) {
            break;
        }

        i--;
    }

    *i = '\0';

    return dirname;
}

#ifndef HAVE_STRNCHR
const char *strnchr(const char *s, size_t length, int c)
{
    const char *pos;
    const unsigned long *word_pos;
    unsigned long word_char;
    unsigned long magic_bits;
    unsigned long char_mask;
    int count;

    count = length;

    // ɶޤǤϥХñ̤ǥå
    for (pos = const_cast<char *>(s);
         (reinterpret_cast<unsigned long>(pos) & (sizeof(long)-1)) != 0;
         pos++) {
        if (*pos == c) {
            return pos;
        } else if (*pos == '\0') {
            return NULL;
        }

        if (--count == 0) {
            return NULL;
        }
    }

    word_pos = reinterpret_cast<const unsigned long *>(pos);

    if (sizeof(long) == 4) {
        magic_bits = 0x7efefeffL;
    } else if (sizeof(long) == 8) {
        magic_bits = ((0x7efefefeL << 16) << 16) | 0xfefefeffL;
    } else {
        abort();
    }

    char_mask = c | (c << 8);
    char_mask |= char_mask << 16;
    if (sizeof(long) == 8) {
        char_mask |= (char_mask << 16) << 16;
    }

    // ñ̤ǥå
    do {
        word_char = *word_pos++;

        if (LIKELY((((((word_char            ) + magic_bits) ^ ~(word_char            )) & ~magic_bits) == 0) &&
                   (((((word_char ^ char_mask) + magic_bits) ^ ~(word_char ^ char_mask)) & ~magic_bits) == 0))) {
            count -= sizeof(long);
            continue;
        }

        pos = reinterpret_cast<const char *>(word_pos - 1);

        if (*pos == c) {
            return pos;
        } else if (*pos == '\0') {
            return NULL;
        } else if (--count == 0) {
            return NULL;
        }

        if (*(++pos) == c) {
            return pos;
        } else if (*pos == '\0') {
            return NULL;
        } else if (--count == 0) {
            return NULL;
        }

        if (*(++pos) == c) {
            return pos;
        } else if (*pos == '\0') {
            return NULL;
        } else if (--count == 0) {
            return NULL;
        }

        if (*(++pos) == c) {
            return pos;
        } else if (*pos == '\0') {
            return NULL;
        } else if (--count == 0) {
            return NULL;
        }

        if (sizeof(long) == 8) {
            if (*(++pos) == c) {
                return pos;
            } else if (*pos == '\0') {
                return NULL;
            } else if (--count == 0) {
                return NULL;
            }

            if (*(++pos) == c) {
                return pos;
            } else if (*pos == '\0') {
                return NULL;
            } else if (--count == 0) {
                return NULL;
            }

            if (*(++pos) == c) {
                return pos;
            } else if (*pos == '\0') {
                return NULL;
            } else if (--count == 0) {
                return NULL;
            }

            if (*(++pos) == c) {
                return pos;
            } else if (*pos == '\0') {
                return NULL;
            } else if (--count == 0) {
                return NULL;
            }
        }
    } while (count > 0);

    return NULL;
}
#endif

#if !defined(HAVE_MEMMEM) || defined(WIN32)
void *memmem(const void *haystack, size_t haystacklen,
             const void *needle, size_t needlelen)
{
    unsigned char *start = (unsigned char *)haystack;
    unsigned char pat = *((unsigned char *)needle);
    unsigned char *pos  = start;
    apr_size_t remain = haystacklen;

    do {
        pos = (unsigned char *)memchr(pos, pat, remain);

        if (pos == NULL) {
            return NULL;
        }

        if ((memcmp(pos, needle, needlelen)) == 0) {
            return pos;
        }
        pos++;

        remain = haystacklen - (pos-start);
    } while (remain >= needlelen);

    return NULL;
}
#endif

/*
 * Local Variables:
 * mode: c++
 * buffer-file-coding-system: euc-japan-dos
 * End:
 */
