/******************************************************************************
 * mod_uploader / ConfigReader.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: ConfigReader.cpp 867 2005-10-20 23:58:58Z svn $
 *****************************************************************************/

#include "ConfigReader.h"
#include "Misc.h"

#include "apr_mmap.h"
#include "apr_strings.h"

#include <ctype.h>

#ifdef DEBUG
#include <iostream>
#endif

const char ConfigReader::COMMENT_CHAR   = '#';
const char ConfigReader::QUOTE_CHAR     = '"';
const char ConfigReader::ASSIGN_CHAR    = '=';


/******************************************************************************
 * public ᥽å
 *****************************************************************************/
ConfigReader::config_map *ConfigReader::read(apr_pool_t *pool,
                                             const char *file_path)
{
    const char *data;
    const char *name;
    const char *value;
    config_map *cmap;

    read_file(pool, file_path, &data);

    cmap = new config_map();
    while (*data != '\0') {
        read_line(pool, &data, &name, &value);
        cmap->insert(config_pair(name, value));
    }

    return cmap;
}


/******************************************************************************
 * private ᥽å
 *****************************************************************************/
void ConfigReader::read_file(apr_pool_t *pool, const char *file_path,
                             const char **data)
{
    apr_file_t *file;
    apr_mmap_t *file_map;
    apr_finfo_t info;

    if (apr_file_open(&file, file_path, APR_READ|APR_BINARY,
                      APR_OS_DEFAULT, pool) != APR_SUCCESS) {
        throw "ե򳫤ޤǤ";
    }

    if (apr_file_info_get(&info, APR_FINFO_SIZE, file) != APR_SUCCESS) {
        throw "եΥǤޤǤ";
    }

    if (apr_mmap_create(&file_map, file,
                        0, static_cast<apr_size_t>(info.size),
                        APR_MMAP_READ, pool) != APR_SUCCESS) {
        throw "ե mmap ˼Ԥޤ";
    }

    APR_PCALLOC(*data, char *, pool, static_cast<apr_size_t>(info.size+1));

    memcpy(const_cast<char *>(*data), file_map->mm, static_cast<apr_size_t>(info.size));

    if (apr_mmap_delete(file_map) != APR_SUCCESS) {
        throw "ե munmap ˼Ԥޤ";
    }

    apr_file_close(file);
}

void ConfigReader::read_line(apr_pool_t *pool, const char **data,
                             const char **name, const char **value)
{
    skip_space(data);

    // ȹԤޤ϶
    while ((**data == COMMENT_CHAR) || (**data == '\n')) {
        skip_line(data);
        skip_space(data);
    }

    if (**data == '\0') {
        return;
    }

    read_param(pool, data, name, value);

    skip_space(data);

    if (**data != '\n') {
        throw "λˡǤ(1)";
    }

    skip_line(data);
}

void ConfigReader::read_param(apr_pool_t *pool, const char **data,
                              const char **name, const char **value)
{
    read_word(pool, data, name);

    skip_space(data);
    if (**data != ASSIGN_CHAR) {
        throw "λˡǤ(2)";
    }
    (*data)++;
    skip_space(data);

    read_word(pool, data, value);
}

void ConfigReader::read_word(apr_pool_t *pool, const char **data,
                             const char **word)
{
    const char *pos;

    if (**data == QUOTE_CHAR) {
        pos = ++(*data);

        // QUOTE_CHAR Υפ̤ݡ
        while ((*pos != '\0') && (*pos != QUOTE_CHAR)) {
            pos++;
        }
    } else {
        pos = *data;

        while (isalnum(*pos & 0xff) || (*pos == '-')) {
            pos++;
        }

    }

    *word = apr_pstrndup(pool, *data, pos - *data);
    *data = pos + 1;
}

void ConfigReader::skip_space(const char **data)
{
    // Ԥϥåפʤ
    while ((**data != '\0') && (**data != '\n') && isspace(**data & 0xff)) {
        (*data)++;
    }
}

void ConfigReader::skip_line(const char **data)
{
    while ((**data != '\0') && (**data != '\n')) {
        (*data)++;
    }

    if (**data == '\n') {
        (*data)++;
    }
}


/******************************************************************************
 * ƥ
 *****************************************************************************/
#ifdef DEBUG_ConfigReader
#include "apr_general.h"

#include <iomanip>

void usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <INPUT>" << endl;
}

int main(int argc, const char * const *argv)
{
    apr_pool_t *pool;
    ConfigReader::config_map *cmap;

    apr_app_initialize(&argc, &argv, NULL);
    apr_pool_create(&pool, NULL);

    try {
        if (argc != 2) {
            throw "ѤΥե뤬ꤵƤޤ";
        }

        cmap = ConfigReader::read(pool, argv[1]);

        ConfigReader::config_map::const_iterator i = cmap->begin();
        while (i != cmap->end()) {
            cout.width(35);
            cout.setf(ios::left, ios::adjustfield);
            cout << ("name = [" + i->first + "] ");
            cout << "value = [" << i->second << "]" << endl;

            i++;
        }
    } catch(const char *message) {
        cerr << "Error: " << message << endl;
        usage(argv[0]);

        return EXIT_FAILURE;
    }

    apr_terminate();

    return EXIT_SUCCESS;
}

#endif

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
