/******************************************************************************
 * mod_uploader / MessageDigest5.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: MessageDigest5.cpp 895 2005-10-23 18:50:50Z svn $
 *****************************************************************************/

#include "MessageDigest5.h"
#include "Misc.h"

#define APR_WANT_MEMFUNC
#define APR_WANT_STDIO
#include "apr_want.h"

#ifdef DEBUG
#include <iostream>
#endif

const apr_byte_t MessageDigest5::PADDING[64] = {
    0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0,    0, 0, 0
};


/******************************************************************************
 * public ᥽å
 *****************************************************************************/
MessageDigest5::MessageDigest5()
    : remain_length_(0),
      is_finish_(false)
{
    size_[0]        = 0;
    size_[1]        = 0;

    state_[0]       = 0x67452301;
    state_[1]       = 0xefcdab89;
    state_[2]       = 0x98badcfe;
    state_[3]       = 0x10325476;

    memset(digest_str_, 0, sizeof(digest_str_));
}

void MessageDigest5::update(const apr_byte_t *input, apr_size_t size)
{
    apr_uint32_t length;
    apr_size_t gap;
    apr_size_t read_size;

#ifdef DEBUG
    if (is_finish_) {
        throw "Ǥ";
    }
#endif

    length = (apr_uint32_t)size;

    if (UNLIKELY((size_[0] += (length << 3)) < (length << 3))) {
        size_[1]++;
    }
    size_[1] += length >> (32-3);

    gap = 64 - remain_length_;

    if (LIKELY(length >= gap)) {
        read_size = 0;

        memcpy(buffer_+remain_length_, input, gap);
        process(buffer_);

        for (read_size = gap; read_size + 64 <= length; read_size += 64) {
            process(input+read_size);
        }

        memcpy(buffer_, input+read_size, length-read_size);
    } else {
        memcpy(buffer_+remain_length_, input, length);
    }

    remain_length_ = (size_[0] >> 3) & 0x3f;
}

void MessageDigest5::finish()
{
    apr_byte_t input_size[8];
    apr_size_t padding_length;

#ifdef DEBUG
    if (is_finish_) {
        throw "Ǥ";
    }
#endif

    int2byte(size_, 2, input_size);

    padding_length = (remain_length_ < ((512-64) >> 3))
        ? (((512   - 64) >> 3) - remain_length_)
        : (((512*2 - 64) >> 3) - remain_length_);
    update(PADDING, padding_length);

    update(input_size, 8);

    create_digest();

    is_finish_ = true;
}

const char *MessageDigest5::c_str()
{
#ifdef DEBUG
    if (!is_finish_) {
        throw "Ǥ";
    }
#endif

    return digest_str_;
}


/******************************************************************************
 * private ᥽å
 *****************************************************************************/
inline apr_uint32_t MessageDigest5::f(apr_uint32_t x, apr_uint32_t y, apr_uint32_t z)
{
    return z ^ (x & (y ^ z));
}

inline apr_uint32_t MessageDigest5::g(apr_uint32_t x, apr_uint32_t y, apr_uint32_t z)
{
    return (x & z) | (y & ~z);
}

inline apr_uint32_t MessageDigest5::h(apr_uint32_t x, apr_uint32_t y, apr_uint32_t z)
{
    return x ^ y ^ z;
}

inline apr_uint32_t MessageDigest5::i(apr_uint32_t x, apr_uint32_t y, apr_uint32_t z)
{
    return y ^ (x | ~z);
}

inline apr_uint32_t MessageDigest5::rotl(apr_uint32_t value, apr_uint32_t shift)
{
    // GCC ϼμñ roll ̿ѴƤ褦Ǥ
    return (value << shift) | (value >> (32-shift));
}

inline apr_uint32_t MessageDigest5::round1(apr_uint32_t a, apr_uint32_t b,
                                           apr_uint32_t c, apr_uint32_t d,
                                           apr_uint32_t x, apr_uint32_t s,
                                           apr_uint32_t t)
{
    return b + rotl(a + f(b, c, d) + x + t, s);
}

inline apr_uint32_t MessageDigest5::round2(apr_uint32_t a, apr_uint32_t b,
                                           apr_uint32_t c, apr_uint32_t d,
                                           apr_uint32_t x, apr_uint32_t s,
                                           apr_uint32_t t)
{
    return b + rotl(a + g(b, c, d) + x + t, s);
}

inline apr_uint32_t MessageDigest5::round3(apr_uint32_t a, apr_uint32_t b,
                                           apr_uint32_t c, apr_uint32_t d,
                                           apr_uint32_t x, apr_uint32_t s,
                                           apr_uint32_t t)
{
    return b + rotl(a + h(b, c, d) + x + t, s);
}

inline apr_uint32_t MessageDigest5::round4(apr_uint32_t a, apr_uint32_t b,
                                           apr_uint32_t c, apr_uint32_t d,
                                           apr_uint32_t x, apr_uint32_t s,
                                           apr_uint32_t t)
{
    return b + rotl(a + i(b, c, d) + x + t, s);
}

void MessageDigest5::byte2int(const apr_byte_t *bytes, apr_uint32_t *ints)
{
    apr_size_t i;

    for (i = 0; i < 16; ++i) {
        ints[i] = ((apr_uint32_t)bytes[i*4    ]) |
                  ((apr_uint32_t)bytes[i*4 + 1] <<  8) |
                  ((apr_uint32_t)bytes[i*4 + 2] << 16) |
                  ((apr_uint32_t)bytes[i*4 + 3] << 24);
    }
}

void MessageDigest5::int2byte(const apr_uint32_t *ints, apr_size_t len, apr_byte_t *bytes)
{
    apr_size_t i;

    for (i = 0; i < len; ++i) {
        bytes[i*4    ] = (ints[i]      ) & 0xff;
        bytes[i*4 + 1] = (ints[i] >>  8) & 0xff;
        bytes[i*4 + 2] = (ints[i] >> 16) & 0xff;
        bytes[i*4 + 3] = (ints[i] >> 24) & 0xff;
    }
}

void MessageDigest5::process(const apr_byte_t block[64])
{
    apr_uint32_t a, b, c, d;
    apr_uint32_t x[16];

    a = state_[0];
    b = state_[1];
    c = state_[2];
    d = state_[3];

    byte2int(block, x);

    a = round1(a, b, c, d, x[ 0],  7, 0xd76aa478);
    d = round1(d, a, b, c, x[ 1], 12, 0xe8c7b756);
    c = round1(c, d, a, b, x[ 2], 17, 0x242070db);
    b = round1(b, c, d, a, x[ 3], 22, 0xc1bdceee);
    a = round1(a, b, c, d, x[ 4],  7, 0xf57c0faf);
    d = round1(d, a, b, c, x[ 5], 12, 0x4787c62a);
    c = round1(c, d, a, b, x[ 6], 17, 0xa8304613);
    b = round1(b, c, d, a, x[ 7], 22, 0xfd469501);
    a = round1(a, b, c, d, x[ 8],  7, 0x698098d8);
    d = round1(d, a, b, c, x[ 9], 12, 0x8b44f7af);
    c = round1(c, d, a, b, x[10], 17, 0xffff5bb1);
    b = round1(b, c, d, a, x[11], 22, 0x895cd7be);
    a = round1(a, b, c, d, x[12],  7, 0x6b901122);
    d = round1(d, a, b, c, x[13], 12, 0xfd987193);
    c = round1(c, d, a, b, x[14], 17, 0xa679438e);
    b = round1(b, c, d, a, x[15], 22, 0x49b40821);
    a = round2(a, b, c, d, x[ 1],  5, 0xf61e2562);
    d = round2(d, a, b, c, x[ 6],  9, 0xc040b340);
    c = round2(c, d, a, b, x[11], 14, 0x265e5a51);
    b = round2(b, c, d, a, x[ 0], 20, 0xe9b6c7aa);
    a = round2(a, b, c, d, x[ 5],  5, 0xd62f105d);
    d = round2(d, a, b, c, x[10],  9, 0x02441453);
    c = round2(c, d, a, b, x[15], 14, 0xd8a1e681);
    b = round2(b, c, d, a, x[ 4], 20, 0xe7d3fbc8);
    a = round2(a, b, c, d, x[ 9],  5, 0x21e1cde6);
    d = round2(d, a, b, c, x[14],  9, 0xc33707d6);
    c = round2(c, d, a, b, x[ 3], 14, 0xf4d50d87);
    b = round2(b, c, d, a, x[ 8], 20, 0x455a14ed);
    a = round2(a, b, c, d, x[13],  5, 0xa9e3e905);
    d = round2(d, a, b, c, x[ 2],  9, 0xfcefa3f8);
    c = round2(c, d, a, b, x[ 7], 14, 0x676f02d9);
    b = round2(b, c, d, a, x[12], 20, 0x8d2a4c8a);
    a = round3(a, b, c, d, x[ 5],  4, 0xfffa3942);
    d = round3(d, a, b, c, x[ 8], 11, 0x8771f681);
    c = round3(c, d, a, b, x[11], 16, 0x6d9d6122);
    b = round3(b, c, d, a, x[14], 23, 0xfde5380c);
    a = round3(a, b, c, d, x[ 1],  4, 0xa4beea44);
    d = round3(d, a, b, c, x[ 4], 11, 0x4bdecfa9);
    c = round3(c, d, a, b, x[ 7], 16, 0xf6bb4b60);
    b = round3(b, c, d, a, x[10], 23, 0xbebfbc70);
    a = round3(a, b, c, d, x[13],  4, 0x289b7ec6);
    d = round3(d, a, b, c, x[ 0], 11, 0xeaa127fa);
    c = round3(c, d, a, b, x[ 3], 16, 0xd4ef3085);
    b = round3(b, c, d, a, x[ 6], 23, 0x04881d05);
    a = round3(a, b, c, d, x[ 9],  4, 0xd9d4d039);
    d = round3(d, a, b, c, x[12], 11, 0xe6db99e5);
    c = round3(c, d, a, b, x[15], 16, 0x1fa27cf8);
    b = round3(b, c, d, a, x[ 2], 23, 0xc4ac5665);
    a = round4(a, b, c, d, x[ 0],  6, 0xf4292244);
    d = round4(d, a, b, c, x[ 7], 10, 0x432aff97);
    c = round4(c, d, a, b, x[14], 15, 0xab9423a7);
    b = round4(b, c, d, a, x[ 5], 21, 0xfc93a039);
    a = round4(a, b, c, d, x[12],  6, 0x655b59c3);
    d = round4(d, a, b, c, x[ 3], 10, 0x8f0ccc92);
    c = round4(c, d, a, b, x[10], 15, 0xffeff47d);
    b = round4(b, c, d, a, x[ 1], 21, 0x85845dd1);
    a = round4(a, b, c, d, x[ 8],  6, 0x6fa87e4f);
    d = round4(d, a, b, c, x[15], 10, 0xfe2ce6e0);
    c = round4(c, d, a, b, x[ 6], 15, 0xa3014314);
    b = round4(b, c, d, a, x[13], 21, 0x4e0811a1);
    a = round4(a, b, c, d, x[ 4],  6, 0xf7537e82);
    d = round4(d, a, b, c, x[11], 10, 0xbd3af235);
    c = round4(c, d, a, b, x[ 2], 15, 0x2ad7d2bb);
    b = round4(b, c, d, a, x[ 9], 21, 0xeb86d391);

    state_[0] += a;
    state_[1] += b;
    state_[2] += c;
    state_[3] += d;
}

void MessageDigest5::create_digest()
{
    int2byte(state_, 4, digest_bytes_);

    sprintf(digest_str_,
            "%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x",
            digest_bytes_[ 0], digest_bytes_[ 1], digest_bytes_[ 2], digest_bytes_[ 3],
            digest_bytes_[ 4], digest_bytes_[ 5], digest_bytes_[ 6], digest_bytes_[ 7],
            digest_bytes_[ 8], digest_bytes_[ 9], digest_bytes_[10], digest_bytes_[11],
            digest_bytes_[12], digest_bytes_[13], digest_bytes_[14], digest_bytes_[15]);
}


/******************************************************************************
 * ƥ
 *****************************************************************************/
#ifdef DEBUG_MessageDigest5
#include "apr_general.h"

static const char TEST_STR[]    = "12345678901234567890123456789012345678901234567890123456789012345678901234567890";

#include <iomanip>

void usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <INPUT>" << endl;
}

int main(int argc, const char * const *argv)
{
    apr_pool_t *pool;
    MessageDigest5 digest;

    apr_app_initialize(&argc, &argv, NULL);
    apr_pool_create(&pool, NULL);

    digest.update(reinterpret_cast<const apr_byte_t *>(TEST_STR), strlen(TEST_STR));
    digest.finish();

    cout << "input   : " << TEST_STR << endl;
    cout << "digest  : " << digest.c_str() << endl;
    cout << "expected: 57edf4a22be3c955ac49da2e2107b67a" << endl;

    apr_terminate();

    return EXIT_SUCCESS;
}
#endif

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
