/******************************************************************************
 * mod_uploader / UploadItemCreator.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not bcktuired.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: UploadItemCreator.cpp 913 2005-10-29 18:33:54Z svn $
 *****************************************************************************/

#include "UploadItemCreator.h"
#include "CharCodeConverter.h"
#include "Auxiliary.h"
#include "Misc.h"

#ifdef DEBUG
#include <iostream>
#endif


/******************************************************************************
 * public ᥽å
 *****************************************************************************/
UploadItem::header *UploadItemCreator::create_header(apr_pool_t *pool,
                                                     RFC1867Data::query_map *qmap,
                                                     const char **tmp_path)
{
    UploadItem::header *data;

    validate_query(pool, qmap);

    APR_PCALLOC(data, UploadItem::header *, pool, sizeof(UploadItem::header));

    strncpy(data->identifier, PACKAGE_NAME, sizeof(PACKAGE_NAME));
    strncpy(data->version, PACKAGE_VERSION, sizeof(PACKAGE_VERSION));

    data->time = apr_time_now();
    strncpy(data->comment, (*qmap)[UploadItem::COMMENT_PARAM].text.c_str(),
            UploadItem::MAX_COMMENT_LENGTH-1);

    strncpy(data->download_pass, (*qmap)[UploadItem::DOWNLOAD_PASS_PARAM].text.c_str(),
            UploadItem::MAX_DOWNLOAD_PASS_LENGTH-1);

    strncpy(data->remove_pass, (*qmap)[UploadItem::REMOVE_PASS_PARAM].text.c_str(),
            UploadItem::MAX_REMOVE_PASS_LENGTH-1);

    set_file_data(pool, data, (*qmap)[UploadItem::FILE_PARAM].file, qmap);

    *tmp_path = (*qmap)[UploadItem::FILE_PARAM].file.tmp_path.c_str();

    return data;
}


/******************************************************************************
 * private ᥽å
 *****************************************************************************/
void UploadItemCreator::validate_query(apr_pool_t *pool,
                                       RFC1867Data::query_map *qmap)
{
    const unsigned char *code_pat;

    // MEMO:
    // - ե륵 0 ξϡ顼Ф
    // - ȤѥɤˤĤƤϡȤǤ⥨顼Фʤ
    // - ե̾MIME סȤ HTML פ롥
    // - ν֤ϡʸѴ -> HTML  -> Ĺåν

    if (qmap->count(UploadItem::CODE_PAT_PARAM) != 1) {
        throw "ʸȽѥλ꤬Ǥ";
    }

    code_pat = reinterpret_cast<const unsigned char *>((*qmap)[UploadItem::CODE_PAT_PARAM].text.c_str());

    if (qmap->count(UploadItem::FILE_PARAM) != 1) {
        throw "եλ꤬Ǥ";
    }
    validate_file(pool, (*qmap)[UploadItem::FILE_PARAM], code_pat);

    if (qmap->count(UploadItem::COMMENT_PARAM) != 1) {
        throw "Ȥλ꤬Ǥ";
    }
    validate_comment(pool, (*qmap)[UploadItem::COMMENT_PARAM], code_pat);

    if (qmap->count(UploadItem::DOWNLOAD_PASS_PARAM) != 1) {
        throw "ɥѥɤλ꤬Ǥ";
    }
    validate_download_pass(pool, (*qmap)[UploadItem::DOWNLOAD_PASS_PARAM]);

    if (qmap->count(UploadItem::REMOVE_PASS_PARAM) != 1) {
        throw "ѥɤλ꤬Ǥ";
    }
    validate_remove_pass(pool, (*qmap)[UploadItem::REMOVE_PASS_PARAM]);
}

void UploadItemCreator::validate_file(apr_pool_t *pool, content& content,
                                      const unsigned char *code_pat)
{
    const char *file_name;

    convert_code(pool, content.text, code_pat);
    escape_html(pool, content.text);

    if ((content.type != RFC1867Data::FILE) || (content.file.size == 0)) {
        throw "ե뤬Ǥ";
    }

    file_name = content.file.name.c_str();

    if (strlen(file_name) >= UploadItem::MAX_FILE_NAME_LENGTH) {
        throw "ե̾Ĺޤ";
    }
    do {
        if ((*file_name == '"') || (*file_name == '\'') ||
            (*file_name == '\r') || (*file_name == '\n') ||
            (*file_name == '\\')) {
            throw "ե̾Ǥ";
        }
    } while (*(++file_name) != '\0');

    escape_html(pool, content.file.mime);
    if (content.file.mime.length() >= UploadItem::MAX_FILE_MIME_LENGTH) {
        throw "ե MIME Type Ĺޤ";
    }
}

void UploadItemCreator::validate_comment(apr_pool_t *pool, content& content,
                                         const unsigned char *code_pat)
{
    convert_code(pool, content.text, code_pat);
    escape_html(pool, content.text);

    if ((content.type != RFC1867Data::TEXT) ||
        (content.text.length() >= UploadItem::MAX_COMMENT_LENGTH)) {
        throw "ȤĹޤ";
    }
}

void UploadItemCreator::validate_download_pass(apr_pool_t *pool, content& content)
{
    const char *download_pass;

    if ((content.type != RFC1867Data::TEXT) ||
        (content.text.length() >= UploadItem::MAX_DOWNLOAD_PASS_LENGTH)) {
        throw "ɥѥɤĹޤ";
    }

    download_pass = content.file.name.c_str();

    do {
        if ((*download_pass == '<') || (*download_pass == '>') ||
            (*download_pass == '"') || (*download_pass == '\'') ||
            (*download_pass == '\r') || (*download_pass == '\n') ||
            (*download_pass == '\\')) {
            throw "ɥѥɤ"
                  "<ס>ס\"ס'ס\\rס\\nס\\"
                  "ϻȤޤ";
        }
    } while (*(++download_pass) != '\0');
}

void UploadItemCreator::validate_remove_pass(apr_pool_t *pool, content& content)
{
    if ((content.type != RFC1867Data::TEXT) ||
        (content.text.length() >= UploadItem::MAX_REMOVE_PASS_LENGTH)) {
        throw "ѥɤĹޤ";
    }
}

void UploadItemCreator::escape_html(apr_pool_t *pool, string &str)
{
    str.assign(::escape_html(pool, str.c_str()));
}

void UploadItemCreator::convert_code(apr_pool_t *pool, string &str,
                                     const unsigned char *code_pat)
{
    str.assign(CharCodeConverter::convert(pool, str.c_str(), code_pat));
}

void UploadItemCreator::set_file_data(apr_pool_t *pool,
                                      UploadItem::header *data,
                                      RFC1867Data::file_content& file,
                                      RFC1867Data::query_map *qmap)
{
    strncpy(data->file_name, file.name.c_str(), UploadItem::MAX_FILE_NAME_LENGTH-1);
    strncpy(data->file_mime, file.mime.c_str(), UploadItem::MAX_FILE_MIME_LENGTH-1);
    strncpy(data->file_digest, file.digest.c_str(), UploadItem::MAX_FILE_DIGEST_LENGTH-1);

    data->file_size = file.size;
}

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
