/******************************************************************************
 * mod_uploader / apache_handler.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: cgi_handler.cpp 914 2005-11-17 15:07:03Z svn $
 *****************************************************************************/

#include "cgi_handler.h"
#include "uploader_func.h"

#include "UploadItemCreator.h"
#include "CGIResponseWriter.h"
#include "RssViewImpl.h"
#include "CharCodeConverter.h"
#include "TemplateExecutorImpl.h"

#ifdef MAKE_THUMBNAIL
#include "ThumbnailWriter.h"
#endif
#include "Auxiliary.h"
#include "Misc.h"

#include "apr_mmap.h"
#include "apr_strings.h"
#define APR_WANT_STDIO
#define APR_WANT_STRFUNC
#include "apr_want.h"

#include <stdlib.h>

#include <iostream>

static const char *PROGRESS_LIST_FILE_NAME  = ".progress_list";
static const char *POSTER_LIST_FILE_NAME    = ".poster_list";

static const char *read_param(apr_pool_t *pool)
{
    char *request;

    APR_PCALLOC(request, char *, pool, MAX_PARAM_LENGTH+1);
    fread(request, sizeof(char), MAX_PARAM_LENGTH, stdin);

    return request;
}

static apr_table_t *get_param_table(apr_pool_t *pool)
{
    const char *param;
    const char *token;
    apr_table_t *table;
    const char *key;
    const char *value;

    table = apr_table_make(pool, MAX_PARAM_NUMBER);

    param = read_param(pool);
    while ((*param != '\0') && (token = get_token(pool, &param, '&'))) {
        key = rfc1738_decode(pool, get_token(pool, &token, '='));
        value = rfc1738_decode(pool, get_token(pool, &token, '='));

        apr_table_set(table, key, value);
    }

    return table;
}

static apr_size_t get_page()
{
    return get_page(getenv("QUERY_STRING"));
}

static apr_size_t get_upload_id()
{
    return get_upload_id(getenv("QUERY_STRING"));
}

static const char *get_name_encoding()
{
    return get_name_encoding(getenv("USER_AGENT"));
}

static void print_last_modified(apr_time_t mtime=apr_time_now())
{
    char time_str[APR_RFC822_DATE_LEN];

    apr_rfc822_date(time_str, mtime);

    cout << "Last-Modified: " << time_str << "\r\n";
}

static void print_content_type(const char *str="text/html")
{
    cout << "Content-Type: " << str << "\r\n";
}

static void print_header_delimiter()
{
    cout << "\r\n";
}

static bool is_header_only()
{
    return strcmp(safe_getenv("REQUEST_METHOD"), "HEAD") == 0;
}

static void read_file(apr_pool_t *pool, uconfig *config,
                       const char *file_name, apr_off_t size, void **data)
{
    char *file_path;
    apr_file_t *file;
    apr_mmap_t *file_map;
    apr_finfo_t info;
    apr_status_t status;

    *data = NULL;

    if (apr_filepath_merge(&file_path, config->file_dir, file_name,
                           APR_FILEPATH_NOTABOVEROOT, pool) != APR_SUCCESS) {
        throw "ǡե̾ǤޤǤ(1)";
    }

    status = apr_file_open(&file, file_path,
                           APR_READ|APR_BINARY|APR_SHARELOCK, APR_OS_DEFAULT,
                           pool);
    if (status != APR_SUCCESS) {
        if (status == APR_ENOENT) {
            // ե뤬¸ߤʤ硥
            return;
        }
        throw "ǡե򳫤ޤǤ(1)";
    }

    if (apr_file_lock(file, APR_FLOCK_SHARED) != APR_SUCCESS) {
        throw "ǡեåǤޤǤ(1)";
    }

    if (apr_file_info_get(&info, APR_FINFO_SIZE, file) != APR_SUCCESS) {
        throw "ǡեΥǤޤǤ(1)";
    }

    if (info.size != size) {
        throw "ǡեΥǤ(1)";
    }

    if (apr_mmap_create(&file_map, file, 0, static_cast<apr_size_t>(info.size),
                        APR_MMAP_READ, pool) != APR_SUCCESS) {
        throw "ǡե mmap ˼Ԥޤ(1)";
    }

    APR_PALLOC(*data, void *, pool, static_cast<apr_size_t>(info.size));
    memcpy(*data, file_map->mm, static_cast<apr_size_t>(info.size));

    if (apr_mmap_delete(file_map) != APR_SUCCESS) {
        throw "ǡե munmap ˼Ԥޤ(1)";
    }

    if (apr_file_unlock(file) != APR_SUCCESS) {
        throw "ǡեΥåǤޤǤ(1)";
    }

    apr_file_close(file);
}

static void write_file(apr_pool_t *pool, uconfig *config,
                       const char *file_name, void *data, apr_size_t size)
{
    char *file_path;
    apr_file_t *file;
    apr_mmap_t *file_map;
    apr_off_t offset;

    if (apr_filepath_merge(&file_path, config->file_dir, file_name,
                           APR_FILEPATH_NOTABOVEROOT, pool) != APR_SUCCESS) {
        throw "ǡե̾ǤޤǤ(2)";
    }

    if (apr_file_open(&file, file_path,
                      APR_READ|APR_WRITE|APR_CREATE|APR_BINARY|APR_SHARELOCK,
                      APR_OS_DEFAULT, pool)  != APR_SUCCESS) {
        throw "ǡե򳫤ޤǤ(2)";
    }

    if (apr_file_lock(file, APR_FLOCK_EXCLUSIVE) != APR_SUCCESS) {
        throw "ǡեåǤޤǤ(2)";
    }

    offset = size - 1;
    if (apr_file_seek(file, APR_SET, &offset) != APR_SUCCESS) {
        throw "ǡե seek ˼Ԥޤ(2)";
    }

    if (apr_file_putc('*', file) != APR_SUCCESS) {
        throw "ǡե write ˼Ԥޤ(2)";
    }

    if (apr_mmap_create(&file_map, file, 0, size,
                        APR_MMAP_READ|APR_MMAP_WRITE, pool) != APR_SUCCESS) {
        throw "ǡե map ˼Ԥޤ(2)";
    }

    memcpy(file_map->mm, data, size);

    if (apr_mmap_delete(file_map) != APR_SUCCESS) {
        throw "ǡե unmap ˼Ԥޤ(2)";
    }

    if (apr_file_unlock(file) != APR_SUCCESS) {
        throw "ǡեΥåǤޤǤ(2)";
    }

    apr_file_close(file);
}

/**
 * åץɤοĽΥꥹȤɤ߹ߤޤ
 *
 * ֤ꥹȤƬϥߡΥȥǼºݤˤϻѤޤ
 * Null Object ѥ
 *
 * @param[in] pool ס
 * @param[in] config åץ
 * @return ĽΥꥹ
 */
static pprogress *load_progress_list(apr_pool_t *pool, uconfig *config)
{
    pprogress *progress_list;

    read_file(pool, config, PROGRESS_LIST_FILE_NAME,
              sizeof(pprogress)*PROGRESS_LIST_NUMBER,
              reinterpret_cast<void **>(&progress_list));

    if (UNLIKELY(progress_list == NULL)) {
        APR_PCALLOC(progress_list, pprogress *, pool,
                    sizeof(pprogress)*PROGRESS_LIST_NUMBER);
    }

    return progress_list;
}

/**
 * åץɤοĽΥꥹȤ񤭽Фޤ
 *
 * @param[in] pool ס
 * @param[in] config åץ
 * @param[in] progress_list ĽΥꥹ
 */
static void store_progress_list(apr_pool_t *pool, uconfig *config,
                                pprogress *progress_list)
{
    write_file(pool, config, PROGRESS_LIST_FILE_NAME, progress_list,
               sizeof(pprogress)*PROGRESS_LIST_NUMBER);
}

/**
 * ƾΥꥹȤɤ߹ߤޤ
 *
 * @param[in] pool ס
 * @param[in] config åץ
 * @return ƾΥꥹ
 */
static poster *load_poster_list(apr_pool_t *pool, uconfig *config)
{
    poster *poster_list;

    read_file(pool, config, POSTER_LIST_FILE_NAME,
              sizeof(poster)*POSTER_LIST_NUMBER,
              reinterpret_cast<void **>(&poster_list));

    if (UNLIKELY(poster_list == NULL)) {
        APR_PCALLOC(poster_list, poster *, pool, sizeof(poster)*POSTER_LIST_NUMBER);
    }

    return poster_list;
}

/**
 * ƾΥꥹȤ񤭽Фޤ
 *
 * @param[in] pool ס
 * @param[in] config åץ
 * @param poster_list ƾΥꥹ
 */
static void store_poster_list(apr_pool_t *pool, uconfig *config,
                              poster *poster_list)
{
    write_file(pool, config, POSTER_LIST_FILE_NAME, poster_list,
               sizeof(poster)*POSTER_LIST_NUMBER);
}

static void print_status(http_status status=HTTP_OK)
{
    switch (status) {
    case HTTP_OK:
        cout << "Status: " << "200 OK" << "\r\n";
        return;
    case HTTP_NO_CONTENT:
        cout << "Status: " << "204 No Content" << "\r\n";
        return;
    case HTTP_BAD_REQUEST:
        cout << "Status: " << "400 Bad Request" << "\r\n";
        return;
    case HTTP_NOT_MODIFIED:
        cout << "Status: " << "304 Not Modified" << "\r\n";
        return;
    case HTTP_FORBIDDEN:
        cout << "Status: " << "403 Forbidden" << "\r\n";
        return;
    }
}

static int error(apr_pool_t *pool, uconfig *config, const char *error)
{
    page_template *tmpl = config->get_error_template(pool);
    auto_ptr<variable_map> vmap(get_error_vmap(pool, config, tmpl, error));

    print_status();
    print_content_type();
    print_header_delimiter();

    CGIResponseWriter writer;
    CGITemplateExecutor texecutor(pool, tmpl->imap, vmap.get(), writer);
    texecutor.exec(tmpl->node);

    return EXIT_SUCCESS;
}

static int redirect_top()
{
    // POST ˥ڡñ˥ɤǤ褦ˤ뤿ᡤHTML 
    // 쥯Ȥ롥
    print_status();
    print_content_type();
    print_header_delimiter();

    cout << "<?xml version=\"1.0\" encoding=\"EUC-JP\"?>";
    cout << "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">";
    cout << "<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"ja\" >";
    cout << " <head>";
    cout << "  <meta http-equiv=\"content-type\" content=\"application/xhtml+xml\" />";
    cout << "  <meta http-equiv=\"refresh\" content=\"0;url=../\" />";
    cout << "  <title>mod_uploader</title>";
    cout << " </head>";
    cout << " <body />";
    cout << "</html>";

    return EXIT_SUCCESS;
}

static int input_pass(apr_pool_t *pool, uconfig *config, const char *arg,
                      UploadItem::header *header)
{
    try {
        page_template *tmpl = config->get_download_template(pool);
        auto_ptr<variable_map> vmap(get_download_vmap(pool, config, tmpl, header));

        print_status();
        print_content_type();
        print_header_delimiter();

        CGIResponseWriter writer;
        CGITemplateExecutor texecutor(pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(tmpl->node);

        return EXIT_SUCCESS;
    } catch(const char *message) {
        cout << "<div class=\"warning\">";
        cout << message;
        cout << "</div>";

        return EXIT_FAILURE;
    }
}

static int view(apr_pool_t *pool, uconfig *config, const char *arg)
{
    if (!is_modified_since(pool, getenv("HTTP_IF_MODIFIED_SINCE"),
                           config->item_list->get_mtime())) {
        print_status(HTTP_NOT_MODIFIED);
        print_header_delimiter();

        return EXIT_SUCCESS;
    }

    print_status();
    print_last_modified(config->item_list->get_mtime());
    print_content_type();
    print_header_delimiter();

    if (is_header_only()) {
        return EXIT_SUCCESS;
    }

    try {
        page_template *tmpl = config->get_view_template(pool);
        auto_ptr<variable_map> vmap(get_view_vmap(pool, config, tmpl, get_page()));

        CGIResponseWriter writer;
        CGITemplateExecutor texecutor(pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(tmpl->node);

        return EXIT_SUCCESS;
    } catch(const char *message) {
        cout << "<div class=\"warning\">";
        cout << message;
        cout << "</div>";

        return EXIT_FAILURE;
    }
}

#ifdef MAKE_THUMBNAIL
static int thumbnail(apr_pool_t *pool, uconfig *config, const char *arg)
{
    if (!is_modified_since(pool, getenv("HTTP_IF_MODIFIED_SINCE"),
                           config->item_list->get_mtime())) {
        print_status(HTTP_NOT_MODIFIED);
        print_header_delimiter();

        return EXIT_SUCCESS;
    }

    print_status();
    print_last_modified(config->item_list->get_mtime());
    print_content_type();
    print_header_delimiter();

    if (is_header_only()) {
        return EXIT_SUCCESS;
    }

    try {
        page_template *tmpl = config->get_thumb_template(pool);
        auto_ptr<variable_map> vmap(get_thumb_vmap(pool, config, tmpl, get_page()));

        CGIResponseWriter writer;
        CGITemplateExecutor texecutor(pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(tmpl->node);

        return EXIT_SUCCESS;
    } catch(const char *message) {
        cout << "<div class=\"warning\">";
        cout << message;
        cout << "</div>";

        return EXIT_FAILURE;
    }
}
#endif

static int rss(apr_pool_t *pool, uconfig *config, const char *arg)
{
    if (!is_modified_since(pool, getenv("HTTP_IF_MODIFIED_SINCE"),
                           config->item_list->get_mtime())) {
        print_status(HTTP_NOT_MODIFIED);
        print_header_delimiter();

        return EXIT_SUCCESS;
    }

    print_status();
    print_last_modified(config->item_list->get_mtime());
    print_content_type(CGIRssView::content_type);
    print_header_delimiter();

    if (is_header_only()) {
        return EXIT_SUCCESS;
    }

    try {
        CGIResponseWriter writer;
        CGIRssView view(pool, writer);
        view.exec(config->url, config->per_page_item_number, config->item_list);

        return EXIT_SUCCESS;
    } catch(const char *message) {
        cout << "<div class=\"warning\">";
        cout << message;
        cout << "</div>";

        return EXIT_FAILURE;
    }
}

static int upload(apr_pool_t *pool, uconfig *config, const char *arg)
{
    const char *address;
    apr_size_t upload_id;
    poster *poster_list;
    pprogress *progress_list;
    pprogress *progress;

    if (strcmp(safe_getenv("REQUEST_METHOD"), "POST")) {
        print_status(HTTP_BAD_REQUEST);
        print_header_delimiter();

        return EXIT_FAILURE;
    }

    address = safe_getenv("REMOTE_ADDR");
    poster_list = load_poster_list(pool, config);
    if (is_poster_list_contain(poster_list, address)) {
        return error(pool, config,
                     apr_psprintf(pool, "%" APR_SIZE_T_FMT  " ô֤ϺƤǤޤ",
                                  POST_INTERVAL_SEC));
    }

    upload_id = get_upload_id();
    progress_list = load_progress_list(pool, config);
    progress = add_progress_list(progress_list, upload_id);

    try {
        UploadItem::header *header;
        const char *tmp_path;
        const char *file_name;
        apr_size_t content_size;

        content_size = static_cast<apr_size_t>(apr_atoi64(safe_getenv("CONTENT_LENGTH")));
        progress->total_size = content_size;

        CGIPostReader reader(progress);
        auto_ptr<RFC1867Data::query_map>
            qmap(config->uparser->parse(pool, reader, safe_getenv("CONTENT_TYPE"),
                                        content_size));

        try {
            header = UploadItemCreator::create_header(pool, qmap.get(), &tmp_path);
        } catch(const char *) {
            CGIRFC1867Parser::clean_tmp_file(pool, qmap.get());

            throw;
        }

        auto_ptr<UploadItemWriter> writer(config->get_uwriter(pool));
        file_name = writer->write(header, tmp_path);

        config->write_lock();
        try {
            config->item_list->add(file_name, true);
            config->write_unlock();
        } catch(const char *) {
            config->write_unlock();
            throw;
        }

        add_poster_list(poster_list, address);
        progress->end_time = apr_time_now();

        store_poster_list(pool, config, poster_list);
        store_progress_list(pool, config, progress_list);

        return redirect_top();
    } catch(const char *message) {
        progress->end_time = apr_time_now();

        return error(pool, config, message);
    }
}

static int progress(apr_pool_t *pool, uconfig *config, const char *arg)
{
    print_status();
    print_content_type();
    print_header_delimiter();

    try {
        page_template *tmpl = config->get_progress_template(pool);

        auto_ptr<variable_map> vmap(get_progress_vmap(pool, config, tmpl));

        CGIResponseWriter writer;
        CGITemplateExecutor texecutor(pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(tmpl->node);

        return EXIT_SUCCESS;
    } catch(const char *message) {
        cout << "<div class=\"warning\">";
        cout << message;
        cout << "</div>";

        return EXIT_FAILURE;
    }
}

static int progress_data(apr_pool_t *pool, uconfig *config, const char *arg)
{
    apr_size_t upload_id;
    pprogress *progress_list;
    pprogress *progress;
    char *progress_str;

    upload_id = get_upload_id();

    print_status();
    print_content_type();
    print_header_delimiter();

    // upload_id  0 λϥ顼
    if (UNLIKELY(upload_id == 0)) {
        cout << "e";
        return EXIT_SUCCESS;
    }

    progress_list = load_progress_list(pool, config);
    progress = get_progress(progress_list, upload_id);

    if (UNLIKELY(progress == progress_list)) {
        cout << "u";
        return EXIT_SUCCESS;
    }

    progress_str = apr_psprintf(pool, "s %" APR_SIZE_T_FMT " %" APR_SIZE_T_FMT,
                                progress->total_size, progress->read_size);
    cout << progress_str;

    return EXIT_SUCCESS;
}

static int download(apr_pool_t *pool, uconfig *config, const char *arg)
{
    const char *name;

    name = get_token(pool, &arg, '/');
    if (name == NULL) {
        print_status(HTTP_BAD_REQUEST);
        print_header_delimiter();

        return EXIT_FAILURE;
    }

    {
        // ե̾Υå
        const char *pos = name;

        while (isalnum(*pos) || (*pos == '.')) {
            pos++;
        }

        if ((static_cast<size_t>(pos-name) != strlen(name)) ||
            (strlen(name) > APR_PATH_MAX)) {
            print_status(HTTP_BAD_REQUEST);
            print_header_delimiter();

            return EXIT_FAILURE;
        }
    }

    try {
        apr_file_t *file;
        UploadItem::header *header;
        const char *file_name;
        const char *password;
        const char *open_mode;
        const char *content_dispos;
        char buffer[2048];
        apr_size_t read_size = sizeof(buffer);

        auto_ptr<UploadItemReader> reader(config->get_ureader(pool));
        header = reader->read(name, &file);

        password = apr_table_get(get_param_table(pool), UploadItem::DOWNLOAD_PASS_PARAM);

        // DL pass Υå
        if ((strlen(header->download_pass) > 0) &&
            ((password == NULL) ||
             (strncmp(header->download_pass, password, strlen(header->download_pass)) != 0))) {
            return input_pass(pool, config, arg, header);
        }

        if (!is_modified_since(pool, getenv("HTTP_IF_MODIFIED_SINCE"), header->time)) {
            print_status(HTTP_NOT_MODIFIED);
            print_header_delimiter();

            return EXIT_SUCCESS;
        }

        print_status();
        print_last_modified(config->item_list->get_mtime());
        print_content_type(header->file_mime);

        if (is_header_only()) {
            print_header_delimiter();
            return EXIT_SUCCESS;
        }

        file_name = CharCodeConverter::convert(pool, header->file_name,
                                               CharCodeConverter::DEFAULT_CODE,
                                               get_name_encoding());

        // ɽ⡼ or ɥ⡼
        open_mode = (strlen(safe_getenv("QUERY_STRING")) == 0) ? "inline" : "attachment";

        // Content-Disposition
        content_dispos = apr_psprintf(pool, "%s; filename=\"%s\"", open_mode, file_name);
        cout << "Content-Disposition: " << content_dispos << "\r\n";

        print_header_delimiter();

        while (apr_file_read(file, buffer, &read_size) == APR_SUCCESS) {
            cout.write(buffer, read_size);
            read_size = sizeof(buffer);
        }

        return EXIT_SUCCESS;
    } catch(const char *message) {
        return error(pool, config, message);
    }
}

static int remove(apr_pool_t *pool, uconfig *config, const char *arg)
{
    apr_table_t *ptable;

    if (strcmp(safe_getenv("REQUEST_METHOD"), "POST")) {
        print_status(HTTP_BAD_REQUEST);
        print_header_delimiter();

        return EXIT_FAILURE;
    }

    try {
        const char *file_name;
        const char *password;

        ptable = get_param_table(pool);

        file_name = apr_table_get(ptable, UploadItem::FILE_PARAM);
        password = apr_table_get(ptable, UploadItem::REMOVE_PASS_PARAM);

        if ((file_name == NULL) || (password == NULL)) {
            return error(pool, config, "եޤϥѥɤꤵƤޤ");
        }

        config->item_list->remove(file_name, password);

        return redirect_top();
    } catch(const char *message) {
        return error(pool, config, message);
    }
}

int command_handler(apr_pool_t *pool, uconfig *config, const char *arg)
{
    const char *command;

    command = get_token(pool, &arg, '/');

    if (LIKELY(*command == '\0')) {
        return view(pool, config, arg);
    } else if (strcmp(command, RSS_COMMAND) == 0) {
        return rss(pool, config, arg);
    } else if (strcmp(command, UPLOAD_COMMAND) == 0) {
        return upload(pool, config, arg);
    } else if (strcmp(command, PROGRESS_COMMAND) == 0) {
        return progress(pool, config, arg);
    } else if (strcmp(command, PROGRESS_DATA_COMMAND) == 0) {
        return progress_data(pool, config, arg);
    } else if (strcmp(command, DOWNLOAD_COMMAND) == 0) {
        return download(pool, config, arg);
    } else if (strcmp(command, REMOVE_COMMAND) == 0) {
        return remove(pool, config, arg);
#ifdef MAKE_THUMBNAIL
    } else if (strcmp(command, THUMBNAIL_COMMAND) == 0) {
        return thumbnail(pool, config, arg);
#endif
    } else {
        return error(pool, config, "ݡȤƤʤޥɤǤ");
    }
}

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
