/******************************************************************************
 * mod_uploader / mod_uploader_apache.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: mod_uploader_apache.cpp 917 2005-11-21 11:19:09Z svn $
 *****************************************************************************/

/**
 * @file
 * @brief Apache ⥸塼Υȥؿ
 */

#include "mod_uploader_apache.h"
#include "apache_handler.h"
#include "uploader_func.h"

#include "ApacheUploaderConfig.h"
#include "Misc.h"

#include "http_core.h"
#include "http_log.h"

#ifdef AP_NEED_SET_MUTEX_PERMS
#ifdef __cplusplus
extern "C" {
#endif
#include "unixd.h"
#ifdef __cplusplus
}
#endif
#endif
#define APR_WANT_STRFUNC
#include "apr_want.h"

#include <stdlib.h>

#ifdef DEBUG
#include <iostream>
#endif

#define DIRECTIVE_ENTRY(directive, member, showing)                     \
    AP_INIT_TAKE1(directive,                                            \
                  reinterpret_cast<const char*(*)()>(set_ ## member), NULL, \
                  ACCESS_CONF, showing)

#define DIRECTIVE_HANDLER(member, value) \
    static const char *set_ ## member(cmd_parms *parms, void *mconfig, char *arg) \
    {                                                                   \
        uconfig *config = static_cast<uconfig *>(mconfig);              \
        config->member = value;                                         \
        return NULL;                                                    \
    }

#define INFO(command, target) \
    ap_log_rerror(APLOG_MARK, APLOG_INFO, 0, r, \
                  "[" PACKAGE_NAME "] (%s) %s", command, target)

#ifdef DEBUG
#define SERROR(format, str)                                             \
    fprintf(stderr, "[" PACKAGE_NAME "] " format, str); fputc('\n', stderr)
#define RERROR(format, str)                                             \
    fprintf(stderr, "[" PACKAGE_NAME "] " format, str); fputc('\n', stderr)
#else
#define SERROR(format, str)                                             \
    ap_log_error(APLOG_MARK, APLOG_CRIT, 0, server, "[" PACKAGE_NAME "] " format, str)
#define RERROR(format, str)                                             \
    ap_log_rerror(APLOG_MARK, APLOG_CRIT, 0, r, "[" PACKAGE_NAME "] " format, str)
#endif


sconfig *get_sconfig(request_rec *r)
{
    return static_cast<sconfig *>(ap_get_module_config(r->server->module_config,
                                                       &uploader_module));
}

ApacheUploaderConfig *get_uconfig(request_rec *r)
{
    return static_cast<uconfig *>(ap_get_module_config(r->per_dir_config,
                                                       &uploader_module));
}

extern "C" {
static int uploader_handler(request_rec *r)
{
    uconfig *config;

    if (strcmp(r->handler, "uploader")) {
        return DECLINED;
    }

    if (UNLIKELY(!ap_is_initial_req(r))) {
        return DECLINED;
    }

    config = get_uconfig(r);

    try {
        config->init(r, get_sconfig(r));

        return command_handler(r, config, r->path_info);
    } catch(const char *message) {
        RERROR("Exception: %s", message);

        return HTTP_INTERNAL_SERVER_ERROR;
    }
}

static apr_status_t finalize_server_config(void *data)
{
    sconfig *config;

    config = static_cast<sconfig *>(data);

    if (config->upload_lock != NULL) {
        apr_global_mutex_destroy(config->upload_lock);
        config->upload_lock = NULL;
    }
    if (config->revision_shm != NULL) {
        apr_shm_destroy(config->revision_shm);
        config->revision_shm = NULL;
    }
    if (config->poster_list_shm != NULL) {
        apr_shm_destroy(config->poster_list_shm);
        config->poster_list_shm = NULL;
    }
    if (config->progress_list_shm != NULL) {
        apr_shm_destroy(config->progress_list_shm);
        config->progress_list_shm = NULL;
    }

    return APR_SUCCESS;
}

static apr_status_t finalize_dir_config(void *data)
{
    uconfig *config;

    config = static_cast<uconfig *>(data);
    config->finalize();

    return APR_SUCCESS;
}

static apr_status_t create_lock(apr_pool_t *pool, apr_global_mutex_t **lock,
                                const char *file_path, server_rec *server)
{
    apr_status_t status;
    char error[MAX_ERROR_MESSAGE];

    status = apr_global_mutex_create(lock, file_path, APR_LOCK_DEFAULT, pool);
    if (status != APR_SUCCESS) {
        SERROR("Can not create global mutex (%s).", file_path);
        SERROR("(%s)", apr_strerror(status, error, sizeof(error)));

        return status;
    }

#ifdef AP_NEED_SET_MUTEX_PERMS
    status = unixd_set_global_mutex_perms(*lock);
    if (status != APR_SUCCESS) {
        SERROR("Can not set permissions on global mutex (%s).", file_path);
        SERROR("(%s)", apr_strerror(status, error, sizeof(error)));

        return status;
    }
#endif

    return APR_SUCCESS;
}

static apr_status_t create_shm(apr_pool_t *pool, apr_shm_t **shm_data,
                               void **data, const char *file_path,
                               apr_size_t size, server_rec *server)
{
    apr_status_t status;
    char error[MAX_ERROR_MESSAGE];

    // ޤϡAnonymous Shared Memory Ƥߤ롥
    status = apr_shm_create(shm_data, size, NULL, pool);
    if (status == APR_ENOTIMPL) {
        status = apr_shm_create(shm_data, size, file_path, pool);
    }
    if (status != APR_SUCCESS) {
        SERROR("Can not create shared segment file (%s).", file_path);
        SERROR("(%s)", apr_strerror(status, error, sizeof(error)));

        return status;
    }

    *data = apr_shm_baseaddr_get(*shm_data);
    memset(*data, 0, size);

    return APR_SUCCESS;
}

static apr_status_t attach_lock(apr_pool_t *pool, apr_global_mutex_t **lock,
                                const char *file_path, server_rec *server)
{
    apr_status_t status;
    char error[MAX_ERROR_MESSAGE];

    status = apr_global_mutex_child_init(lock, file_path, pool);

    if (status != APR_SUCCESS) {
           SERROR("Can not attach to global mutex (%s).", file_path);
           SERROR("(%s)", apr_strerror(status, error, sizeof(error)));

           return status;
    }

    return APR_SUCCESS;
}

static apr_status_t attach_shm(apr_pool_t *pool, apr_shm_t **shm_data,
                               void **data, const char *file_path,
                               server_rec *server)
{
    apr_status_t status;
    char error[MAX_ERROR_MESSAGE];

    if (*shm_data == NULL) {
        status = apr_shm_attach(shm_data, file_path, pool);

        if (status != APR_SUCCESS) {
            SERROR("Can not attach to shared segment file (%s).", file_path);
            SERROR("(%s)", apr_strerror(status, error, sizeof(error)));

            return status;
        }
    }

    *data = apr_shm_baseaddr_get(*shm_data);

    return APR_SUCCESS;
}

static void *create_server_config(apr_pool_t *p, server_rec *s)
{
    sconfig *config;
    const char *tmp_dir;

    APR_PCALLOC(config, sconfig *, p, sizeof(sconfig));

    apr_temp_dir_get(&tmp_dir, p);

    apr_filepath_merge(&(config->upload_lock_path), tmp_dir,
                       apr_pstrcat(p, UPLOAD_LOCK_NAME, "_", s->server_hostname, NULL),
                       APR_FILEPATH_NOTABOVEROOT, p);
    apr_filepath_merge(&(config->revision_path), tmp_dir,
                       apr_pstrcat(p, REVISION_NAME, "_", s->server_hostname, NULL),
                       APR_FILEPATH_NOTABOVEROOT, p);
    apr_filepath_merge(&(config->poster_list_path), tmp_dir,
                       apr_pstrcat(p, POSTER_LIST_NAME, "_", s->server_hostname, NULL),
                       APR_FILEPATH_NOTABOVEROOT, p);
    apr_filepath_merge(&(config->progress_list_path), tmp_dir,
                       apr_pstrcat(p, PROGRESS_LIST_NAME, "_", s->server_hostname, NULL),
                       APR_FILEPATH_NOTABOVEROOT, p);

    config->revision_shm = NULL;
    config->poster_list_shm = NULL;
    config->progress_list_shm = NULL;

    return config;
}

static void *create_dir_config(apr_pool_t *pool, char *dirspec)
{
    uconfig *config;

    APR_PCALLOC(config, uconfig *, pool, sizeof(uconfig));
    new(config) uconfig;

    apr_pool_cleanup_register(pool, config, finalize_dir_config, apr_pool_cleanup_null);

    return config;
}

static int post_config_server(apr_pool_t *pool, server_rec *server)
{
    sconfig *config;

    config = static_cast<sconfig *>(ap_get_module_config(server->module_config, &uploader_module));

    if (create_lock(pool, &(config->upload_lock), config->upload_lock_path,
                    server) != APR_SUCCESS) {
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    if (create_shm(pool, &(config->revision_shm),
                   reinterpret_cast<void **>(&config->revision),
                   config->revision_path, sizeof(apr_size_t),
                   server) != APR_SUCCESS) {
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    if (create_shm(pool, &(config->poster_list_shm),
                   reinterpret_cast<void **>(&config->poster_list),
                   config->poster_list_path, sizeof(poster)*POSTER_LIST_NUMBER,
                   server) != APR_SUCCESS) {
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    if (create_shm(pool, &(config->progress_list_shm),
                   reinterpret_cast<void **>(&(config->progress_list)),
                   config->progress_list_path, sizeof(pprogress)*PROGRESS_LIST_NUMBER,
                   server) != APR_SUCCESS) {
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    apr_pool_cleanup_register(pool, config, finalize_server_config, apr_pool_cleanup_null);

    return OK;
}

static void child_init_server(apr_pool_t *pool, server_rec *server)
{
    sconfig *config;

    config = static_cast<sconfig *>(ap_get_module_config(server->module_config, &uploader_module));

    if (attach_lock(pool, &(config->upload_lock), config->upload_lock_path,
                    server)  != APR_SUCCESS) {
        return;
    }

    if (attach_shm(pool, &(config->revision_shm),
                   reinterpret_cast<void **>(&config->revision),
                   config->revision_path, server) != APR_SUCCESS) {
        return;
    }

    if (attach_shm(pool, &(config->poster_list_shm),
                   reinterpret_cast<void **>(&config->poster_list),
                   config->poster_list_path, server) != APR_SUCCESS) {
        return;
    }

    if (attach_shm(pool, &(config->progress_list_shm),
                   reinterpret_cast<void **>(&config->progress_list),
                   config->progress_list_path, server) != APR_SUCCESS) {
        return;
    }
}

static int post_config(apr_pool_t *pconf, apr_pool_t *plog,
                       apr_pool_t *ptemp, server_rec *s)
{
    void *user_data;
    apr_status_t status;

    apr_pool_userdata_get(&user_data, USER_DATA_KEY, s->process->pool);
    if (user_data == NULL) {
        apr_pool_userdata_set(reinterpret_cast<const void *>(1), USER_DATA_KEY,
                              apr_pool_cleanup_null, s->process->pool);
        return OK;
    }

    do {
        status = post_config_server(plog, s);

        if (status != OK) {
            return status;
        }
    } while ((s = s->next) != NULL);

    // windows Ǥ򤹤ȥӥ̾ѤäƤޤ餷
#ifndef WIN32
    ap_add_version_component(pconf, PACKAGE_NAME "/" PACKAGE_VERSION);
#endif

    return OK;
}

static void child_init(apr_pool_t *p, server_rec *s)
{
    do {
        child_init_server(p, s);
    } while ((s = s->next) != NULL);
}

DIRECTIVE_HANDLER(url,                      apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(file_dir,                 apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(thumb_dir,                apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(tmp_dir,                  apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(view_tmpl_path,           apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(progress_tmpl_path,       apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(download_tmpl_path,       apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(thumb_tmpl_path,          apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(error_tmpl_path,          apr_pstrdup(parms->pool, arg));
DIRECTIVE_HANDLER(max_file_size,            atoi(arg) * 1024);
DIRECTIVE_HANDLER(total_file_size_limit,    apr_atoi64(arg) * 1024);
DIRECTIVE_HANDLER(total_file_number_limit,  atoi(arg));
DIRECTIVE_HANDLER(per_page_item_number,     atoi(arg));

static const command_rec uploader_cmds[] = {
    DIRECTIVE_ENTRY(ApacheUploaderConfig::URL_PARAM,
                    url,
                    "Uploader URL"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::FILE_DIRECTORY_PARAM,
                    file_dir,
                    "Upload File Directory"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::THUMB_DIRECTORY_PARAM,
                    thumb_dir,
                    "Thumbnail File Directory"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::TMP_DIRECTORY_PARAM,
                    tmp_dir,
                    "Temporary File Directory"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::VIEW_TEMPLATE_FILE_PARAM,
                    view_tmpl_path,
                    "View Template File"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::PROGRESS_TEMPLATE_FILE_PARAM,
                    progress_tmpl_path,
                    "Progress Template File"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::DOWNLOAD_TEMPLATE_FILE_PARAM,
                    download_tmpl_path,
                    "Download Template File"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::THUMB_TEMPLATE_FILE_PARAM,
                    thumb_tmpl_path,
                    "Thumbnail Template File"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::ERROR_TEMPLATE_FILE_PARAM,
                    error_tmpl_path,
                    "Error Template File"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::MAX_FILE_SIZE_PARAM,
                    max_file_size,
                    "Max File Size (KB)"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::TOTAL_FILE_SIZE_LIMIT_PARAM,
                    total_file_size_limit,
                    "Total File Size Limit (KB)"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::TOTAL_FILE_NUMBER_LIMIT_PARAM,
                    total_file_number_limit,
                    "Total File Number Limit"),
    DIRECTIVE_ENTRY(ApacheUploaderConfig::PER_PAGE_ITEM_NUMBER_PARAM,
                    per_page_item_number,
                    "Per Page Item Number"),
    { NULL }
};

static void uploader_register_hooks(apr_pool_t *pool)
{
    ap_hook_post_config(post_config, NULL, NULL, APR_HOOK_REALLY_FIRST);
    ap_hook_child_init(child_init, NULL, NULL, APR_HOOK_REALLY_FIRST);
    ap_hook_handler(uploader_handler, NULL, NULL, APR_HOOK_MIDDLE);
}

module AP_MODULE_DECLARE_DATA uploader_module = {
    STANDARD20_MODULE_STUFF,
    create_dir_config,
    NULL,
    create_server_config,
    NULL,
    uploader_cmds,
    uploader_register_hooks
};
}

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
