/******************************************************************************
 * mod_uploader / mod_uploader_lighttpd.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: mod_uploader_lighttpd.cpp 901 2005-10-26 19:32:10Z svn $
 *****************************************************************************/

/**
 * @file
 * @brief lighttpd ⥸塼Υȥؿ
 */

extern "C" {
#include "base.h"
#include "buffer.h"
#include "connections.h"
#include "http_chunk.h"
#include "joblist.h"
#include "log.h"
#include "stream.h"
}
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION

#include "mod_uploader_lighttpd.h"
#include "lighttpd_handler.h"
#include "uploader_func.h"

#include "Misc.h"

#include "apr_strings.h"
#define APR_WANT_MEMFUNC
#include "apr_want.h"

#include <sys/select.h>

#include <map>

#ifdef DEBUG
#include <iostream>
#endif

#define RWARN(str)                                                      \
    log_error_write(srv, __FILE__, __LINE__, "ss", "Warning:", str);

#define RERROR(str)                                                     \
    log_error_write(srv, __FILE__, __LINE__, "ss", "Exception:", str);

static const apr_size_t READ_BUFFER_SIZE = 2048;


/**
 * ܤνԤޤ
 */
static void init_config(plugin_config *pconfig)
{
    pconfig->url                        = buffer_init();
    pconfig->path                       = buffer_init();
    pconfig->file_dir                   = buffer_init();
    pconfig->thumb_dir                  = buffer_init();
    pconfig->tmp_dir                    = buffer_init();
    pconfig->view_tmpl_path             = buffer_init();
    pconfig->progress_tmpl_path         = buffer_init();
    pconfig->download_tmpl_path         = buffer_init();
    pconfig->thumb_tmpl_path            = buffer_init();
    pconfig->error_tmpl_path            = buffer_init();
    pconfig->max_file_size              = buffer_init();
    pconfig->total_file_size_limit      = buffer_init();
    pconfig->total_file_number_limit    = buffer_init();
    pconfig->per_page_item_number       = buffer_init();
}

/**
 * ܤνλԤޤ
 */
static void free_config(plugin_config *pconfig)
{
    buffer_free(pconfig->url);
    buffer_free(pconfig->path);
    buffer_free(pconfig->file_dir);
    buffer_free(pconfig->thumb_dir);
    buffer_free(pconfig->tmp_dir);
    buffer_free(pconfig->view_tmpl_path);
    buffer_free(pconfig->progress_tmpl_path);
    buffer_free(pconfig->download_tmpl_path);
    buffer_free(pconfig->thumb_tmpl_path);
    buffer_free(pconfig->error_tmpl_path);
    buffer_free(pconfig->max_file_size);
    buffer_free(pconfig->total_file_size_limit);
    buffer_free(pconfig->total_file_number_limit);
    buffer_free(pconfig->per_page_item_number);
}

static void free_uconfig_map(uploader_config_map *uconfig_map)
{
    uploader_config_map::const_iterator i = uconfig_map->begin();

    while (i != uconfig_map->end()) {
        delete i->second;
        i++;
    }
}

/**
 * ΥޡԤޤ
 */
static int mod_uploader_patch_connection(server *srv, connection *con,
                                         plugin_data *p)
{
    plugin_config *pconfig;
    data_config *dconfig;
    data_unset *dset;
    size_t i;
    size_t j;

    memcpy(&(p->pconfig), p->pconfig_storage[0], sizeof(plugin_config));

    for (i = 1; i < srv->config_context->used; ++i) {
        dconfig = (data_config *)srv->config_context->data[i];
        pconfig = p->pconfig_storage[i];

        if (!config_check_cond(srv, con, dconfig)) {
            continue;
        }

        for (j = 0; j < dconfig->value->used; j++) {
            dset = dconfig->value->data[j];

            // is_activated ޡʤ
            if (buffer_is_equal_string(dset->key,
                                       lighttpdUploaderConfig::ACTIVATE_PARAM,
                                       strlen(lighttpdUploaderConfig::ACTIVATE_PARAM))) {
                p->pconfig.is_activated = pconfig->is_activated;
            }
        }
    }

    return 0;
}

static uconfig *create_uconfig(plugin_config *pconfig)
{
    uconfig *config;
    apr_pool_t *pool;

    apr_pool_create(&pool, NULL);

    config = new uconfig();

    if (pconfig->url->used != 0) {
        config->url = apr_pstrdup(pool, pconfig->url->ptr);
    }
    if (pconfig->file_dir->used != 0) {
        config->file_dir = apr_pstrdup(pool, pconfig->file_dir->ptr);
    }
    if (pconfig->thumb_dir->used != 0) {
        config->thumb_dir = apr_pstrdup(pool, pconfig->thumb_dir->ptr);
    }
    if (pconfig->tmp_dir->used != 0) {
        config->tmp_dir = apr_pstrdup(pool, pconfig->tmp_dir->ptr);
    }
    if (pconfig->view_tmpl_path->used != 0) {
        config->view_tmpl_path = apr_pstrdup(pool, pconfig->view_tmpl_path->ptr);
    }
    if (pconfig->progress_tmpl_path->used != 0) {
        config->progress_tmpl_path = apr_pstrdup(pool, pconfig->progress_tmpl_path->ptr);
    }
    if (pconfig->download_tmpl_path->used != 0) {
        config->download_tmpl_path = apr_pstrdup(pool, pconfig->download_tmpl_path->ptr);
    }
    if (pconfig->thumb_tmpl_path->used != 0) {
        config->thumb_tmpl_path = apr_pstrdup(pool, pconfig->thumb_tmpl_path->ptr);
    }
    if (pconfig->error_tmpl_path->used != 0) {
        config->error_tmpl_path = apr_pstrdup(pool, pconfig->error_tmpl_path->ptr);
    }
    if (pconfig->total_file_size_limit->used != 0) {
        config->total_file_size_limit = apr_atoi64(pconfig->total_file_size_limit->ptr) * 1024;
    }
    if (pconfig->max_file_size->used != 0) {
        config->max_file_size = atoi(pconfig->max_file_size->ptr) * 1024;
    }
    if (pconfig->total_file_number_limit->used != 0) {
        config->total_file_number_limit = atoi(pconfig->total_file_number_limit->ptr);
    }
    if (pconfig->per_page_item_number->used != 0) {
        config->per_page_item_number = atoi(pconfig->per_page_item_number->ptr);
    }

    config->init(pool);

    return config;
}

static const char *get_path_info(apr_pool_t *pool, buffer *uri, buffer *path)
{
    char *path_info;
    size_t size;

    size = uri->used - path->used;
    APR_PCALLOC(path_info, char *, pool, size + 1);
    memcpy(path_info, uri->ptr+path->used, size);

    return path_info;
}

/**
 * ץ饰νԤޤ
 */
INIT_FUNC(mod_uploader_init)
{
    plugin_data *p;

    apr_initialize();

    CALLOC(p, plugin_data *, 1, sizeof(plugin_data));
    p->uconfig_map = new uploader_config_map();

    return p;
}

/**
 * ץ饰Ԥޤ
 */
SETDEFAULTS_FUNC(mod_uploader_set_defaults)
{
    /** ץ饰 */
    static config_values_t config_entry[] = {
        { lighttpdUploaderConfig::ACTIVATE_PARAM,
          NULL, T_CONFIG_BOOLEAN, T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::PATH_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::URL_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::FILE_DIRECTORY_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::THUMB_DIRECTORY_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::TMP_DIRECTORY_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::VIEW_TEMPLATE_FILE_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::PROGRESS_TEMPLATE_FILE_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::DOWNLOAD_TEMPLATE_FILE_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::THUMB_TEMPLATE_FILE_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::ERROR_TEMPLATE_FILE_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        // ʲιܤϿͤǻꤵɡT_CONFIG_SHORT Ǥ٤­ʤΤʸ󰷤
        { lighttpdUploaderConfig::MAX_FILE_SIZE_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::TOTAL_FILE_SIZE_LIMIT_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::TOTAL_FILE_NUMBER_LIMIT_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { lighttpdUploaderConfig::PER_PAGE_ITEM_NUMBER_PARAM,
          NULL, T_CONFIG_STRING,  T_CONFIG_SCOPE_CONNECTION },
        { NULL, NULL, T_CONFIG_STRING, T_CONFIG_SCOPE_UNSET }
    };

    plugin_data *p = reinterpret_cast<plugin_data *>(p_d);
    plugin_config *pconfig;
    size_t i;

    if (p == NULL) {
        return HANDLER_ERROR;
    }

    CALLOC(p->pconfig_storage, plugin_config **, 1,
           srv->config_context->used * sizeof(specific_config *));

    for (i = 0; i < srv->config_context->used; ++i) {
        CALLOC(pconfig, plugin_config *, 1, sizeof(plugin_config));

        init_config(pconfig);

        config_entry[ 0].destination    = &(pconfig->is_activated);
        config_entry[ 1].destination    = pconfig->path;
        config_entry[ 2].destination    = pconfig->url;
        config_entry[ 3].destination    = pconfig->file_dir;
        config_entry[ 4].destination    = pconfig->thumb_dir;
        config_entry[ 5].destination    = pconfig->tmp_dir;
        config_entry[ 6].destination    = pconfig->view_tmpl_path;
        config_entry[ 7].destination    = pconfig->progress_tmpl_path;
        config_entry[ 8].destination    = pconfig->download_tmpl_path;
        config_entry[ 9].destination    = pconfig->thumb_tmpl_path;
        config_entry[10].destination    = pconfig->error_tmpl_path;
        config_entry[11].destination    = pconfig->max_file_size;
        config_entry[12].destination    = pconfig->total_file_size_limit;
        config_entry[13].destination    = pconfig->total_file_number_limit;
        config_entry[14].destination    = pconfig->per_page_item_number;

        p->pconfig_storage[i] = pconfig;

        if (config_insert_values_global(srv,
                                        ((data_config *)srv->config_context->data[i])->value,
                                        config_entry) != 0) {
            return HANDLER_ERROR;
       }
    }

    return HANDLER_GO_ON;
}

/**
 * ץ饰νλԤޤ
 */
FREE_FUNC(mod_uploader_free)
{
    plugin_data *p = reinterpret_cast<plugin_data *>(p_d);

    plugin_config *pconfig;
    size_t i;

     if (p == NULL) {
        return HANDLER_GO_ON;
    }

    if (p->pconfig_storage) {
        for (i = 0; i < srv->config_context->used; ++i) {
            pconfig = p->pconfig_storage[i];

            if (pconfig == NULL) {
                continue;
            }

            free_config(pconfig);
            free(pconfig);
        }
        free(p->pconfig_storage);
    }

    free_uconfig_map(p->uconfig_map);
    delete p->uconfig_map;

    free(p);

    return HANDLER_GO_ON;
}

URIHANDLER_FUNC(mod_uploader_uri_clean)
{
    plugin_data *p = reinterpret_cast<plugin_data *>(p_d);
    uconfig *config;
    apr_pool_t *pool = NULL;
    handler_t status;

    if (con->uri.path->used == 0) {
        return HANDLER_GO_ON;
    }

    mod_uploader_patch_connection(srv, con, p);

    if (!p->pconfig.is_activated) {
        return HANDLER_GO_ON;
    }

    apr_pool_create(&pool, NULL);

    try {
        if (UNLIKELY(p->uconfig_map->count(p->pconfig.path->ptr) == 0)) {
            config = create_uconfig(&(p->pconfig));
            p->uconfig_map->insert(uploader_config_pair(p->pconfig.path->ptr,
                                                        config));
        } else {
            config = p->uconfig_map->find(p->pconfig.path->ptr)->second;
        }

        status = command_handler(srv, con, p, config, pool,
                                 get_path_info(pool, con->uri.path, p->pconfig.path));

        apr_pool_destroy(pool);

        return status;
    } catch(const char *message) {
        apr_pool_destroy(pool);

        RERROR(message);

        con->http_status = 500;
        con->file_finished = 1;

        return HANDLER_FINISHED;
    }
}

URIHANDLER_FUNC(mod_uploader_physical)
{
    plugin_data *p = reinterpret_cast<plugin_data *>(p_d);
    uconfig *config;
    apr_pool_t *pool = NULL;
    handler_t status;

    if (con->uri.path->used == 0) {
        return HANDLER_GO_ON;
    }

    mod_uploader_patch_connection(srv, con, p);

    if (!p->pconfig.is_activated) {
        return HANDLER_GO_ON;
    }

    apr_pool_create(&pool, NULL);

    try {
        if (UNLIKELY(p->uconfig_map->count(p->pconfig.path->ptr) == 0)) {
            config = create_uconfig(&(p->pconfig));
            p->uconfig_map->insert(uploader_config_pair(p->pconfig.path->ptr,
                                                        config));
        } else {
            config = p->uconfig_map->find(p->pconfig.path->ptr)->second;
        }

        status = sub_command_handler(srv, con, p, config, pool,
                                     get_path_info(pool, con->uri.path, p->pconfig.path));

        apr_pool_destroy(pool);

        return status;
    } catch(const char *message) {
        apr_pool_destroy(pool);

        RERROR(message);

        con->http_status = 500;
        con->file_finished = 1;

        return HANDLER_FINISHED;
    }
}

extern "C" {
int mod_uploader_plugin_init(plugin *p)
{
    p->version                  = LIGHTTPD_VERSION_ID;
    p->name                     = buffer_init_string("uploader");

    p->init                     = mod_uploader_init;
    p->handle_uri_clean         = mod_uploader_uri_clean;
    p->handle_physical          = mod_uploader_physical; // :-)

    p->set_defaults             = mod_uploader_set_defaults;
    p->cleanup                  = mod_uploader_free;

    p->data                     = NULL;

    return 0;
}
}

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
