/******************************************************************************
 * mod_uploader / lighttpd_handler.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: lighttpd_handler.cpp 1003 2006-03-14 13:52:47Z svn $
 *****************************************************************************/

#define __STDC_CONSTANT_MACROS

extern "C" {
#include "http_chunk.h"
#include "response.h"
#include "chunk.h"
#include "log.h"
}
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION

#include "lighttpd_handler.h"
#include "uploader_func.h"

#include "UploadItemCreator.h"
#include "CharCodeConverter.h"
#include "TemplateExecutorImpl.h"
#include "RssViewImpl.h"
#include "Auxiliary.h"
#include "Misc.h"

#include "apr_strings.h"
#include "apr_time.h"
#define APR_WANT_STDIO
#define APR_WANT_STRFUNC
#include "apr_want.h"

#define INFO(command, target)                                           \
    log_error_write(srv, __FILE__, __LINE__, "sss", "Info:", command, target);

#define RERROR(str)                                                     \
    log_error_write(srv, __FILE__, __LINE__, "ss", "Exception:", str);

static const apr_size_t DEFAULT_BUFFER_SIZE = 16*1024;

static const char *read_param(apr_pool_t *pool, chunkqueue *queue)
{
    chunk *chunk;
    char *request;
    char *pos;
    apr_size_t read_size;
    apr_size_t done_size;

    APR_PCALLOC(request, char *, pool, MAX_PARAM_LENGTH+1);

    pos = request;
    done_size = 0;

    for (chunk = queue->first; chunk != NULL; chunk = queue->first) {
        if (done_size == MAX_PARAM_LENGTH) {
            break;
        }

        switch(chunk->type) {
        case chunk::FILE_CHUNK:
            throw "ꥯȤΥ礭ޤ";
        case chunk::MEM_CHUNK:
            read_size = min(chunk->mem->used - chunk->offset - 1,
                            static_cast<unsigned long int>(MAX_PARAM_LENGTH - done_size));

            memcpy(pos, chunk->mem->ptr + chunk->offset, read_size);
            pos += read_size;
            done_size += read_size;

            chunk->offset += read_size;
            queue->bytes_out += read_size;

            break;
        default:
            throw "̤бηϤǤ";
        }
        chunkqueue_remove_finished_chunks(queue);
    }

    return request;
}

static apr_table_t *get_param_table(apr_pool_t *pool, connection *con)
{
    const char *param;
    const char *token;
    apr_table_t *table;
    const char *key;
    const char *value;

    table = apr_table_make(pool, MAX_PARAM_NUMBER);

    if (con->request.content_length == 0 ) {
        return table;
    }

    param = read_param(pool, con->request_content_queue);
    while ((*param != '\0') && (token = get_token(pool, &param, '&'))) {
        key = rfc1738_decode(pool, get_token(pool, &token, '='));
        value = rfc1738_decode(pool, get_token(pool, &token, '='));

        apr_table_set(table, key, value);
    }

    return table;
}

static apr_size_t get_page(connection *con)
{
    return (con->uri.query->used == 0) ? 1 : get_page(con->uri.query->ptr);
}

static apr_uint32_t get_upload_id(connection *con)
{
    return (con->uri.query->used == 0) ? 1 : get_upload_id(con->uri.query->ptr);
}

static const char *get_name_encoding(connection *con)
{
    data_string *data_str;
    const char *user_agent;

    data_str = reinterpret_cast<data_string *>(array_get_element(con->request.headers, "User-Agent"));

    if (data_str != NULL) {
        user_agent = data_str->value->ptr;
    } else {
        user_agent = NULL;
    }

    return get_name_encoding(user_agent);
}

static void set_last_modified(server *srv, connection *con, apr_time_t mtime=apr_time_now())
{
    char time_str[APR_RFC822_DATE_LEN];

    apr_rfc822_date(time_str, mtime);

    response_header_insert(srv, con, CONST_STR_LEN("Last-Modified"), time_str,
                           APR_RFC822_DATE_LEN-1);
}

static handler_t handler_finish(connection *con, apr_size_t status_code=200)
{
    con->http_status = status_code;
    con->file_finished = 1;

    return HANDLER_FINISHED;
}

static handler_t error(server *srv, connection *con, uconfig *config,
                 apr_pool_t *pool, buffer *res, const char *error)
{
    page_template *tmpl = config->get_error_template(pool);

    auto_ptr<variable_map> vmap(get_error_vmap(pool, config, tmpl, error));

    response_header_insert(srv, con, CONST_STR_LEN("Content-Type"),
                           CONST_STR_LEN("text/html"));

    lighttpdResponseWriter writer(res);
    lighttpdTemplateExecutor texecutor(pool, tmpl->imap, vmap.get(), writer);
    texecutor.exec(tmpl->node);

    return handler_finish(con);
}

static handler_t redirect_top(server *srv, connection *con, buffer *res)
{
    // POST ˥ڡñ˥ɤǤ褦ˤ뤿ᡤHTML 
    // 쥯Ȥ롥

    response_header_insert(srv, con, CONST_STR_LEN("Content-Type"),
                           CONST_STR_LEN("text/html"));

    BUFFER_APPEND_STRING_CONST(res, "<?xml version=\"1.0\" encoding=\"EUC-JP\"?>");
    BUFFER_APPEND_STRING_CONST(res, "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">");
    BUFFER_APPEND_STRING_CONST(res, "<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"ja\" >");
    BUFFER_APPEND_STRING_CONST(res, " <head>");
    BUFFER_APPEND_STRING_CONST(res, "  <meta http-equiv=\"content-type\" content=\"application/xhtml+xml\" />");
    BUFFER_APPEND_STRING_CONST(res, "  <meta http-equiv=\"refresh\" content=\"0;url=../\" />");
    BUFFER_APPEND_STRING_CONST(res, "  <title>mod_uploader</title>");
    BUFFER_APPEND_STRING_CONST(res, " </head>");
    BUFFER_APPEND_STRING_CONST(res, " <body />");
    BUFFER_APPEND_STRING_CONST(res, "</html>");

    return handler_finish(con);
}

static handler_t input_pass(server *srv, connection *con, uconfig *config,
                            apr_pool_t *pool, const char *arg, buffer *res,
                            UploadItem::header *header)
{
    try {
        page_template *tmpl = config->get_download_template(pool);

        auto_ptr<variable_map> vmap(get_download_vmap(pool, config, tmpl, header));

        response_header_insert(srv, con, CONST_STR_LEN("Content-Type"),
                               CONST_STR_LEN("text/html"));

        lighttpdResponseWriter writer(res);
        lighttpdTemplateExecutor texecutor(pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(tmpl->node);
    } catch(const char *message) {
        BUFFER_APPEND_STRING_CONST(res, "<div class=\"warning\">");
        buffer_append_string(res, message);
        BUFFER_APPEND_STRING_CONST(res, "</div>");
    }

    return handler_finish(con);
}

static handler_t view(server *srv, connection *con, uconfig *config,
                      apr_pool_t *pool, const char *arg, buffer *res)
{
    if (!is_modified_since(pool, con->request.http_if_modified_since,
                           config->item_list->get_mtime())) {
        return handler_finish(con, 304);
    }

    set_last_modified(srv, con, config->item_list->get_mtime());

    response_header_insert(srv, con, CONST_STR_LEN("Content-Type"),
                           CONST_STR_LEN("text/html"));

    if (con->request.http_method == HTTP_METHOD_HEAD) {
        return handler_finish(con);
    }

    try {
        page_template *tmpl = config->get_view_template(pool);

        auto_ptr<variable_map> vmap(get_view_vmap(pool, config, tmpl, get_page(con)));

        lighttpdResponseWriter writer(res);
        lighttpdTemplateExecutor texecutor(pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(tmpl->node);
    } catch(const char *message) {
        BUFFER_APPEND_STRING_CONST(res, "<div class=\"warning\">");
        buffer_append_string(res, message);
        BUFFER_APPEND_STRING_CONST(res, "</div>");
    }

    return handler_finish(con);
}

#ifdef MAKE_THUMBNAIL
static handler_t thumbnail(server *srv, connection *con, uconfig *config,
                           apr_pool_t *pool, const char *arg, buffer *res)
{
    if (!is_modified_since(pool, con->request.http_if_modified_since,
                           config->item_list->get_mtime())) {
        return handler_finish(con, 304);
    }

    set_last_modified(srv, con, config->item_list->get_mtime());

    response_header_insert(srv, con, CONST_STR_LEN("Content-Type"),
                           CONST_STR_LEN("text/html"));

    if (con->request.http_method == HTTP_METHOD_HEAD) {
        return handler_finish(con);
    }

    try {
        page_template *tmpl = config->get_thumb_template(pool);

        auto_ptr<variable_map> vmap(get_thumb_vmap(pool, config, tmpl, get_page(con)));

        lighttpdResponseWriter writer(res);
        lighttpdTemplateExecutor texecutor(pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(tmpl->node);
    } catch(const char *message) {
        BUFFER_APPEND_STRING_CONST(res, "<div class=\"warning\">");
        buffer_append_string(res, message);
        BUFFER_APPEND_STRING_CONST(res, "</div>");
    }

    return handler_finish(con);
}
#endif

static handler_t rss(server *srv, connection *con, uconfig *config,
                     apr_pool_t *pool, const char *arg, buffer *res)
{
    if (!is_modified_since(pool, con->request.http_if_modified_since,
                           config->item_list->get_mtime())) {
        return handler_finish(con, 304);
    }

    set_last_modified(srv, con, config->item_list->get_mtime());

    response_header_insert(srv, con, CONST_STR_LEN("Content-Type"),
                           lighttpdRssView::content_type,
                           strlen(lighttpdRssView::content_type));

    if (con->request.http_method == HTTP_METHOD_HEAD) {
        return handler_finish(con);
    }

    try {
        lighttpdResponseWriter writer(res);
        lighttpdRssView view(pool, writer);
        view.exec(config->url, config->per_page_item_number, config->item_list);
    } catch(const char *message) {
        BUFFER_APPEND_STRING_CONST(res, "<div class=\"warning\">");
        buffer_append_string(res, message);
        BUFFER_APPEND_STRING_CONST(res, "</div>");
    }

    return handler_finish(con);
}

static handler_t upload_init(server *srv, connection *con, uconfig *config,
                             apr_pool_t *pool, const char *arg, plugin_data *p)
{
    const char *address;
    apr_uint32_t upload_id;

    if (con->request.http_method != HTTP_METHOD_POST) {
        return handler_finish(con, 400);
    }

    if (con->request.content_length == 0) {
        return handler_finish(con, 204);
    }

    address = con->dst_addr_buf->ptr;
    if (is_poster_list_contain(p->poster_list, address)) {
        return error(srv, con, config, pool,
                     chunkqueue_get_append_buffer(con->write_queue),
                     apr_psprintf(pool, "%" APR_SIZE_T_FMT  " ô֤ϺƤǤޤ",
                                  POST_INTERVAL_SEC));
    }

    upload_id = get_upload_id(con);
    add_progress_list(p->progress_list, upload_id, con);

    return HANDLER_GO_ON;
}

static handler_t upload(server *srv, connection *con, uconfig *config,
                        apr_pool_t *pool, const char *arg, plugin_data *p,
                        buffer *res)
{
    const char *address;
    apr_uint32_t upload_id;
    pprogress *progress;

    address = con->dst_addr_buf->ptr;
    upload_id = get_upload_id(con);

    progress = get_progress(p->progress_list, upload_id);
    progress->total_size = con->request.content_length;
    progress->read_size  = con->request_content_queue->bytes_in;

    try {
        UploadItem::header *header;
        const char *tmp_path;
        const char *file_name;

        lighttpdPostReader reader(progress, con->request_content_queue);
        auto_ptr<RFC1867Data::query_map>
            qmap(config->uparser->parse(pool, reader,
                                        con->request.http_content_type,
                                        con->request.content_length));

        try {
            header = UploadItemCreator::create_header(pool, qmap.get(), &tmp_path);
        } catch(const char *) {
            lighttpdRFC1867Parser::clean_tmp_file(pool, qmap.get());
            throw;
        }
        auto_ptr<UploadItemWriter> writer(config->get_uwriter(pool));
        file_name = writer->write(header, tmp_path);

        config->item_list->add(file_name, true);

        add_poster_list(p->poster_list, address);
        progress->end_time = apr_time_now();

        INFO("upload", file_name);

        return redirect_top(srv, con, res);
    } catch(const char *message) {
        return error(srv, con, config, pool, res, message);
    }
}

static handler_t progress(server *srv, connection *con, uconfig *config,
                          apr_pool_t *pool, const char *arg, buffer *res)
{
    try {
        page_template *tmpl = config->get_progress_template(pool);

        auto_ptr<variable_map> vmap(get_progress_vmap(pool, config, tmpl));

        lighttpdResponseWriter writer(res);
        lighttpdTemplateExecutor texecutor(pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(tmpl->node);
    } catch(const char *message) {
        BUFFER_APPEND_STRING_CONST(res, "<div class=\"warning\">");
        buffer_append_string(res, message);
        BUFFER_APPEND_STRING_CONST(res, "</div>");
    }

    return handler_finish(con);
}

static handler_t progress_data(server *srv, connection *con, uconfig *config,
                               apr_pool_t *pool, const char *arg,
                               plugin_data *p, buffer *res)
{
    apr_uint32_t upload_id;
    pprogress *progress;
    const char *progress_str;
    connection *post_con;

    upload_id = get_upload_id(con);

    // upload_id  0 λϥ顼
    if (UNLIKELY(upload_id == 0)) {
        BUFFER_APPEND_STRING_CONST(res, "e");
        return handler_finish(con);
    }

    progress = get_progress(p->progress_list, upload_id);

    if (UNLIKELY(progress == p->progress_list)) {
        BUFFER_APPEND_STRING_CONST(res, "u");
        return handler_finish(con);
    }

    post_con = reinterpret_cast<connection *>(progress->data);

    if (progress->total_size != 0) {
        progress_str = apr_psprintf(pool, "s %" APR_UINT64_T_FMT " %" APR_UINT64_T_FMT,
                                    progress->total_size, progress->read_size);
    } else {
        progress_str = apr_psprintf(pool, "s %" APR_UINT64_T_FMT " %" APR_UINT64_T_FMT,
                                    static_cast<apr_uint64_t>(post_con->request.content_length),
                                    static_cast<apr_uint64_t>(post_con->request_content_queue->bytes_in));
    }

    buffer_append_string(res, progress_str);

    return handler_finish(con);
}

static handler_t download(server *srv, connection *con, uconfig *config,
                          apr_pool_t *pool, const char *arg, buffer *res)
{
    const char *name;

    name = get_token(pool, &arg, '/');
    if (name == NULL) {
        return handler_finish(con, 400);
    }

    {
        // ե̾Υå
        const char *pos = name;

        while (isalnum(*pos) || (*pos == '.')) {
            pos++;
        }

        if ((static_cast<size_t>(pos-name) != strlen(name)) ||
            (strlen(name) > APR_PATH_MAX)) {
            return handler_finish(con, 400);
        }
    }

    try {
        apr_file_t *file;
        UploadItem::header *header;
        const char *file_name;
        buffer *file_path;
        const char *password;
        const char *open_mode;
        const char *content_dispos;

        auto_ptr<UploadItemReader> reader(config->get_ureader(pool));
        header = reader->read(name, &file);

        password = apr_table_get(get_param_table(pool, con), UploadItem::DOWNLOAD_PASS_PARAM);

        // DL pass Υå
        if ((strlen(header->download_pass) > 0) &&
            ((password == NULL) ||
             (strncmp(header->download_pass, password, strlen(header->download_pass)) != 0))) {
            return input_pass(srv, con, config, pool, arg, res, header);
        }

        if (!is_modified_since(pool, con->request.http_if_modified_since,
                               header->time)) {
            return handler_finish(con, 304);
        }

        set_last_modified(srv, con, header->time);

        response_header_insert(srv, con, CONST_STR_LEN("Content-Type"),
                               header->file_mime, strlen(header->file_mime));

        if (con->request.http_method == HTTP_METHOD_HEAD) {
            return handler_finish(con);
        }

        file_name = CharCodeConverter::convert(pool, header->file_name,
                                               CharCodeConverter::DEFAULT_CODE,
                                               get_name_encoding(con));

        file_path = buffer_init_string(config->file_dir);
        buffer_append_string(file_path, "/");
        buffer_append_string(file_path, name);

        // ɽ⡼ or ɥ⡼
        open_mode = (con->uri.query->used == 0) ? "inline" : "attachment";

        // Content-Disposition
        content_dispos = apr_psprintf(pool, "%s; filename=\"%s\"", open_mode, file_name);
        response_header_insert(srv, con, CONST_STR_LEN("Content-Disposition"),
                               content_dispos, strlen(content_dispos));

        http_chunk_append_file(srv, con, file_path, sizeof(UploadItem::header),
                               header->file_size);

        buffer_free(file_path);

        return handler_finish(con);
    } catch(const char *message) {
        return error(srv, con, config, pool, res, message);
    }
}

static handler_t remove(server *srv, connection *con, uconfig *config,
                        apr_pool_t *pool, const char *arg, buffer *res)
{
    apr_table_t *ctable;

    if (con->request.http_method != HTTP_METHOD_POST) {
        return handler_finish(con, 400);
    }

    try {
        const char *file_name;
        const char *password;

        ctable = get_param_table(pool, con);

        file_name = apr_table_get(ctable, UploadItem::FILE_PARAM);
        password = apr_table_get(ctable, UploadItem::REMOVE_PASS_PARAM);

        if ((file_name == NULL) || (password == NULL)) {
            return error(srv, con, config, pool, res,
                         "եޤϥѥɤꤵƤޤ");
        }

        config->item_list->remove(file_name, password);
        config->remove_item_number++;

        INFO("remove", file_name);

        return redirect_top(srv, con, res);
    } catch(const char *message) {
        return error(srv, con, config, pool, res, message);
    }
}

handler_t command_handler(server *srv, connection *con, plugin_data *p,
                          uconfig *config, apr_pool_t *pool, const char *arg)
{
    const char *command;

    command = get_token(pool, &arg, '/');

    if (LIKELY(*command == '\0')) {
        return view(srv, con, config, pool, arg,
                    chunkqueue_get_append_buffer(con->write_queue));
    } else if (strcmp(command, RSS_COMMAND) == 0) {
        return rss(srv, con, config, pool, arg,
                   chunkqueue_get_append_buffer(con->write_queue));
    } else if (strcmp(command, UPLOAD_COMMAND) == 0) {
        return upload_init(srv, con, config, pool, arg, p);
    } else if (strcmp(command, PROGRESS_COMMAND) == 0) {
        return progress(srv, con, config, pool, arg,
                   chunkqueue_get_append_buffer(con->write_queue));
    } else if (strcmp(command, PROGRESS_DATA_COMMAND) == 0) {
        return progress_data(srv, con, config, pool, arg, p,
                             chunkqueue_get_append_buffer(con->write_queue));
    } else if (strcmp(command, DOWNLOAD_COMMAND) == 0) {
        return download(srv, con, config, pool, arg,
                        chunkqueue_get_append_buffer(con->write_queue));
    } else if (strcmp(command, REMOVE_COMMAND) == 0) {
        return remove(srv, con, config, pool, arg,
                      chunkqueue_get_append_buffer(con->write_queue));
#ifdef MAKE_THUMBNAIL
    } else if (strcmp(command, THUMBNAIL_COMMAND) == 0) {
        return thumbnail(srv, con, config, pool, arg,
                         chunkqueue_get_append_buffer(con->write_queue));
#endif
    } else {
        return error(srv, con, config, pool,
                     chunkqueue_get_append_buffer(con->write_queue),
                     "ݡȤƤʤޥɤǤ");
    }
}

handler_t sub_command_handler(server *srv, connection *con, plugin_data *p,
                              uconfig *config, apr_pool_t *pool,
                              const char *arg)
{
    const char *command;

    command = get_token(pool, &arg, '/');

    if (strncmp(command, UPLOAD_COMMAND, strlen(UPLOAD_COMMAND)) == 0) {
        return upload(srv, con, config, pool, arg, p,
                      chunkqueue_get_append_buffer(con->write_queue));
    } else {
        return error(srv, con, config, pool,
                     chunkqueue_get_append_buffer(con->write_queue),
                     "餯ХǤ");
    }
}

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
