/******************************************************************************
 * mod_uploader / UnixThumbnailWriter.cpp
 ******************************************************************************
 * Copyright (C) 2004 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not bcktuired.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: UnixThumbnailWriter.cpp 1081 2006-05-27 15:44:24Z svn $
 *****************************************************************************/

#include "UnixThumbnailWriter.h"

#ifdef MAKE_THUMBNAIL
#include "UploadItem.h"
#include "Auxiliary.h"
#include "Misc.h"

#include "apr_file_io.h"
#include "apr_strings.h"

#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#include <Magick++.h>
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION

#ifdef MOVIE_THUMBNAIL
#include <ffmpeg/avio.h>
#include <ffmpeg/avcodec.h>
#endif

#ifdef SEGV_SAFE
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#endif

#ifdef DEBUG
#include <iostream>
#endif

#ifdef MOVIE_THUMBNAIL
const PixelFormat UnixThumbnailWriter::FRAME_PIXEL_FORMAT   = PIX_FMT_RGB24;
#endif


/******************************************************************************
 * public ᥽å
 *****************************************************************************/
UnixThumbnailWriter::UnixThumbnailWriter(apr_pool_t *pool, const char *file_dir,
                                 const char *thumb_dir)
    : ThumbnailWriter(pool, file_dir, thumb_dir)
{

}

bool UnixThumbnailWriter::write(const char *file_name)
{
    apr_pool_t *pool;
    bool is_created = false;

    if (apr_pool_create(&pool, pool_) != APR_SUCCESS) {
        throw "γݤ˼Ԥޤ";
    }

    try {
#ifdef MOVIE_THUMBNAIL
#ifdef SEGV_SAFE
        is_created = create_movie_thumb_safe(pool, file_name);
#else
        is_created = create_movie_thumb(pool, file_name);
#endif
#endif

        if (!is_created) {
            is_created = create_image_thumb(pool, file_name);
        }

        apr_pool_destroy(pool);

        return is_created;
    } catch(const char *) {
        apr_pool_destroy(pool);
        throw;
    }
}


/******************************************************************************
 * private ᥽å
 *****************************************************************************/
#ifdef MOVIE_THUMBNAIL
#ifdef SEGV_SAFE
bool UnixThumbnailWriter::create_movie_thumb_safe(apr_pool_t *pool, const char *file_name)
{
    static const int SUCCEED        = 0;
    static const int FAIL           = -1;

    static const int NICE_VALUE     = 20;
    static const int TIMEOUT_SEC    = 120;

    pid_t pid;
    int status;

    switch (pid = fork()) {
    case 0: // ҥץ
        apr_pool_create(&pool, NULL);

        // ffmpeg ˽Ȥк
        nice(NICE_VALUE);
        alarm(TIMEOUT_SEC);

        exit(create_movie_thumb(pool, file_name) ? SUCCEED : FAIL);
    case -1: // 顼
        return false;
    default: // ƥץ
        waitpid(pid, &status, 0);

        return WIFEXITED(status) && (WEXITSTATUS(status) == SUCCEED);
    }
}
#endif

bool UnixThumbnailWriter::create_movie_thumb(apr_pool_t *pool, const char *file_name)
{
    list<Magick::Image> frame_list;
    AVFormatContext *format_context;
    AVCodecContext *codec_context;
    AVCodec *codec;
    AVFrame *frame;
    AVPicture *picture;
    int video_index;
    apr_size_t buffer_size;

    if (!open_movie_context(pool, file_name, &format_context, &codec_context,
                            &codec, &video_index)) {
        return false;
    }

    frame = avcodec_alloc_frame();
    picture = reinterpret_cast<AVPicture *>(avcodec_alloc_frame());
    if ((frame == NULL) || (picture == NULL)) {
        return false;
    }

    buffer_size = avpicture_get_size(FRAME_PIXEL_FORMAT,
                                     codec_context->width, codec_context->height);

    apr_byte_t buffer[buffer_size];

    avpicture_fill(picture, buffer, FRAME_PIXEL_FORMAT,
                   codec_context->width, codec_context->height);

    try {
        read_movie_frames(format_context, codec_context,frame, picture, video_index,
                          ThumbnailWriter::FRAME_SAMPLE_SEC,
                          ThumbnailWriter::FRAME_DELAY_SEC,
                          ThumbnailWriter::FRAME_NUMBER,
                          frame_list);
    } catch(const char *) {
        av_free(picture);
        av_free(frame);
        close_movie_context(format_context, codec_context);

        return false;
    }

    if (frame_list.size() == 0) {
        av_free(picture);
        av_free(frame);
        close_movie_context(format_context, codec_context);

        return false;
    }

    writeImages(frame_list.begin(), frame_list.end(),
                create_thumb_path(pool, thumb_dir_, file_name));

    av_free(picture);
    av_free(frame);
    close_movie_context(format_context, codec_context);

    return true;
}

bool UnixThumbnailWriter::open_movie_file(apr_pool_t *pool, const char *file_name,
                                      AVFormatContext **format_context)
{
    char *file_path;
    AVInputFormat *format;
    AVProbeData probe_data;
    ByteIOContext io_context;
    apr_byte_t buffer[BUFFER_SIZE];

    if (apr_filepath_merge(&file_path, file_dir_, file_name,
                           APR_FILEPATH_NOTABOVEROOT, pool) != APR_SUCCESS) {
        throw "åץɥե̾ǤޤǤ";
    }

    if (url_fopen(&io_context, file_path, URL_RDONLY) < 0) {
        return false;
    }

    io_context.seek(io_context.opaque, UploadItem::ITEM_HEADER_SIZE, SEEK_SET);

    probe_data.filename = file_path;
    probe_data.buf = buffer;
    probe_data.buf_size = 0;
    probe_data.buf_size = get_buffer(&io_context, buffer, sizeof(buffer));

    url_fseek(&io_context, 0, SEEK_SET);

    format = av_probe_input_format(&probe_data, 1);

    if (!format) {
        url_fclose(&io_context);
        return false;
    }

    if (av_open_input_stream(format_context, &io_context, file_path, format, NULL)) {
        url_fclose(&io_context);
        return false;
    }

    return true;
}

bool UnixThumbnailWriter::open_movie_context(apr_pool_t *pool, const char *file_path,
                                         AVFormatContext **format_context,
                                         AVCodecContext **codec_context,
                                         AVCodec **codec, int *video_index)
{
    av_register_all();

    if (!open_movie_file(pool, file_path, format_context)) {
        return false;
    }

    if (av_find_stream_info(*format_context) < 0) {
        return false;
    }

    *video_index = -1;
    for (int i = 0; i < (*format_context)->nb_streams; ++i) {
        if ((*format_context)->streams[i]->codec->codec_type == CODEC_TYPE_VIDEO) {
            *video_index = i;
            break;
        }
    }

    if (*video_index == -1) {
        return false;
    }

    *codec_context = (*format_context)->streams[*video_index]->codec;

    *codec = avcodec_find_decoder((*codec_context)->codec_id);
    if (*codec == NULL) {
        return false;
    }

    if (avcodec_open(*codec_context, *codec) < 0) {
        return false;
    }

    return true;
}

void UnixThumbnailWriter::close_movie_context(AVFormatContext *format_context,
                                          AVCodecContext *codec_context)
{
    avcodec_close(codec_context);
    av_close_input_file(format_context);
}

void UnixThumbnailWriter::read_movie_frames(AVFormatContext *format_context,
                                        AVCodecContext *codec_context,
                                        AVFrame *frame, AVPicture *picture, int video_index,
                                        double frame_sample, double frame_delay,
                                        apr_size_t frame_number,
                                        list<Magick::Image>& frame_list)
{
    static const apr_size_t MAX_INCOMPLETE_COUNT   = 1024 * 1024 / BUFFER_SIZE;

    AVPacket packet;
    apr_size_t movie_fps;
    int is_frame_finish;
    apr_size_t decode_bytes;
    apr_size_t incomplete_count;
    apr_size_t i;
    apr_size_t j;
    Magick::Geometry thumb_size(codec_context->width, codec_context->height);

    adjust_size(thumb_size);
    movie_fps = codec_context->time_base.den / codec_context->time_base.num / 1000;
    if (movie_fps == 0) {
        movie_fps = 1; // 0 Ȥޤ.
    }

    // TODO: av_seek_frame 夤ƤʲΥ롼פ롥

    incomplete_count = 0;
    i = j = 0;
    while (av_read_frame(format_context, &packet) >= 0) {
        if (packet.stream_index != video_index) {
            av_free_packet(&packet);
            continue;
        }

        if ((decode_bytes = avcodec_decode_video(codec_context, frame, &is_frame_finish,
                                                 packet.data, packet.size)) < 0) {
            av_free_packet(&packet);

            throw "ưΥǥɤ˼Ԥޤ(1)";
        }

        if ((is_frame_finish == 0) ||
            ((++i % static_cast<apr_size_t>(movie_fps*frame_sample)) != 0)) {
            av_free_packet(&packet);

            if (decode_bytes != 0) {
                continue;
            }

            if (++incomplete_count != MAX_INCOMPLETE_COUNT) {
                continue;
            }

            // Ԥ³ϡǥɤڤ夲롥
            if (frame_list.size() == 0) {
                throw "ưΥǥɤ˼Ԥޤ(2)";
            } else {
                return;
            }
        }
        incomplete_count = 0;

        if (img_convert(picture, FRAME_PIXEL_FORMAT,
                        reinterpret_cast<AVPicture *>(frame),
                        codec_context->pix_fmt, codec_context->width, codec_context->height) < 0) {
            av_free_packet(&packet);

            throw "ե졼ΥեޥåѴ˼Ԥޤ";
        }

        Magick::Image frame_image(codec_context->width, codec_context->height,
                                  "RGB", Magick::CharPixel, picture->data[0]);

        frame_image.scale(thumb_size);
        frame_image.animationDelay(static_cast<apr_size_t>(frame_delay*100));
        frame_image.animationIterations(0);

        frame_list.push_back(frame_image);

        if (++j == frame_number) {
            return;
        }

        av_free_packet(&packet);
    }
}
#endif


/******************************************************************************
 * ƥ
 *****************************************************************************/
#ifdef DEBUG_UnixThumbnailWriter

#include "apr_general.h"
#include "apr_file_io.h"

static const char THUMB_DIR[]            = "/tmp";

void usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <FILE> <THUMBDIR>" << endl;
}

int main(int argc, const char * const *argv)
{
    apr_pool_t *pool;
    const char *thumb_dir;

    apr_app_initialize(&argc, &argv, NULL);
    apr_pool_create(&pool, NULL);

    try {
        if (argc < 2) {
            throw "ѤΥե뤬ꤵƤޤ";
        }

        if (argc == 2) {
            thumb_dir = THUMB_DIR;
        } else {
            thumb_dir = argv[2];
        }

        UnixThumbnailWriter writer(pool, dirname_ex(pool, argv[1]), thumb_dir);
        writer.write(basename_ex(argv[1]));
    } catch(const char *message) {
        cerr << "Error: " << message << endl;
        usage(argv[0]);

        return EXIT_FAILURE;
    }

    apr_terminate();

    return EXIT_SUCCESS;
}
#endif

#endif

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
