/******************************************************************************
 * mod_uploader / UploaderConfig.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: UploaderConfig.cpp 1241 2006-07-30 15:42:10Z svn $
 *****************************************************************************/

#define __STDC_CONSTANT_MACROS

#include "UploaderConfig.h"
#include "Misc.h"

#include "apr_mmap.h"
#include "apr_strings.h"

#define DESTROY(pool) if (pool != NULL) { apr_pool_destroy(pool); pool = NULL;}

#ifdef DEBUG
#include <iostream>
#endif

const apr_uint64_t UploaderConfig::TOTAL_FILE_SIZE_LIMIT= 1*1024*1024*1024;
const apr_size_t UploaderConfig::TOTAL_FILE_NUMBER_LIMIT= 1000;
const apr_uint64_t UploaderConfig::MAX_FORM_FILE_SIZE   = 100*1024*1024;
const apr_size_t UploaderConfig::MAX_FORM_TEXT_SIZE     = 1024;
const apr_size_t UploaderConfig::MAX_FORM_ITEM_NUMBER   = 5;
const apr_size_t UploaderConfig::PER_PAGE_ITEM_NUMBER   = 50;
const apr_size_t UploaderConfig::ITEM_RELOAD_THRESHOLD  = 10;


/******************************************************************************
 * public ᥽å
 *****************************************************************************/
UploaderConfig::UploaderConfig()
    : url(""),
      file_dir(NULL),
      thumb_dir(NULL),
      tmp_dir(NULL),
      view_tmpl_path(NULL),
      progress_tmpl_path(NULL),
      download_tmpl_path(NULL),
      thumb_tmpl_path(NULL),
      error_tmpl_path(NULL),
      total_file_size_limit(TOTAL_FILE_SIZE_LIMIT),
      total_file_number_limit(TOTAL_FILE_NUMBER_LIMIT),
      max_file_size(MAX_FORM_FILE_SIZE),
      per_page_item_number(PER_PAGE_ITEM_NUMBER),
      remove_item_number(0),
      item_list(NULL),
      view_tmpl_(NULL),
      view_tmpl_pool_(NULL),
      progress_tmpl_(NULL),
      progress_tmpl_pool_(NULL),
      download_tmpl_(NULL),
      download_tmpl_pool_(NULL),
      thumb_tmpl_(NULL),
      thumb_tmpl_pool_(NULL),
      error_tmpl_(NULL),
      error_tmpl_pool_(NULL),
      pool_(NULL),
      list_pool_(NULL),
      is_init_(false)
{

}

UploaderConfig::~UploaderConfig()
{
    finalize();
}

void UploaderConfig::finalize()
{
    delete item_list;

    delete view_tmpl_;
    delete progress_tmpl_;
    delete download_tmpl_;
#ifdef MAKE_THUMBNAIL
    delete thumb_tmpl_;
#endif
    delete error_tmpl_;

    DESTROY(list_pool_);
    DESTROY(view_tmpl_pool_);
    DESTROY(progress_tmpl_pool_);
    DESTROY(download_tmpl_pool_);
#ifdef MAKE_THUMBNAIL
    DESTROY(thumb_tmpl_pool_);
#endif
    DESTROY(error_tmpl_pool_);
}

page_template *UploaderConfig::get_view_template(apr_pool_t *pool)
{
    if (UNLIKELY(get_mtime(pool, view_tmpl_path) > view_tmpl_->mtime)) {
        load_template(pool, &view_tmpl_pool_, view_tmpl_path, view_tmpl_);
    }

    return view_tmpl_;
}

page_template *UploaderConfig::get_progress_template(apr_pool_t *pool)
{
    if (UNLIKELY(get_mtime(pool, progress_tmpl_path) > progress_tmpl_->mtime)) {
        load_template(pool, &progress_tmpl_pool_, progress_tmpl_path, progress_tmpl_);
    }

    return progress_tmpl_;
}

page_template *UploaderConfig::get_download_template(apr_pool_t *pool)
{
    if (UNLIKELY(get_mtime(pool, download_tmpl_path) > download_tmpl_->mtime)) {
        load_template(pool, &download_tmpl_pool_, download_tmpl_path, download_tmpl_);
    }

    return download_tmpl_;
}

page_template *UploaderConfig::get_thumb_template(apr_pool_t *pool)
{
    if (UNLIKELY(get_mtime(pool, thumb_tmpl_path) > thumb_tmpl_->mtime)) {
        load_template(pool, &thumb_tmpl_pool_, thumb_tmpl_path, thumb_tmpl_);
    }

    return thumb_tmpl_;
}

page_template *UploaderConfig::get_error_template(apr_pool_t *pool)
{
    if (UNLIKELY(get_mtime(pool, error_tmpl_path) > error_tmpl_->mtime)) {
        load_template(pool, &error_tmpl_pool_, error_tmpl_path, error_tmpl_);
    }

    return error_tmpl_;
}

UploadItemReader *UploaderConfig::get_ureader(apr_pool_t *pool)
{
    return new UploadItemReader(pool, file_dir);
}

UploadItemWriter *UploaderConfig::get_uwriter(apr_pool_t *pool)
{
    return new UploadItemWriter(pool, file_dir);
}


/******************************************************************************
 * protected ᥽å
 *****************************************************************************/
void UploaderConfig::validate()
{
    if (file_dir            == NULL) throw "FileDirectory is not specified.";
#ifdef MAKE_THUMBNAIL
    if (thumb_dir           == NULL) throw "ThumbDirectory is not specified.";
#endif
    if (tmp_dir             == NULL) throw "TmpDirectory is not specified.";
    if (view_tmpl_path      == NULL) throw "ViewTemplateFile is not specified.";
    if (progress_tmpl_path  == NULL) throw "ProgressTemplateFile is not specified.";
    if (download_tmpl_path  == NULL) throw "DownloadTemplateFile is not specified.";
    if (thumb_tmpl_path     == NULL) throw "ThumbTemplateFile is not specified.";
    if (error_tmpl_path     == NULL) throw "ErrorTemplateFile is not specified.";
}

void UploaderConfig::init_template()
{
    view_tmpl_      = new page_template();
    progress_tmpl_  = new page_template();
    download_tmpl_  = new page_template();
#ifdef MAKE_THUMBNAIL
    thumb_tmpl_     = new page_template();
#endif
    error_tmpl_     = new page_template();
}

void UploaderConfig::load_list(apr_size_t revision)
{
    delete item_list;
    DESTROY(list_pool_);

    if (apr_pool_create(&list_pool_, pool_) != APR_SUCCESS) {
        throw "åץɥƥΥꥹѤΥݤ˼Ԥޤ";
    }

    item_list = UploadItemList::load(list_pool_, file_dir, thumb_dir,
                                     total_file_number_limit, total_file_size_limit);
    item_list->set_revision(revision);
    remove_item_number = 0;
}

void UploaderConfig::load_template(apr_pool_t *pool, apr_pool_t **tmpl_pool,
                                   const char *template_path, page_template *tmpl)
{
    apr_file_t *file;
    apr_mmap_t *file_map = NULL;
    apr_finfo_t info;
    TemplateLexer::ident_map *old_imap = NULL;
    apr_pool_t *old_tmpl_pool = NULL;

    if (apr_file_open(&file, template_path, APR_READ,
                      APR_OS_DEFAULT, pool) != APR_SUCCESS) {
        throw "Can not open page template.";
    }

    try {
        if (apr_file_info_get(&info, APR_FINFO_SIZE|APR_FINFO_MTIME, file) != APR_SUCCESS) {
            throw "ƥץ졼ȤΥǤޤǤ";
        }

        if (apr_mmap_create(&file_map, file, 0, static_cast<apr_size_t>(info.size), APR_MMAP_READ,
                            pool) != APR_SUCCESS) {
            throw "ƥץ졼Ȥ mmap ˼Ԥޤ";
        }

        old_imap = tmpl->imap;
        old_tmpl_pool = *tmpl_pool;

        if (apr_pool_create(tmpl_pool, pool_) != APR_SUCCESS) {
            throw "ƥץ졼ѤΥݤ˼Ԥޤ";
        }

        TemplateLexer lexer(*tmpl_pool, static_cast<const char *>(file_map->mm),
                            static_cast<apr_size_t>(info.size));
        TemplateParser parser(*tmpl_pool);

        tmpl->imap = new TemplateLexer::ident_map(*(lexer.get_ident_map()));
        tmpl->node = parser.parse(lexer.get_token_list(), lexer.get_ident_map());
        tmpl->mtime = info.mtime;

        apr_mmap_delete(file_map);
        apr_file_close(file);

        delete old_imap;
        DESTROY(old_tmpl_pool);
    } catch(const char *) {
        if (file_map != NULL) {
            apr_mmap_delete(file_map);
        }
        apr_file_close(file);

        if (old_imap != NULL) {
            delete old_imap;
        }
        DESTROY(old_tmpl_pool);

        throw;
    }
}

apr_time_t UploaderConfig::get_mtime(apr_pool_t *pool, const char *path)
{
    apr_finfo_t info;

    if (UNLIKELY(apr_stat(&info, path, APR_FINFO_MTIME, pool) != APR_SUCCESS)) {
        throw "Can not get information of file/directory.";
    }

    return info.mtime;
}


// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
