/******************************************************************************
 * mod_uploader / apache_handler.cpp
 ******************************************************************************
 * Copyright (C) 2005 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: apache_handler.cpp 1247 2006-08-05 05:45:40Z svn $
 *****************************************************************************/

#include "apache_handler.h"
#include "uploader_func.h"

#include "UploadItemCreator.h"
#include "ApacheResponseWriter.h"
#include "ApacheUploaderConfig.h"
#include "RssViewImpl.h"
#include "CharCodeConverter.h"
#include "TemplateExecutorImpl.h"

#ifdef MAKE_THUMBNAIL
#include "ThumbnailWriter.h"
#endif
#include "Auxiliary.h"
#include "Misc.h"

#define CORE_PRIVATE
#include "http_core.h"
#include "http_log.h"

#define APR_WANT_STRFUNC
#include "apr_want.h"

#include <stdlib.h>
#include <ctype.h>

#ifdef DEBUG
#include <iostream>
#endif

#define INFO(command, target)                   \
    ap_log_rerror(APLOG_MARK, APLOG_INFO, 0, r, \
                  "[" PACKAGE_NAME "] (%s) %s", command, target)


static apr_table_t *get_param_table(request_rec *r)
{
    const char *param;
    const char *token;
    apr_table_t *table;
    char *key;
    char *value;

    table = apr_table_make(r->pool, MAX_PARAM_NUMBER);

    APR_PCALLOC(param, char *, r->pool, MAX_PARAM_LENGTH+1);

    if ((ap_setup_client_block(r, REQUEST_CHUNKED_ERROR) != OK) ||
        !ap_should_client_block(r)) {
        return table;
    }

    ap_get_client_block(r, const_cast<char *>(param), MAX_PARAM_LENGTH);
    while ((*param != '\0') && (token = ap_getword(r->pool, &param, '&'))) {
        key = ap_getword(r->pool, &token, '=');
        value = ap_getword(r->pool, &token, '=');
        ap_unescape_url(key);
        ap_unescape_url(value);

        apr_table_set(table, key, value);
    }

    return table;
}

static apr_size_t get_page(request_rec *r)
{
    return get_page(r->args);
}

static apr_uint32_t get_upload_id(request_rec *r)
{
    return get_upload_id(r->args);
}

static const char *get_name_encoding(request_rec *r)
{
    return get_name_encoding(apr_table_get(r->headers_in, "User-Agent"));
}

/**
 * åץɤοĽΥꥹȤ֤ޤ
 *
 * ֤ꥹȤƬϥߡΥȥǼºݤˤϻѤޤ
 * Null Object ѥ
 *
 * @param[in] r ꥯ
 */
static pprogress *get_progress_list(request_rec *r)
{
    return static_cast<sconfig *>(ap_get_module_config(r->server->module_config,
                                                       &uploader_module))->progress_list;
}

/**
 * ƾΥꥹȤ֤ޤ
 *
 * @param[in] r ꥯ
 * @return ƾΥꥹ
 */
static poster *get_poster_list(request_rec *r)
{
    return static_cast<sconfig *>(ap_get_module_config(r->server->module_config,
                                                       &uploader_module))->poster_list;
}

static int error(request_rec *r, uconfig *config, const char *error)
{
    page_template *tmpl = config->get_error_template(r->pool);

    auto_ptr<variable_map> vmap(get_error_vmap(r->pool, config, tmpl, error));

    ap_set_content_type(r, "text/html");

    ApacheResponseWriter writer(r);
    ApacheTemplateExecutor texecutor(r->pool, tmpl->imap, vmap.get(), writer);
    texecutor.exec(r->pool, tmpl->node);

    return OK;
}

static int redirect_top(request_rec *r)
{
    // POST ˥ڡñ˥ɤǤ褦ˤ뤿ᡤHTML 
    // 쥯Ȥ롥

    ap_set_content_type(r, "text/html");

    ap_rputs("<?xml version=\"1.0\" encoding=\"EUC-JP\"?>", r);
    ap_rputs("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">", r);
    ap_rputs("<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"ja\" >", r);
    ap_rputs(" <head>", r);
    ap_rputs("  <meta http-equiv=\"content-type\" content=\"application/xhtml+xml\" />", r);
    ap_rputs("  <meta http-equiv=\"refresh\" content=\"0;url=../\" />", r);
    ap_rputs("  <title>mod_uploader</title>", r);
    ap_rputs(" </head>", r);
    ap_rputs(" <body />", r);
    ap_rputs("</html>", r);

    return OK;
}

static int input_pass(request_rec *r, uconfig *config, const char *arg,
                      UploadItem::header *header)
{
    try {
        page_template *tmpl = config->get_download_template(r->pool);

        auto_ptr<variable_map> vmap(get_download_vmap(r->pool, config, tmpl, header));

        ap_set_content_type(r, "text/html");

        ApacheResponseWriter writer(r);
        ApacheTemplateExecutor texecutor(r->pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(r->pool, tmpl->node);
    } catch(const char *message) {
        ap_rputs("<div class=\"warning\">", r);
        ap_rputs(message, r);
        ap_rputs("</div>", r);
    }

    return OK;
}

static int view(request_rec *r, uconfig *config, const char *arg)
{
    apr_status_t status;

    config->read_lock();

    ap_update_mtime(r, config->item_list->get_mtime());
    ap_set_last_modified(r);
    ap_set_etag(r);

    if ((status = ap_meets_conditions(r)) != OK) {
        config->read_unlock();
        return status;
    }

    ap_set_content_type(r, "text/html");
    if (r->header_only) {
        config->read_unlock();
        return OK;
    }

    try {
        page_template *tmpl = config->get_view_template(r->pool);

        auto_ptr<variable_map> vmap(get_view_vmap(r->pool, config, tmpl, get_page(r)));

        ApacheResponseWriter writer(r);
        ApacheTemplateExecutor texecutor(r->pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(r->pool, tmpl->node);

        config->read_unlock();
    } catch(const char *message) {
        config->read_unlock();

        ap_rputs("<div class=\"warning\">", r);
        ap_rputs(message, r);
        ap_rputs("</div>", r);
    }

    return OK;
}

#ifdef MAKE_THUMBNAIL
static int thumbnail(request_rec *r, uconfig *config, const char *arg)
{
    apr_status_t status;

    config->read_lock();

    ap_update_mtime(r, config->item_list->get_mtime());
    ap_set_last_modified(r);
    ap_set_etag(r);

    if ((status = ap_meets_conditions(r)) != OK) {
        config->read_unlock();
        return status;
    }

    ap_set_content_type(r, "text/html");
    if (r->header_only) {
        config->read_unlock();
        return OK;
    }

    try {
        page_template *tmpl = config->get_thumb_template(r->pool);

        auto_ptr<variable_map> vmap(get_thumb_vmap(r->pool, config, tmpl, get_page(r)));

        ApacheResponseWriter writer(r);
        ApacheTemplateExecutor texecutor(r->pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(r->pool, tmpl->node);

        config->read_unlock();
    } catch(const char *message) {
        config->read_unlock();

        ap_rputs("<div class=\"warning\">", r);
        ap_rputs(message, r);
        ap_rputs("</div>", r);
    }

    return OK;
}
#endif

static int rss(request_rec *r, uconfig *config, const char *arg)
{
    apr_status_t status;

    config->read_lock();

    ap_update_mtime(r, config->item_list->get_mtime());
    ap_set_last_modified(r);
    ap_set_etag(r);

    if ((status = ap_meets_conditions(r)) != OK) {
        config->read_unlock();
        return status;
    }

    ap_set_content_type(r, ApacheRssView::content_type);
    if (r->header_only) {
        config->read_unlock();
        return OK;
    }

    try {
        ApacheResponseWriter writer(r);
        ApacheRssView view(r->pool, writer);
        view.exec(config->url, config->per_page_item_number, config->item_list);

        config->read_unlock();
    } catch(const char *message) {
        config->read_unlock();

        ap_rputs("<div class=\"warning\">", r);
        ap_rputs(message, r);
        ap_rputs("</div>", r);
    }

    return OK;
}

static int upload(request_rec *r, uconfig *config, const char *arg)
{
    const char *address;
    apr_uint32_t upload_id;
    pprogress *progress;
    apr_status_t status;

    if (r->method_number != M_POST) {
        return HTTP_BAD_REQUEST;
    }
    if ((status = ap_setup_client_block(r, REQUEST_CHUNKED_ERROR)) != OK) {
        return status;
    }
    if (!ap_should_client_block(r)) {
        return HTTP_NO_CONTENT;
    }

    address = r->connection->remote_ip;
    if (is_poster_list_contain(get_poster_list(r), address)) {
        return error(r, config,
                     apr_psprintf(r->pool, "%" APR_SIZE_T_FMT  " ô֤ϺƤǤޤ",
                                  POST_INTERVAL_SEC));
    }

    upload_id = get_upload_id(r);
    progress = add_progress_list(get_progress_list(r), upload_id);

    try {
        UploadItem::header *header;
        const char *tmp_path;
        const char *file_name;
        apr_size_t content_size;

        content_size = static_cast<apr_size_t>(apr_atoi64(apr_table_get(r->headers_in,
                                                                        "Content-Length")));
        progress->total_size = content_size;

        ApachePostReader reader(progress, r);
        auto_ptr<RFC1867Data::query_map>
            qmap(config->uparser->parse(r->pool, reader,
                                        apr_table_get(r->headers_in, "Content-Type"),
                                        content_size));

        try {
            header = UploadItemCreator::create_header(r->pool, qmap.get(), &tmp_path);
        } catch(const char *) {
            ApacheRFC1867Parser::clean_tmp_file(r->pool, qmap.get());

            throw;
        }

        auto_ptr<UploadItemWriter> writer(config->get_uwriter(r->pool));
        file_name = writer->write(header, tmp_path);

        config->gwrite_lock();
        try {
            sconfig *sconfig = get_sconfig(r);

            config->item_list->add(file_name, true);

            add_poster_list(get_poster_list(r), address);
            progress->end_time = apr_time_now();

            *(sconfig->revision) = *(sconfig->revision) + 1;
            config->item_list->set_revision(*(sconfig->revision));

            config->gwrite_unlock();
        } catch(const char *) {
            config->gwrite_unlock();
            throw;
        }

        INFO("upload", file_name);

        return redirect_top(r);
    } catch(const char *message) {
        progress->end_time = apr_time_now();

        return error(r, config, message);
    }
}

static int progress(request_rec *r, uconfig *config, const char *arg)
{
    try {
        page_template *tmpl = config->get_progress_template(r->pool);

        auto_ptr<variable_map> vmap(get_progress_vmap(r->pool, config, tmpl));

        ap_set_content_type(r, "text/html");

        ApacheResponseWriter writer(r);
        ApacheTemplateExecutor texecutor(r->pool, tmpl->imap, vmap.get(), writer);
        texecutor.exec(r->pool, tmpl->node);
    } catch(const char *message) {
        ap_rputs("<div class=\"warning\">", r);
        ap_rputs(message, r);
        ap_rputs("</div>", r);
    }

    return OK;
}

static int progress_data(request_rec *r, uconfig *config, const char *arg)
{
    apr_uint32_t upload_id;
    pprogress *progress;

    upload_id = get_upload_id(r);

    // upload_id  0 λϥ顼
    if (UNLIKELY(upload_id == 0)) {
        ap_rputs("e", r);
        return OK;
    }

    progress = get_progress(get_progress_list(r), upload_id);

    if (UNLIKELY(progress == get_progress_list(r))) {
        ap_rputs("u", r);
        return OK;
    }

    ap_rprintf(r, "s %" APR_UINT64_T_FMT " %" APR_UINT64_T_FMT,
               progress->total_size, progress->read_size);
    return OK;
}

static int http_sendfile(apr_file_t *file, request_rec *r, apr_off_t offset,
                         apr_size_t length,
                         bool is_sendfile_enabled, bool is_mmap_enabled)
{
    conn_rec *c;
    apr_bucket_brigade *bb;
    apr_bucket *b;

    c = r->connection;

    bb = apr_brigade_create(r->pool, c->bucket_alloc);

    b = apr_bucket_file_create(file, offset, length, r->pool, c->bucket_alloc);
#if APR_HAS_MMAP
    if (!is_mmap_enabled) {
        apr_bucket_file_enable_mmap(b, 0);
    }
#endif

    APR_BRIGADE_INSERT_TAIL(bb, b);
    b = apr_bucket_eos_create(c->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(bb, b);

    if (ap_pass_brigade(r->output_filters, bb) != APR_SUCCESS) {
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    return OK;
}

static void get_core_option(request_rec *r,
                            bool *is_sendfile_enabled,
                            bool *is_mmap_enabled)
{
    core_dir_config *core_config;

    core_config = (core_dir_config *)ap_get_module_config(r->per_dir_config,
                                                          &core_module);

    *is_sendfile_enabled = false;
    *is_mmap_enabled = false;

#if APR_HAS_SENDFILE
    *is_sendfile_enabled = (core_config->enable_sendfile != ENABLE_SENDFILE_OFF);
#endif

#if APR_HAS_MMAP
    *is_mmap_enabled = (core_config->enable_mmap != ENABLE_MMAP_OFF);
#endif
}

static int download(request_rec *r, uconfig *config, const char *arg)
{
    const char *name;

    name = ap_getword(r->pool, &arg, '/');
    if (name == NULL) {
        return HTTP_BAD_REQUEST;
    }

    {
        // ե̾Υå
        const char *pos = name;

        while (isalnum(*pos) || (*pos == '.')) {
            pos++;
        }

        if ((static_cast<size_t>(pos-name) != strlen(name)) ||
            (strlen(name) > APR_PATH_MAX)) {
            return HTTP_FORBIDDEN;
        }
    }

    try {
        apr_file_t *file;
        UploadItem::header *header;
        const char *file_name;
        const char *password;
        const char *open_mode;
        apr_status_t status;
        bool is_sendfile_enabled;
        bool is_mmap_enabled;

        get_core_option(r, &is_sendfile_enabled, &is_mmap_enabled);

        auto_ptr<UploadItemReader> reader(config->get_ureader(r->pool));
        header = reader->read(name, &file, false, is_sendfile_enabled);

        password = apr_table_get(get_param_table(r), UploadItem::DOWNLOAD_PASS_PARAM);

        // DL pass Υå
        if ((strlen(header->download_pass) > 0) &&
            ((password == NULL) ||
             (strncmp(header->download_pass, password, strlen(header->download_pass)) != 0))) {
            return input_pass(r, config, arg, header);
        }

        ap_update_mtime(r, header->time);
        ap_set_last_modified(r);
        ap_set_etag(r);
        apr_table_setn(r->headers_out, "Accept-Ranges", "bytes");

        if ((status = ap_meets_conditions(r)) != OK) {
            return status;
        }

        ap_set_content_type(r, header->file_mime);

        if (r->header_only) {
            return OK;
        }

        file_name = CharCodeConverter::convert(r->pool, header->file_name,
                                               CharCodeConverter::DEFAULT_CODE,
                                               get_name_encoding(r));

        // ɽ⡼ or ɥ⡼
        open_mode = (r->args == NULL) ? "inline" : "attachment";

        // Content-Disposition
        apr_table_setn(r->headers_out, "Content-Disposition",
                       apr_psprintf(r->pool,
                                    "%s; filename=\"%s\"", open_mode, file_name));

        return http_sendfile(file, r, sizeof(UploadItem::header),
                             static_cast<apr_size_t>(header->file_size), // MEMO: 4GB 
                             is_sendfile_enabled, is_mmap_enabled);
    } catch(const char *message) {
        return error(r, config, message);
    }
}

static int remove(request_rec *r, uconfig *config, const char *arg)
{
    apr_table_t *ptable;

    if (r->method_number != M_POST) {
        return HTTP_BAD_REQUEST;
    }

    try {
        const char *file_name;
        const char *password;

        ptable = get_param_table(r);

        file_name = apr_table_get(ptable, UploadItem::FILE_PARAM);
        password = apr_table_get(ptable, UploadItem::REMOVE_PASS_PARAM);

        if ((file_name == NULL) || (password == NULL)) {
            return error(r, config, "եޤϥѥɤꤵƤޤ");
        }

        config->gwrite_lock();
        try {
            sconfig *sconfig = get_sconfig(r);

            config->item_list->remove(file_name, password);

            *(sconfig->revision) = *(sconfig->revision) + 1;
            config->item_list->set_revision(*(sconfig->revision));

            config->remove_item_number++;
            config->gwrite_unlock();

            INFO("remove", file_name);
        } catch(const char *) {
            config->gwrite_unlock();
            throw;
        }

        return redirect_top(r);
    } catch(const char *message) {
        return error(r, config, message);
    }
}

int command_handler(request_rec *r, uconfig *config, const char *arg)
{
    const char *command;

    if (UNLIKELY(*arg == '\0')) {
        if (UNLIKELY(*(r->uri + strlen(r->uri) - 1) == '/')) {
            return error(r, config,
                         apr_psprintf(r->pool,
                                      "ǥ쥯ȥޤϥե%sפ¸ߤƤޤ",
                                      r->filename));
        }
        apr_table_set(r->headers_out, "Location", apr_pstrcat(r->pool, r->uri, "/", NULL));
        return HTTP_TEMPORARY_REDIRECT;
    } else {
        arg++;
    }

    command = ap_getword(r->pool, &arg, '/');

    if (LIKELY(*command == '\0')) {
        return view(r, config, arg);
    } else if (strcmp(command, RSS_COMMAND) == 0) {
        return rss(r, config, arg);
    } else if (strcmp(command, UPLOAD_COMMAND) == 0) {
        return upload(r, config, arg);
    } else if (strcmp(command, PROGRESS_COMMAND) == 0) {
        return progress(r, config, arg);
    } else if (strcmp(command, PROGRESS_DATA_COMMAND) == 0) {
        return progress_data(r, config, arg);
    } else if (strcmp(command, DOWNLOAD_COMMAND) == 0) {
        return download(r, config, arg);
    } else if (strcmp(command, REMOVE_COMMAND) == 0) {
        return remove(r, config, arg);
#ifdef MAKE_THUMBNAIL
    } else if (strcmp(command, THUMBNAIL_COMMAND) == 0) {
        return thumbnail(r, config, arg);
#endif
    } else {
        return error(r, config, "ݡȤƤʤޥɤǤ");
    }
}

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
