/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or nolockied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: UploadItemManager.h 2098 2006-11-26 09:11:25Z svn $
 *****************************************************************************/

#ifndef UPLOAD_ITEM_MANAGER_H
#define UPLOAD_ITEM_MANAGER_H

#include "Environment.h"

#include "apr_shm.h"
#include "AtomicWrapper.h"

// #include "BBSCommentIterator.h"
#include "UploadItemReader.h"
#include "UploadItemWriter.h"

// TODO: 削除
#include "UploadItemList.h"
#include "Uncopyable.h"

using namespace std;

// class UploadItem;
// class BBSCommentIterator;
// class Locker;
// class UploadItemList;

/**
 * @brief BBS のスレッドを管理するクラス．
 *
 * 更新日時の新しい MGR_ON_MEMORY_ITEM_COUNT 個のスレッドの内容をメモリ上で
 * 管理します.
 */
class UploadItemManager: public Uncopyable
{
public:
    /**
     * 初期化を行います．
     *
     * @param[in] item_list スレッドのリスト
     */
    void init(UploadItemList *item_list);
    void add(apr_pool_t *pool,
             const char *file_name, const char *file_mime,
             apr_off_t file_size, const char *file_digest,
             const char *file_temp_path, const char *remove_pass,
             const char *download_pass, const char *comment,
             const char *ip_address);
    void remove(apr_pool_t *pool, UploadItem *uitem);
    UploadItem *get_item(apr_pool_t *pool, apr_size_t item_id);
    apr_file_t *get_item_file(apr_pool_t *pool, UploadItem *uitem);
    apr_time_t get_mtime() const throw()
    {
        return *mtime_;
    };
    /**
     * 共有メモリを使ってインスタンスを生成します．
     *
     * @param[in] shm 共有メモリ
     * @param[in] pool プールメモリ
     * @param[in] data_dir_path アイテムが保存されているディレクトリ
     * @param[in] file_dir_path ファイルが保存されているディレクトリ
     * @param[in] thumb_dir_path サムネイルが保存されているディレクトリ
     * @return UploadItemManager のインスタンス
     */
    static UploadItemManager *get_instance(apr_shm_t *shm, apr_pool_t *pool,
                                           const char *data_dir_path,
                                           const char *file_dir_path,
                                           const char *thumb_dir_path);
    /**
     * メモリ上にあるアイテムデータをディスクに書き出します．
     */
    void sync_all_item(apr_pool_t *pool);
    /**
     * 共有メモリを使って子プロセス用の初期化を行います．
     *
     * @param[in] shm 共有メモリ
     * @param[in] item_list アイテムのリスト
     * @return UploadItemManager のインスタンス
     */
    static UploadItemManager *child_init(apr_shm_t *shm,
                                         UploadItemList *item_list);
    /**
     * マネージャのデータの最大サイズを計算します．
     *
     * @return マネージャのデータの最大サイズ
     */
    static apr_size_t get_memory_size();
    /**
     * 終了状態のチェックを行います．
     * マネージャがロックされている場合は例外を投げます．
     *
     * @param[in] item_manager マネージャ
     */
    static void check_finalize_state(UploadItemManager *item_manager);

private:
    UploadItemManager(apr_pool_t *pool, const char *data_dir_path,
                      const char *file_dir_path, const char *thumb_dir_path);
    /**
     * 子プロセス用の初期化を行います．
     *
     * @param[in] item_list スレッドのリスト
     */
    void child_init_impl(UploadItemList *item_list);
    /**
     * 共有メモリをクリアします．
     */
    void clear_shm();
    /**
     * 共有メモリを初期化します．
     */
    void init_shm();
    /**
     * 共有メモリを子プロセス用に初期化します．
     */
    void attach_shm();
    /**
     * ユニークなアイテム ID とインデックスを生成します．
     */
    void create_item_id_index(apr_size_t *item_id,
                              apr_size_t *item_index) const;

    apr_pool_t *pool_;

    const char *data_dir_path_;
    const char *file_dir_path_;
    const char *thumb_dir_path_;

    UploadItemWriter item_writer_;

    UploadItemList *item_list_;
    /** 最終更新時刻 (in in 共有メモリ) */
    apr_time_t *mtime_;
    /** マネージャへの参照の数 (in in 共有メモリ) */
    apr_atomic_t *manager_ref_count_;
};

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
