/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: UploadItemListReader.cpp 2077 2006-11-25 11:38:44Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <memory>

#include "AtomicWrapper.h"
#include "apr_file_io.h"
#include "apr_mmap.h"
#include "apr_strings.h"

#include "UploadItemListReader.h"
#include "UploadItemList.h"
#include "UploadItemReader.h"
#include "TemporaryPool.h"
#include "File.h"
#include "Auxiliary.h"
#include "Message.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: UploadItemListReader.cpp 2077 2006-11-25 11:38:44Z svn $");

/******************************************************************************
 * public メソッド
 *****************************************************************************/
UploadItemList *UploadItemListReader::read(apr_pool_t *pool,
                                           const char *dir_path,
                                           apr_shm_t *shm,
                                           apr_size_t max_list_size)
{
    UploadItemList *item_list;
    apr_dir_t *dir;
    apr_finfo_t sub_dir_info;
    char *sub_dir_path;
    TemporaryPool temp_pool(pool);

    UploadItemReader uitem_reader(temp_pool.get(), dir_path);

    if (apr_dir_open(&dir, dir_path, temp_pool.get()) != APR_SUCCESS) {
        THROW(MESSAGE_UPLOAD_ITEM_DIR_OPEN_FAILED);
    }

    item_list = UploadItemList::get_instance(shm, max_list_size);

    while (apr_dir_read(&sub_dir_info, APR_FINFO_NAME|APR_FINFO_TYPE,
                        dir) == APR_SUCCESS) {
        if ((sub_dir_info.filetype != APR_DIR) ||
            !is_subdir_name_valid(sub_dir_info.name)) {
            continue;
        }
        if (apr_filepath_merge(&sub_dir_path, dir_path,
                               sub_dir_info.name, APR_FILEPATH_NOTABOVEROOT,
                               temp_pool.get()) != APR_SUCCESS) {
            THROW(MESSAGE_UPLOAD_ITEM_SUB_DIR_PATH_CREATION_FAILED);
        }

        get_subdir_threads(temp_pool.get(), sub_dir_path, &uitem_reader,
                           item_list);
    }
    apr_dir_close(dir);

    return item_list;
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
void UploadItemListReader::get_subdir_threads(apr_pool_t *pool,
                                             const char *sub_dir_path,
                                             UploadItemReader *uitem_reader,
                                             UploadItemList *item_list)
{
    apr_dir_t *sub_dir;
    apr_finfo_t file_info;
    TemporaryPool temp_pool(pool);
    char buffer[sizeof(UploadItem)];
    UploadItem *uitem = AS_UITEM(buffer);

    if (apr_dir_open(&sub_dir, sub_dir_path, temp_pool.get()) != APR_SUCCESS) {
        THROW(MESSAGE_UPLOAD_ITEM_SUB_DIR_OPEN_FAILED);
    }

    while (apr_dir_read(&file_info,
                        APR_FINFO_NAME|APR_FINFO_TYPE|APR_FINFO_SIZE,
                        sub_dir) == APR_SUCCESS) {
        if ((file_info.filetype != APR_REG) ||
            !is_file_name_valid(file_info.name)) {
            continue;
        }

        uitem_reader->read(atosize(file_info.name), uitem);
        item_list->add(uitem);
    }
    apr_dir_close(sub_dir);
}

bool UploadItemListReader::is_subdir_name_valid(const char *subdir_name)
{
    apr_size_t i;

    for (i = 0; isxdigit(static_cast<unsigned char>(subdir_name[i])) != 0; i++);

    return (subdir_name[i] == '\0');
}

bool UploadItemListReader::is_file_name_valid(const char *file_name)
{
    apr_size_t i;

    for (i = 0; isdigit(static_cast<unsigned char>(file_name[i])) != 0; i++);

    return (file_name[i] == '\0');
}

/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_UploadItemListReader
#include <memory>

#include "apr_shm.h"

#include "TestRunner.h"

static const apr_size_t MAX_LIST_SIZE   = 1000;

void show_usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <DAT_DIR_PATH>" << endl;
}

static void run_sort_by_mtime(UploadItemList *item_list)
{
    show_test_name("sort by mtime");
}

void run_all(apr_pool_t *pool, int argc, const char * const *argv)
{
    const char *data_dir_path;
    apr_shm_t *shm;
    apr_size_t dump_level;

    if (argc == 1) {
        THROW(MESSAGE_ARGUMENT_INVALID);
    }

    data_dir_path = argv[1];
    if (argc >= 3) {
        dump_level = atoi(argv[2]);
    } else {
        dump_level = 0;
    }

    if (!is_exist(pool, data_dir_path)) {
        THROW(MESSAGE_DATA_DIR_NOT_FOUND);
    }

    shm = create_shm(pool, UploadItemList::get_memory_size(MAX_LIST_SIZE));

    UploadItemList *item_list = UploadItemListReader::read(pool, data_dir_path,
                                                           shm, MAX_LIST_SIZE);

    show_item("data_dir", data_dir_path);
    show_item("manager memory",
              UploadItemList::get_memory_size(MAX_LIST_SIZE) /
              static_cast<double>(1024), " KB");
    show_line();

    if (dump_level > 2) {
        show_test_name("read");
        UploadItemList::dump_header_list(pool, item_list);
        show_line();
    }

    run_sort_by_mtime(item_list);
}

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
