/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: UploadItemReader.cpp 2077 2006-11-25 11:38:44Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <algorithm>

#include "apr_file_info.h"

#include "UploadItemReader.h"
#include "UploadItem.h"
#include "TemporaryPool.h"
#include "TemporaryFile.h"
#include "Message.h"
#include "Macro.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: UploadItemReader.cpp 2077 2006-11-25 11:38:44Z svn $");

/******************************************************************************
 * public メソッド
 *****************************************************************************/
UploadItemReader::UploadItemReader(apr_pool_t *pool, const char *data_dir_path,
                                   const char *file_dir_path,
                                   const char *thumb_dir_path)
  : UploadItemIO(pool, data_dir_path, file_dir_path, thumb_dir_path)
{

}

void UploadItemReader::read(apr_size_t item_id, UploadItem *bbs_thread)
{
    const char *uitem_file_path;
    apr_mmap_t *uitem_file_map;

    TemporaryPool temp_pool(pool_);
    uitem_file_path = get_data_path(temp_pool.get(), item_id);
    File uitem_file(temp_pool.get(), uitem_file_path);

    read_and_check(&uitem_file, &uitem_file_map);

    memcpy(bbs_thread, uitem_file_map->mm, sizeof(UploadItem));
}

void UploadItemReader::read(apr_size_t item_id, UploadItem **bbs_thread,
                            apr_pool_t *pool, File& uitem_file) const
{
    const char *uitem_file_path;
    apr_mmap_t *uitem_file_map;

    uitem_file_path = get_data_path(pool, item_id);
    uitem_file = File(pool, uitem_file_path);

    read_and_check(&uitem_file, &uitem_file_map);

    *bbs_thread = AS_UITEM(uitem_file_map->mm);
}

bool UploadItemReader::is_exist(apr_size_t item_id) const
{
    const char *uitem_file_path;
    apr_finfo_t info;
    TemporaryPool temp_pool;

    uitem_file_path = get_data_path(temp_pool.get(), item_id);

    return !APR_STATUS_IS_ENOENT(apr_stat(&info, uitem_file_path,
                                          APR_FINFO_MIN, temp_pool.get()));
}

UploadItem *UploadItemReader::read(apr_pool_t *pool, const char *dir_path,
                                 apr_size_t item_id)
{
    UploadItem *uitem;

    UploadItemReader reader(pool, dir_path);

    APR_PALLOC(uitem, UploadItem *, pool, sizeof(UploadItem));
    reader.read(item_id, uitem);

    return uitem;
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
inline void UploadItemReader::read_and_check(File *uitem_file,
                                            apr_mmap_t **uitem_file_map) const
{
    uitem_file->open(APR_READ|APR_BINARY);
    *uitem_file_map = uitem_file->mmap(0, 0, APR_MMAP_READ, true);

    check_format(*uitem_file_map);
}

inline void UploadItemReader::check_format(apr_mmap_t *uitem_file_map)
{
    UploadItem *uitem;

    uitem = AS_UITEM(uitem_file_map->mm);

    if (uitem_file_map->size < sizeof(UploadItem)) {
        THROW(MESSAGE_UPLOAD_ITEM_FORMAT_INVALID);
    }

    if (UNLIKELY(strncmp(uitem->get_identifier(), PACKAGE_NAME,
                         ITM_MAX_IDENTIFIER_SIZE) != 0)) {
        THROW(MESSAGE_UPLOAD_ITEM_FORMAT_INVALID);
    }
    if (UNLIKELY(strncmp(uitem->get_version(), PACKAGE_VERSION,
                         ITM_MAX_VERSION_SIZE) > 0)) {
        THROW(MESSAGE_PROGRAM_TOO_OLD);
    }
}

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
