/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: UploadItemRss.cpp 2096 2006-11-26 08:43:57Z svn $
 *****************************************************************************/

#ifndef TEMPLATE_INSTANTIATION
#include "Environment.h"
#endif

#include "UploadItemRss.h"
#include "UploadItem.h"
#include "UploadItemIterator.h"
#include "Auxiliary.h"
#include "SourceInfo.h"

#ifndef TEMPLATE_INSTANTIATION
SOURCE_INFO_ADD("$Id: UploadItemRss.cpp 2096 2006-11-26 08:43:57Z svn $");
#endif

template<class W> const char
UploadItemRss<W>::CONTENT_TYPE[]      = "application/xml";
template<class W> const apr_size_t
UploadItemRss<W>::MAX_TIME_SIZE       = sizeof("0000-00-00T00:00:00+09:00");
template<class W> const char
UploadItemRss<W>::TIME_FORMAT[]       = "%Y-%m-%dT%H:%M:%S" SYS_TIMEZONE_OFFSET_SUFFIX;
template<class W> const apr_int32_t
UploadItemRss<W>::TIME_ZONE_OFFSET    = SYS_TIMEZONE_OFFSET;

/******************************************************************************
 * public メソッド
 *****************************************************************************/
template<class W>
UploadItemRss<W>::UploadItemRss(apr_pool_t *pool, W& writer)
  : pool_(pool),
    writer_(writer)
{

}

template<class W>
void UploadItemRss<W>::print(apr_pool_t *pool, const char *base_url,
                             UploadItemIterator *item_iter, apr_time_t mtime)
{
    UploadItem *uitem;;

    print("<?xml version=\"1.0\" encoding=\"EUC-JP\"?>\n");
    print("<rdf:RDF xmlns:rdf=\"http://www.w3.org/1999/02/22-rdf-syntax-ns#\"\n");
    print("         xmlns=\"http://purl.org/rss/1.0/\"\n");
    print("         xmlns:dc=\"http://purl.org/dc/elements/1.1/\">\n");
    print(" <channel rdf:about=\""); print(base_url); print("\">\n");
    print("  <title>" PACKAGE_NAME "</title>\n");
    print("  <link>"); print(base_url); print("</link>\n");
    print("  <description>" PACKAGE_STRING "</description>\n");
    print("  <dc:language>ja-jp</dc:language>\n");
    print("  <dc:date>");
    print(time_str(pool_, mtime));
    print("</dc:date>\n");

    if (item_iter->size() != 0) {
        print(" <items>\n");
        print("  <rdf:Seq>\n");
        do {
            uitem = item_iter->get();

            print("   <rdf:li rdf:resource=\"");
            print(base_url);
            print("/download/");
            print(uitem->get_id());
            print("/");
            print("\" />\n");
        } while (item_iter->next());
        print("  </rdf:Seq>\n");
        print(" </items>\n");

        item_iter->reset();

        do {
            print(pool, base_url, item_iter->get());
        } while (item_iter->next());
    }

    print(" </channel>\n");
    print("</rdf:RDF>\n");
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
template<class W>
void UploadItemRss<W>::print(apr_pool_t *pool, const char *base_url,
                             UploadItem *uitem)
{
    print("  <item rdf:about=\"");
    print(base_url);
    print("/download/");
    print(uitem->get_id());
    print("/");
    print("\">\n");

    print("   <title>");
    print(uitem->get_file_name());
    print("</title>\n");

    print("<link>");
    print(base_url);
    print("/download/");
    print(uitem->get_id());
    print("/");
    print("</link>\n");

    print("   <description><![CDATA[");
    print(uitem->get_comment());
    print("]]></description>\n");

    print("   <dc:date>");
    print(time_str(pool_, uitem->get_mtime()));
    print("</dc:date>\n");

    print("  </item>\n");
}

template<class W>
inline void UploadItemRss<W>::print(const char *str, apr_size_t length)
{
    writer_.write(str, length);
}

template<class W>
inline void UploadItemRss<W>::print(const char *str)
{
    writer_.write(str);
}

template<class W>
inline void UploadItemRss<W>::print(apr_size_t i)
{
    writer_.write(static_cast<int>(i));
}

template<class W>
const char *UploadItemRss<W>::time_str(apr_pool_t *pool,
                                       apr_time_t time)
{
    apr_time_exp_t time_exp;
    char *time_str;
    apr_size_t size;

    APR_PALLOC(time_str, char *, pool, sizeof(char)*MAX_TIME_SIZE);

    apr_time_exp_tz(&time_exp, time, TIME_ZONE_OFFSET);
    apr_strftime(time_str, &size, MAX_TIME_SIZE, TIME_FORMAT, &time_exp);

    return time_str;
}


/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_UploadItemRss
// #include <memory>

// #include "TestRunner.h"
// #include "BBSThreadReader.h"
// #include "ShmCommentIterator.h"
// #include "CGIResponseWriter.h"

// static const char *TEST_URL     = "http://www.example.com";

// void show_usage(const char *prog_name)
// {
//     cerr << "Usage: " << prog_name << " <DATA_DIR_PATH> <THREAD_ID>" << endl;
// }

// static void run_exec(apr_pool_t *pool, const char *data_dir_path,
//                      apr_size_t thread_id)
// {
//     BBSThread *bbs_thread;
//     UploadItemIterator *bcomment_iter;
//     apr_atomic_t lock_count;
//     UploadItemIterator::range_t ranges[1];
//     apr_size_t range_count;

//     show_test_name("print");

//     apr_atomic_set(&lock_count, 0);

//     BBSThreadReader reader(pool, data_dir_path);
//     bbs_thread = BBSThread::get_instance(pool);
//     reader.read(thread_id, bbs_thread);

//     ranges[0].start_no = UploadItemIterator::RANGE_FROM_LAST;
//     ranges[0].stop_no = 10;
//     range_count = 1;

//     bcomment_iter = new ShmCommentIterator(bbs_thread, &lock_count,
//                                            ranges, range_count);
//     auto_ptr<UploadItemIterator> bcomment_iter_ap(bcomment_iter);

//     CGIResponseWriter writer;
//     UploadItemRss<CGIResponseWriter> comment_rss(pool, writer);
//     comment_rss.print(TEST_URL, bcomment_iter);

//     show_spacer();
// }

// void run_all(apr_pool_t *pool, int argc, const char * const *argv)
// {
//     const char *data_dir_path;
//     apr_size_t thread_id;

//     if (argc != 3) {
//         THROW(MESSAGE_ARGUMENT_INVALID);
//     }

//     data_dir_path = argv[1];
//     thread_id = atoi(argv[2]);

//     if (!is_exist(pool, data_dir_path)) {
//         THROW(MESSAGE_DATA_DIR_NOT_FOUND);
//     }

//     show_item("data dir", data_dir_path);
//     show_item("thread id", thread_id);
//     show_line();

//     run_exec(pool, data_dir_path, thread_id);
// }

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
