/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: ReadWriteLocker.h 2459 2007-06-29 16:38:34Z svn $
 *****************************************************************************/

#ifndef READ_WRITE_LOCKER_H
#define READ_WRITE_LOCKER_H

#include "Environment.h"

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SCHED_H
#include <sched.h>
#endif

#include "AtomicWrapper.h"
#include "apr_thread_proc.h"
#include "apr_time.h"

#include "Uncopyable.h"
#include "Macro.h"

/**
 * @brief 読み書きロックの設定を集めたクラス．
 */
class ReadWriteLocker: public Uncopyable
{
public:
    static const apr_uint32_t NOT_LOCKED        = 0x0000;

protected:
    /**
     * コンストラクタです．
     *
     * @param[out] lock ロック変数
     */
    ReadWriteLocker(apr_atomic_t *lock)
     : lock_(lock),
       start_age_(0xFFFF),
       start_time_(apr_time_now()),
       status_(0)
    {

    };

    ReadWriteLocker(const ReadWriteLocker& locker)
      : lock_(locker.lock_)
    {

    };

    ReadWriteLocker();

    /**
     * プロセッサーを明け渡します．
     */
    void yield()
    {
#if APR_HAS_THREADS
        // APR 0.x だと UNIX 環境の apr_thread_yield の中身は空なので
#if (APR_MAJOR_VERSION == 0) && defined( _POSIX_PRIORITY_SCHEDULING)
        sched_yield();
#else
        apr_thread_yield();
#endif
#endif
    };
    /**
     * タイムアウト処理すべきかどうか判断します．
     *
     * @param[in] status ロックの状態
     */
    bool is_timeout(apr_uint32_t status);

    // 注意: ReadWriteLocker 型のポインタ経由で使用されることはないので，
    // デストラクタは virtual にしない．

    static const apr_uint32_t WRITE_LOCKED      = 0xFFFF;

    static const apr_size_t AGE_MASK            = 0xFF;
    static const apr_size_t AGE_SHIFT           = 16;
    static const apr_size_t READ_COUNT_MASK     = 0xFFFF;
    static const apr_size_t READ_COUNT_SHIFT    = 0;

    static const apr_size_t TIMEOUT_CHECK_MASK  = 0x1F;
    static const apr_size_t TIMEOUT_SEC         = 30;

    static apr_uint32_t get_age(apr_uint32_t status)
    {
        return (status >> AGE_SHIFT) & AGE_MASK;
    };
    static apr_uint32_t get_read_count(apr_uint32_t status)
    {
        return (status >> READ_COUNT_SHIFT) & READ_COUNT_MASK;
    };
    static apr_uint32_t make_status(apr_uint32_t age, apr_uint32_t read_count)
    {
        return ((age & AGE_MASK) << AGE_SHIFT) |
            ((read_count & READ_COUNT_MASK) << READ_COUNT_SHIFT);
    };

    /** 読み出し中のスレッドの数へのポインタ */
    apr_atomic_t *lock_;
    apr_uint32_t start_age_;
    apr_time_t start_time_;
    apr_uint32_t status_;
};

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
