/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: ReadLocker.h 2483 2007-07-03 13:17:30Z svn $
 *****************************************************************************/

#ifndef READ_LOCKER_H
#define READ_LOCKER_H

#include "Environment.h"

#include "ReadWriteLocker.h"

#ifdef DEBUG
#include <iostream>
#endif
using namespace std;

/**
 * @brief 読み出しロックを表すクラス．
 */
class ReadLocker: public ReadWriteLocker
{
public:
    /**
     * コンストラクタです．
     *
     * @param[out] lock ロック変数
     */
    ReadLocker(apr_atomic_t *lock)
      : ReadWriteLocker(lock)
    {
        apr_size_t try_count;
        apr_uint32_t status;
        apr_uint32_t new_status;

        try_count = 0;
        while (1) {
            try_count++;

            status = apr_atomic_read(lock_);
            if (LIKELY((get_read_count(status) != WRITE_LOCKED) &&
                       ((get_wait_write(status) != WAIT_WRITE) ||
                        (get_prefer_write(status) != PREFER_WRITE)))) {
                new_status = make_status(get_age(status),
                                         get_read_count(status) + 1,
                                         get_prefer_write(status),
                                         get_wait_write(status));
                if (apr_atomic_cas(lock_, new_status, status) == status) {
                    status_ = new_status;
                    log_try_count("ReadLock", try_count);
                    return;
                }
            } else if (((try_count & TIMEOUT_CHECK_MASK) == 0) &&
                       is_timeout(status)) {
                // タイムアウトしたら age を増やす
                new_status = make_status(get_age(status) + 1, 1,
                                         get_prefer_write(status),
                                         get_wait_write(status));
                if (apr_atomic_cas(lock_, new_status, status) == status) {
                    status_ = new_status;
                    log_try_count("ReadLock(TIME OUT)", try_count);
                    return;
                }
            }

            yield();
        }
    };
    ~ReadLocker()
    {
        apr_size_t try_count;
        apr_uint32_t status;

        try_count = 0;
        while (1) {
            try_count++;

            status = apr_atomic_read(lock_);
            if (get_age(status) != get_age(status_)) {
                // 他のスレッドにてタイムアウト処理がされた
                log_try_count("ReadUnLock(TIME OUT)", try_count);
                return;
            }

            if (apr_atomic_cas(lock_,
                               make_status(get_age(status),
                                           get_read_count(status) - 1,
                                           PREFER_WRITE,
                                           get_wait_write(status)),
                               status) == status) {
                log_try_count("ReadUnLock", try_count);
                return;
            }
        }
    };
    static bool is_locked(apr_atomic_t *lock) {
        return get_read_count(apr_atomic_read(lock)) != NOT_LOCKED;
    };
};

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
