/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: WriteLocker.h 2483 2007-07-03 13:17:30Z svn $
 *****************************************************************************/

#ifndef WRITE_LOCKER_H
#define WRITE_LOCKER_H

#include "Environment.h"

#include "apr_time.h"
#include "apr_global_mutex.h"

#include "ReadWriteLocker.h"

#ifdef DEBUG
#include <iostream>
#endif

using namespace std;

/**
 * @brief 書き込みロックを表すクラス．
 */
class WriteLocker: public ReadWriteLocker
{
public:
    /**
     * コンストラクタです．
     *
     * @param[out] lock ロック変数
     */
    WriteLocker(apr_atomic_t *lock)
      : ReadWriteLocker(lock)
    {
        apr_size_t try_count;
        apr_uint32_t status;
        apr_uint32_t new_status;

        try_count = 0;
        while (1) {
            try_count++;

            status = apr_atomic_read(lock_);
            if (LIKELY(get_read_count(status) == NOT_LOCKED)) {
                new_status = make_status(get_age(status), WRITE_LOCKED);
                if (apr_atomic_cas(lock_, new_status, status) == status) {
                    status_ = new_status;
                    log_try_count("WriteLock", try_count);
                    return;
                }
            } else {
                if (((try_count & TIMEOUT_CHECK_MASK) == 0) &&
                    is_timeout(status)) {
                    // タイムアウトしたら age を増やす
                    new_status = make_status(get_age(status) + 1, WRITE_LOCKED);
                    if (apr_atomic_cas(lock_, new_status, status) == status) {
                        status_ = new_status;
                        log_try_count("WriteLock(TIME OUT)", try_count);
                        return;
                    }
                }

                if (get_wait_write(status) != WAIT_WRITE) {
                    (void)apr_atomic_cas(lock_,
                                         make_status(get_age(status),
                                                     get_read_count(status),
                                                     get_prefer_write(status),
                                                     WAIT_WRITE),
                                         status);
                }
            }

            yield();
        }
    };
    ~WriteLocker()
    {
        apr_size_t try_count;
        apr_uint32_t status;

        try_count = 0;
        while (1) {
            try_count++;

            status = apr_atomic_read(lock_);
            if (get_age(status) != get_age(status_)) {
                // 他のスレッドにてタイムアウト処理がされた
                log_try_count("WriteUnLock(TIME OUT)", try_count);
                return;
            }

            if (apr_atomic_cas(lock_,
                               make_status(get_age(status), NOT_LOCKED,
                                           PREFER_NONE,
                                           get_wait_write(status)),
                               status) == status) {
                log_try_count("WriteUnLock", try_count);
                return;
            }
        }
    };
    static bool is_locked(apr_atomic_t *lock) {
        return get_read_count(apr_atomic_read(lock)) == WRITE_LOCKED;
    };
};

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
