#!/exec/ash
#========================================================#
# Make One Linux General Startup Script 4                #
#  (C)2005-2006 Keicho Kondo <dgel@users.sourceforge.jp> #
#========================================================#
#

### === 1.Initialization === ###
#:Summary
#  Initialize Make One Linux.
#:Output
#  Global Variables
#  Global Functions
#  Initialize (type starting message, and so on)
#  Get Boot Options
#

##< Global Variables >##
moddir="/modules"   # Kernel module directory
sysdir="/MOL"       # MOL System directory
looproot="/MOL/MOL" # Loopbak image file
BASE_FLAG="/MOL/MOLFLAG"
#BASE_FLAG="/etc/mol/MOLTYPE"

PATH="/sbin:/bin:/usr/sbin:/usr/bin"
PATH="${PATH}:/BASE/sbin:/BASE/bin:/BASE/usr/sbin:/BASE/usr/bin"
PATH="${PATH}:${sysdir}/sbin:${sysdir}/bin:${sysdir}/usr/sbin:${sysdir}/usr/bin"
PATH="${PATH}:/MOL+/sbin:/MOL+/bin:/MOL+/usr/sbin:/MOL+/usr/bin"
PATH="${PATH}:/UNIONFS/sbin:/UNIONFS/bin:/UNIONFS/usr/sbin:/UNIONFS/usr/bin"
PATH="${PATH}:/exec:."
export PATH

RESET="]R"       # Reset fb color mode
CRE="
[K"       # Erase to end of line
CLEAR="c"        # Clear and reset Screen
NORMAL="[0;39m"  # Normal Color
RED="[1;31m"     # Failure or error message
GREEN="[1;32m"   # Success message
YELLOW="[1;33m"  # Descriptions
BLUE="[1;34m"    # System message
MAGENTA="[1;35m" # Found devices or drivers
CYAN="[1;36m"    # Questions
WHITE="[1;37m"   # Hint


##< Global Functions >##
#     name: Mount Function
# synopsys: pmount device_file_path mount_point option
#   output: mount given device or return 1.
pmount()
{
	local BUILTIN_FS="iso9660 ext3 vfat ntfs ext2 reiserfs"
	for FOUND_FS in ${BUILTIN_FS}
	do
		if [ -b "${1}" ] && mount -t ${FOUND_FS} ${3} ${1} ${2} >/dev/null 2>&1
		then
			return 0
		fi
	done
	return 1
}

#     name: Check Boot Parameter
# synopsys: checkbootparam parameter
#   output: if success, return 0. if not, return 1.
checkbootparam()
{
	case "${CMDLINE}" in
		*\ ${1}*) return 0 ;;
		${1}*)    return 0 ;;
	esac
	return 1
}

#     name: Get Boot Parameter
# synopsys: getbootparam parameter
getbootparam()
{
	local result=""
	case "${CMDLINE}" in
		*\ ${1}=*) ;;
		${1}=*) ;;
		*) return 1 ;;
	esac
	result="${CMDLINE##*$1=}"
	result="${result%%[     ]*}"
	echo "${result}"
	return 0
}

#     name: Emergency Shell
# synopsys: startash
#   output: interrupts boot process and exec ash shell.
startash()
{
	echo "${CRE}${RED}Can't boot ${MAGENTA}Make One Linux System${RED}, sorry.${NORMAL}"
	echo "${RED}Dropping you to a (very limited) shell.${NORMAL}"
	echo "${RED}Press reset button to quit.${NORMAL}"
	echo

	PS1=" mod # "
	export PS1
	echo "6" >/proc/sys/kernel/printk
	trap 1 2 3 15

	exec /exec/ash
}

#     name: Check Loading Module
# synopsys: chkinsmod module
#   output: load module.
chkinsmod()
{
	if [ -f "${1}" ]
	then
		insmod "${1}" >/dev/null 2>&1
	fi
}

#     name: Ask user for modules
# synopsys: askmodules module_type module ...
#   output: MODULES variable (answered modules).
askmodules()
{
	# probe module type
	local TYPE="${1}"; shift

	# probe available modules
	echo "${BLUE}${TYPE} modules available:${WHITE}"
	local FLAG=""
	local m=""
	for m in "$@"
	do
		if [ -f "${moddir}/scsi/${m}" ]
		then
			[ -z "${FLAG}" ] && { echo -n "	${m}"; FLAG="on"; } || { echo "		${m}"; FLAG=""; }
		fi
	done
	[ -n "${FLAG}" ] && echo

	# asking
	echo "${CYAN}Load ${TYPE} Modules?${NORMAL}"
	echo "${CYAN}[Enter full filename(s) (space-separated), Return for autoprobe, ${WHITE}n${CYAN} for none] ${NORMAL}"
	echo -n "${CYAN}insmod module(s)> ${NORMAL}"
	read MODULES
	case "${MODULES}" in n|N) MODULES=""; ;; y|"") MODULES="$*"; ;; esac
}

#     name: Load the given modules
# synopsys: loadmodules module_type module ...
#   output: load the given modules.
#           SCSI_FOUND variable (if scsi module is loaded).
loadmodules()
{
    # probe module type
	local TYPE="${1}"; shift

	# for interactive
	[ -n "${INTERACTIVE}" ] && echo "6" >/proc/sys/kernel/printk

	# load given modules
	local m=""
	for m in "$@"
	do
		echo -n "${CRE}${BLUE}Probing ${TYPE}... ${MAGENTA}${m}${NORMAL}"
		if [ -f "${moddir}/scsi/${m}" ] && insmod -f ${moddir}/scsi/${m} >/dev/null 2>&1
		then
			case "${TYPE}" in scsi|SCSI) FOUND_SCSI="yes"; ;; esac
		fi
	done
	echo -n "${CRE}"

	# for interactive
	[ -n "${INTERACTIVE}" ] && echo "0" > /proc/sys/kernel/printk
}

#     name: Mount modules floppy disk
# synopsys: mountmodules MODULE_TYPE
#   output: mount module floppy disk on ${moddir}/scsi.
mountmodules()
{
	local TYPE="${1}"
	echo -n "${CRE}${CYAN}Please insert ${TYPE} modules floppy disk and hit Return. ${NORMAL}"
	read a
	echo -n "${CRE}${BLUE}Mounting ${TYPE} modules disk... ${NORMAL}"
	if pmount /dev/fd0 ${moddir}/scsi "-o ro"
	then
		echo "${GREEN}OK.${NORMAL}"
		return 0
	fi
	echo "${RED}NOT FOUND.${NORMAL}"
	return 1
}

#     name: Unmount modules floppy disk
# synopsys: umountmodules MODULE_TYPE
#   output: unmount module floppy disk.
umountmodules()
{
	local TYPE="${1}"
	echo -n "${CRE}${BLUE}Unmounting ${TYPE} modules floppy disk... ${NORMAL}"
	umount ${moddir}/scsi 2>/dev/null
	echo "${GREEN}DONE.${NORMAL}"
}

#     name: Make Empty Directory
# synopsys: make_empty_dir (mol+:)dir
#   output: make empty directory on MOL+ system.
make_empty_dir()
{
	local d=""
	for d in ${1}; do
		[ -d "/MOL+${d}" ] && rm -rf /MOL+${d}
		mkdir -p /MOL+${d}
	done
	return $?
}

#     name: Make Empty File
# synopsys: make_empty_file (mol+:)file
#   output: make empty file on MOL+ system.
make_empty_file()
{
	local f=""
	for f in ${1}; do
		[ -f "/MOL+${f}" ] && rm -f /MOL+${f}
		touch /MOL+${f}
	done
	return $?
}

#     name: Make Read-Only System File Writable
# synopsys: make_writable (mol+:)file
#   output: copy specified file to MOL+ system.
make_writable()
{
	local e=""
	for e in ${1}; do
		[ -e "${sysdir}${e}" ] || continue
		[ -e "/MOL+${e}" ] && rm -rf /MOL+${e}
		cp -a ${sysdir}${e} /MOL+${e%/*}/ >/dev/null 2>&1
	done
	return $?
}

#     name: Change File Access Permissions
# synopsys: change_mode (mol+:)file
#   output: change file access permissions on MOL+ system.
change_mode()
{
	local e=""
	for e in ${2}; do
		[ -n "${1}" -a -e "/MOL+${e}" ] || continue
		chmod ${1} /MOL+${e}
	done
	return $?
}

#     name: Delete symlink
# synopsys: del_link link|dir
#   output: delete specified link or all links under the specified dir.
del_link()
{
	local e=""
	for e in ${1}; do
		[ -e "/MOL+${e}" ] || continue
		find /MOL+${e} -type l -exec rm -f {} \;
	done
	return $?
}

#     name: Fix symlink
# synopsys: fix_link (mol:)source (mol+:)dest
#   output: make symlink from MOL system to MOL+ system.
fix_link()
{
	## argument
	#local TARGET="${1}"
	local LINK="${1}"

	## environment
	# RWDIR is a writable directory.
	local RWDIR="/MOL+"
	# RODIR is a read-only directory.
	local RODIR="${sysdir}"

	## error
	[ -L "${sysdir}${LINK}" ] || return 1
	TARGET="$(readlink -n ${sysdir}${LINK})"

	## start
	cd "${RWDIR}"
	local CURRENT_TARGET="${RODIR}"
	local current=""
	
	for current in $(echo "${LINK}" | sed -e 's@/@ @g')
	do
		# for target
		if [ "${current}" = "${LINK##*/}" ]
		then
			rm -rf "${current}"
			ln -s "${TARGET}" "${current}"
			break
		# for not exist
		elif [ ! -e "${current}" ]
		then
			mkdir -p "${RWDIR}${LINK}"
		# for symlink
		elif [ -L "${current}" ]
		then
			rm -f "${current}"
			mkdir -p "${RWDIR}${LINK}"
		# for directory
		elif [ -d "${current}" ]
		then :
		# else
		else
			echo "Bad Destination for Symlink. failed!"
			return 1
		fi
		cd "${current}"
		local CURRENT_TARGET="${CURRENT_TARGET}/${current}"
		ln -sn ${CURRENT_TARGET}/* . >/dev/null 2>&1
	done
}

#     name: Remove File
# synopsys: remove (mol+:)file
#   output: remove specified file or directory on MOL+ system.
remove()
{
	local e=""
	for e in ${1}; do
		[ -e "/MOL+${e}" ] || continue
		rm -rf /MOL+${e} 2>/dev/null
	done
	return $?
}

#     name: Make Filesystem Hierarchy
# synopsys: mkhierarchy func list
#   output: Make Filesystem Hierarchy for MOL on MOL+ system.
#           func is fix_link(), make_empty_dir(), make_writable(),
#           make_empty_file(), change_mode(), or remove().
#           list is the list file for this function.
mkhierarchy()
{
	eval "$(grep ^${1} ${2})"
	return $?
}


##< Initialize >##
# trap 1 2 3 15 signals
trap "" 1 2 3 15
# clean console i/o
exec >/dev/console </dev/console 2>&1
# set umask 022
umask 022
# type starting message
echo; echo "${RED}Make ${CYAN}One ${MAGENTA}Linux ${WHITE}Start ${YELLOW}!!${NORMAL}"
# mount procfs
mount -t proc /proc /proc
# disable kernel messages
echo "0" >/proc/sys/kernel/printk


##< Get Boot Options >##
[ -f /proc/cmdline ] && CMDLINE="$(cat /proc/cmdline)"
# device on/off parameter
checkbootparam "noscsi" && { NOSCSI="yes"; }
checkbootparam "nousb2" && { NOUSB2="yes"; }
checkbootparam "nofirewire" && { NOFIREWIRE="yes"; }
checkbootparam "noraid" && { NORAID="yes"; }
checkbootparam "nodma" && { NODMA="yes"; }
checkbootparam "addmodules" && { ADDMODULES="yes"; }
# interactive setup
checkbootparam "expert" && { INTERACTIVE="yes"; :>/interactive; }
# base system search path
checkbootparam "from=" && { BOOTFROM="${from}"; }
# toram/hd
checkbootparam "toram=" && { COPYTO="yes"; COPYTORAM="${toram}"; }
checkbootparam "tohd=" && { COPYTO="yes"; COPYTOHD="${tohd}"; }
# "boot=" contains "romode", "rwmode", or "overlay".
checkbootparam "boot=" && { MODE="${boot}"; }
checkbootparam "rwdevice=" && { MOLPLUS_DEV="${rwdevice}"; }
checkbootparam "image=" && { looproot="${image}"; }
# "home=" option contains "skel" or "myconf".
checkbootparam "home=" && { HOME="$home"; }
# for all devices setup
checkbootparam "noaudio" && { NOAUDIO="yes"; }
checkbootparam "nonet" && { NONETWORK="yes"; }
checkbootparam "novideo" && { NOVIDEO="yes"; }
checkbootparam "nodhcp" && { NODHCP="yes"; }
checkbootparam "ipaddr" && { IPADDR="$(getbootparam ipaddr 2>/dev/null)"; }
checkbootparam "netmask" && { NETMASK="$(getbootparam netmask 2>/dev/null)"; }
checkbootparam "gateway" && { GATEWAY="$(getbootparam gateway 2>/dev/null)"; }
checkbootparam "netdev" && { NETDEV="$(getbootparam netdev 2>/dev/null)"; }
# if not eject when halt
checkbootparam "noeject" && { NOEJECT="yes"; }



### === 2.Setup Storage Devices === ###
#:Summary
#  Make Storage Devices available to mount BASE System.
#:Output
#  Load all necessary storage kernel modules.
#  Be available DMA for IDE Device.
#

##< Load all necessary storage kernel modules >##
echo "${BLUE}Loading Modules for Storage Devices${NORMAL}"
## for scsi
chkinsmod ${moddir}/scsi/scsi_mod.*o
chkinsmod ${moddir}/cdrom.*o
chkinsmod ${moddir}/scsi/sr_mod.*.o
## for usb storage
chkinsmod ${moddir}/usbcore.*o
[ -z "${NOUSB2}" ] && chkinsmod ${moddir}/ehci-hcd.*o
chkinsmod ${moddir}/uhci-hcd.*o
chkinsmod ${moddir}/ohci-hcd.*o
chkinsmod ${moddir}/storage/usb-storage.*o
chkinsmod ${moddir}/storage/sd_mod.*o
## for usb keyboard
chkinsmod ${moddir}/usbhid.*o
## for loop
chkinsmod ${moddir}/loop.*o
## for cdrom
chkinsmod ${moddir}/ide-cd.*o
chkinsmod ${moddir}/isofs.*o
## for native language support
chkinsmod ${moddir}/nls_cp437.*o
chkinsmod ${moddir}/nls_utf8.*o
# for some filesystems
chkinsmod ${moddir}/fat.*o
chkinsmod ${moddir}/vfat.*o
chkinsmod ${moddir}/ntfs.*o
chkinsmod ${moddir}/jbd.*o
chkinsmod ${moddir}/ext3.*o
chkinsmod ${moddir}/reiserfs.*o

## check for scsi
if [ -z "${NOSCSI}" ]
then
	echo -n "${CRE}${BLUE}Checking for SCSI...${NORMAL}"
	# probing scsi modules
	if [ -n "${INTERACTIVE}" ]; then
		# Let the user select interactively
		askmodules SCSI $(cd ${moddir}/scsi; echo *.*o)
	else
		# trying to do kind of /proc/pci hardware detection
		PROCPCI="$(cat /proc/pci 2>/dev/null)"
		ISA_SCSI="aha1740.*o aha1542.*o aha152x.*o pas16.*o psi240i.*o qlogicfas.*o qlogicfc.*o seagate.*o t128.*o u14-34f.*o wd7000.*o"
		SCSI_PROBE="${ISA_SCSI}"
		case "${PROCPCI}" in *[Aa][Ii][Cc]-|*[Aa][Hh][Aa]-*) SCSI_PROBE="${SCSI_PROBE} aic7xxx.*o" ;; esac
		case "${PROCPCI}" in *[Bb][Uu][Ss][Ll][Oo][Gg][Ii][Cc]*) SCSI_PROBE="${SCSI_PROBE} BusLogic.*o" ;; esac
		case "${PROCPCI}" in *53[Cc]8*) SCSI_PROBE="${SCSI_PROBE} ncr53c8xx.*o" ;; esac
		case "${PROCPCI}" in *53[Cc]406*) SCSI_PROBE="${SCSI_PROBE} NCR53c406a.*o" ;; esac
		case "${PROCPCI}" in *[Ii][Nn][Ii][Tt][Ii][Oo]\ *|*[Ii][Nn][Ii]-[Aa]100[Uu]2[Ww]*) SCSI_PROBE="${SCSI_PROBE} initio.*o" ;; esac
		case "${PROCPCI}" in *[Mm][Pp][Tt]*[Ss][Cc][Ss][Ii]*) SCSI_PROBE="${SCSI_PROBE} mptscsih.*o" ;; esac
		case "${PROCPCI}" in *[Aa][Dd][Vv][Aa][Nn][Cc][Ee][Dd]\ [Ss][Yy][Ss]*) SCSI_PROBE="${SCSI_PROBE} advansys.*o" ;; esac
		case "${PROCPCI}" in *[Aa][Tt][Pp]8|*[Aa][Ee][Cc]6*) SCSI_PROBE="${SCSI_PROBE} atp870u.*o" ;; esac
		case "${PROCPCI}" in *[Dd][Tt][Cc]*) SCSI_PROBE="${SCSI_PROBE} dtc.*o" ;; esac
		case "${PROCPCI}" in *[Ee][Aa][Tt][Aa]*) SCSI_PROBE="${SCSI_PROBE} eata.*o" ;; esac
		case "${PROCPCI}" in *[Ff]*[Dd][Oo][Mm][Aa][Ii][Nn]*) SCSI_PROBE="${SCSI_PROBE} fdomain.*o" ;; esac
		case "${PROCPCI}" in *[Gg][Dd][Tt]\ *) SCSI_PROBE="${SCSI_PROBE} gdth.*o" ;; esac
		case "${PROCPCI}" in *[Mm][Ee][Gg][Aa][Rr][Aa][Ii][Dd]*) SCSI_PROBE="${SCSI_PROBE} megaraid.*o" ;; esac
		case "${PROCPCI}" in *[Pp][Cc][Ii]-22*) SCSI_PROBE="${SCSI_PROBE} pci2220i.*o" ;; esac
		case "${PROCPCI}" in *[Pp][Cc][Ii]-2000*) SCSI_PROBE="${SCSI_PROBE} pci2000.*o" ;; esac
		case "${PROCPCI}" in *[Qq][Ll][Oo][Gg][Ii][Cc]*) SCSI_PROBE="${SCSI_PROBE} qlogicisp.*o" ;; esac
		case "${PROCPCI}" in *53[Cc]974*) SCSI_PROBE="${SCSI_PROBE} tmscsim.*o" ;; esac
		case "${PROCPCI}" in *[Uu][Ll][Tt][Rr][Aa][Ss][Tt][Oo][Rr]*) SCSI_PROBE="${SCSI_PROBE} ultrastor.*o" ;; esac
		case "${PROCPCI}" in *3[Ww][Aa][Rr][Ee]*) SCSI_PROBE="${SCSI_PROBE} 3w-xxxx.*o" ;; esac
		# these are the autoprobe-safe modules
		MODULES="${SCSI_PROBE}"
	fi

	# loading scsi modules
	[ -n "$MODULES" ] && loadmodules SCSI $MODULES
	echo -n "${CRE}"
fi

## check for ide-raid devices
if [ -z "${NORAID}" ]
then
	echo -n "${CRE}${BLUE}Checking for RAID...${NORMAL}"
	FLAG=""
	[ -z "${FLAG}" ] && chkinsmod ${moddir}/scsi/ataraid.*o && chkinsmod ${moddir}/scsi/silraid.*o && FLAG="on"
	[ -z "${FLAG}" ] && chkinsmod ${moddir}/scsi/medley.*o  && FLAG="on"
	[ -z "${FLAG}" ] && chkinsmod ${moddir}/scsi/pdcraid.*o
	unset FLAG
	echo -n "${CRE}"
fi

## check for firewire
if [ -z "${NOFIREWIRE}" ]
then
	echo -n "${CRE}${BLUE}Checking for Firewire...${NORMAL}"
	FLAG=""
	[ -z "${FLAG}" ] && chkinsmod ${moddir}/storage/ieee1394.*o || FLAG="on"
	[ -z "${FLAG}" ] && chkinsmod ${moddir}/storage/ohci1394.*o || FLAG="on"
	[ -z "${FLAG}" ] && chkinsmod ${moddir}/storage/sbp2.*o sbp2_serialize_io=1
	unset FLAG
	echo -n "${CRE}"
fi

## check for additional modules
if [ -n "${ADDMODULES}" ]
then
	echo -n "${CRE}${BLUE}Checking for Additional Modules...${NORMAL}"
	chkinsmod ${moddir}/add/*.*o
	echo -n "${CRE}"
fi

## check for misc modules in expert mode
if [ -n "${INTERACTIVE}" ]
then
	another=""
	answer=""
	while [ "${answer}" != "n" -a "${answer}" != "N" ]
	do
		echo -n "${CYAN}Do you want to load additional modules from ${another} floppy disk? [${WHITE}Y${CYAN}/n] ${NORMAL}"
		another="another"
		read answer
		case "${answer}" in n*|N*) break; ;; esac
		if mountmodules new; then
			askmodules new $(cd ${moddir}/scsi; echo *.*o)
			[ -n "${MODULES}" ] && loadmodules new ${MODULES}
			umountmodules current
		fi
	done
fi


##< Be available DMA for IDE Device >##
echo "${BLUE}Setting Direct Memory Access${NORMAL}"
for d in $(cd /proc/ide 2>/dev/null && echo hd[a-z]); do
	if [ -d "/proc/ide/${d}" ]; then
		MODEL="$(cat /proc/ide/${d}/model 2>/dev/null)"
		[ -z "${MODEL}" ] && MODEL="[GENERIC IDE DEVICE]"
		if [ -z "${NODMA}" ]; then
			echo " ${GREEN}Enabling DMA acceleration for: ${MAGENTA}${d} 	${YELLOW}[${MODEL}]${NORMAL}"
			echo "using_dma:1" >/proc/ide/${d}/settings
		else
			echo " ${GREEN}Disabling DMA acceleration for: ${MAGENTA}${d} 	${YELLOW}[${MODEL}]${NORMAL}"
			echo "using_dma:0" >/proc/ide/${d}/settings
		fi
	fi
done



### === 3.Mount BASE System === ###
#:Summary
#  Mount BASE System (BASE System is mostly used as ISO Image Filesystem).
#  If TORAM/HD Option is specified, then mount it as BASE System.
#:Output
#  BASE System on /BASE directory.
#  MOL System Type Information (RAW or Image) in MOL_TYPE variable.
#  /etc/mtab.
#
echo "${BLUE}Mounting ${MAGENTA}BASE ${BLUE}System${NORMAL}"

##< Mount BASE System on /BASE directory >##
## probe base system
# set DEVICES variable
loop0="/dev/loop0"
if [ -z "${BOOTFROM}" ]; then
	DEVICES="/dev/hd? /dev/pcd?"
	[ -f "/proc/scsi/scsi" ] && FOUND_SCSI="yes"
	[ -n "${FOUND_SCSI}" ] && DEVICES="/dev/scd? /dev/scd?? ${DEVICES}"
	[ -n "${FOUND_SCSI}" -a -z "${NOSCSI}" ] && DEVICES="${DEVICES} /dev/sd?[1-9] /dev/sd?[1-9][0-9]"
	DEVICES="${DEVICES} /dev/ub?[1-9] /dev/ub?[1-9][0-9] /dev/hd?[1-9] /dev/hd?[1-9][0-9]"
else
	DEVICES="${BOOTFROM}"
fi

# search base system
for d in ${DEVICES}
do
	echo -n "${CRE}${BLUE}Looking for CD/DVD Meida in: ${MAGENTA}${d}${NORMAL}   "
	if pmount "${d}" /BASE "-o ro" >/dev/null 2>&1
	then
		if [ -f "/BASE${BASE_FLAG}" ]
		then
			#MOL_TYPE="$(cat /BASE${BASE_FLAG})"
			. /BASE${BASE_FLAG}
			echo -n "${CRE} ${GREEN}Accessing Make One Linux ${MOLVERSION} ${MOL_TYPE} Filesystem at ${MAGENTA}${d}${GREEN}...${NORMAL}"
			break
		fi
		umount /BASE
	fi
done; echo

# check base system
if [ ! -f "/BASE${BASE_FLAG}" ]
then
	echo "${CRE}${RED}<< Cannot Mount BASE System!! >>${NORMAL}"
	echo
	startash
fi

BASE_DEV="${d}"
BASE_FS="${FOUND_FS}"


## set toram/tohd system
if [ -n "${COPYTO}" ]
then
	echo -n "${CRE} ${GREEN}Copying SYSTEM to "

	mkdir /system
	umount /BASE
	pmount "${BASE_DEV}" /system "-o ro" >/dev/null 2>&1
	FLAG=""
	# for toram
	if [ -n "${COPYTORAM}" ]
	then
		echo -n "${MAGENTA}ramdisk${GREEN}... Please be patient.${NORMAL}"
		mount -t tmpfs -o size="${COPYTORAM}" /dev/shm /BASE >/dev/null 2>&1 || FLAG="on"
		BASE_DEV="/dev/shm"
		BASE_FS="tmpfs"
	# for tohd
	elif [ -n "${COPYTOHD}" ]
	then
		echo -n "${MAGENTA}${COPYTOHD}${GREEN}... Please be patient.${NORMAL}"
		pmount "${COPYTOHD}" /BASE "-o rw" >/dev/null 2>&1 || FLAG="on"
		BASE_DEV="${COPYTOHD}"
		BASE_FS="${FOUND_FS}"
	fi
	echo
	# mount copyto device?
	if [ -z "${FLAG}" ]
	then
		cp -a /system/.[^.]* /BASE/ >/dev/null 2>&1
		cp -a /system/* /BASE/ >/dev/null 2>&1
		mount -o remount,ro "${BASE_DEV}" /BASE >/dev/null 2>&1
	else
		echo "${CRE} ${RED}Copying failed. ${BASE_DEV} is not mountable. ${NORMAL}"
		echo
		startash
	fi
	unset FLAG
	umount /system
	rmdir /system
fi


## fstab and mtab for base system
if [ "${MODE}" = "rwmode" ]
then
	# remount base system for writable
	mount -o remount,rw "${BASE_DEV}" /BASE
	#echo -e "${BASE_DEV}\t\t\t/BASE\t\t${BASE_FS}\t\tdefaults\t\t\t0 0" >> /etc/fstab.in
	echo "${BASE_DEV} /BASE ${BASE_FS} rw 0 0" >> /etc/mtab.in
	#echo "${BASE_DEV} /BASE ${BASE_FS} rw 0 0" >> /etc/mtab
else
	#echo -e "${BASE_DEV}\t\t\t/BASE\t\t${BASE_FS}\t\tro\t\t\t\t0 0" >> /etc/fstab.in
	echo "${BASE_DEV} /BASE ${BASE_FS} ro 0 0" >> /etc/mtab.in
	#echo "${BASE_DEV} /BASE ${BASE_FS} ro 0 0" >> /etc/mtab
fi



### === 4.Mount MOL System === ###
#:Summary
#  Mount MOL System (MOL System is mostly used as Compressed Image Filesystem).
#  2 types of MOL System are available, RAW system or Image(KNOPPIX) system.
#:Output
#  MOL System on /MOL directory.
#  /etc/mtab (Image filesystem mode only).
#
echo "${BLUE}Mounting ${MAGENTA}MOL ${BLUE}System${NORMAL}"

##< Mount MOL System on /MOL directory >##
## raw filesystem mode
if [ "${MOL_TYPE}" == "RAW" ]
then
	for i in bin boot lib opt sbin srv usr; do
		[ -z "${i}" ] && continue
		/BASE/lib/ld-linux.so.2 --library-path /BASE/lib /BASE/bin/rm -rf /${i}
		/BASE/lib/ld-linux.so.2 --library-path /BASE/lib /BASE/bin/ln -snf /BASE/${i} /${i}
	done
	rmdir ${sysdir}
	ln -s /BASE "${sysdir}"
	sysdir="/BASE"
	MOL_FS="$BASE_FS"
	MOL_DEV="$BASE_DEV"

## image(knoppix) filesystem mode
elif [ "${MOL_TYPE}" == "Image" ]
then
	FLAG=""
	MOL_DEV="${loop0}"
	# for squashfs
	if [ -z "${FLAG}" ]
	then
		chkinsmod ${moddir}/squashfs.*o
		losetup "${loop0}" /BASE${looproot} >/dev/null 2>&1
		mount -t squashfs -o ro "${loop0}" "${sysdir}" >/dev/null 2>&1 && FLAG="on"
		MOL_FS="squashfs"
	fi
	# for cloop
	if [ -z "${FLAG}" ]
	then
    	echo "6" > /proc/sys/kernel/printk
		THREAD="clp_thread=1"
		for i in $CMDLINE; do
			case "$i" in chkblk=*|CHKBLK=*) eval $i;; esac
			[ -n "$chkblk" ] && CHKBLK="$chkblk";
			case "$i" in *noclpthread*|*NOCLPTHREAD*) THREAD="" ;; esac
		done
		RBLKS=""
		[ -n "$CHKBLK" ] && RBLKS="rblks=$chkblk"
		chkinsmod ${moddir}/cloop.*o "$RBLKS" "$THREAD" file="/BASE${looproot}"
		losetup "${loop0}" /BASE${looproot} >/dev/null 2>&1
    	echo "0" > /proc/sys/kernel/printk
		mount -t cloop -o ro "${loop0}" "${sysdir}" >/dev/null 2>&1 && FLAG="on"
		MOL_FS="cloop"
	fi
	# for other image
	if [ -z "${FLAG}" ]
	then
		losetup "${loop0}" /BASE${looproot} >/dev/null 2>&1
		pmount "${loop0}" "${sysdir}" "-o ro" >/dev/null 2>&1 && FLAG="on"
		[ "${MODE}" = "rwmode" ] && mount -o remount,rw "${loop0}" "${sysdir}"
		MOL_FS="${FOUND_FS}"
	fi
	# found image?
	if [ -z "${FLAG}" ]
	then
		echo "${CRE}${RED}<< Mount Failed !! >>${NORMAL}"
		echo
		startash
	fi

	# fstab and mtab for mol system
	if [ "${MODE}" = "rwmode" ]
	then
		#echo -e "${MOL_DEV}\t\t\t/MOL\t\t\t${MOL_FS}\t\tdefaults\t\t\t0 0" >> /etc/fstab.in
		echo "${MOL_DEV} /MOL ${MOL_FS} rw 0 0" >> /etc/mtab.in
		#echo "${MOL_DEV} /MOL ${MOL_FS} rw 0 0" >> /etc/mtab
	else
		#echo -e "${MOL_DEV}\t\t\t/MOL\t\t\t${MOL_FS}\t\tro\t\t\t\t0 0" >> /etc/fstab.in
		echo "${MOL_DEV} /MOL ${MOL_FS} ro 0 0" >> /etc/mtab.in
		#echo "${MOL_DEV} /MOL ${MOL_FS} ro 0 0" >> /etc/mtab
	fi

else
	echo "${CRE}${RED}<< Cannot Mount MOL System!! >>${NORMAL}"
	echo
	startash
fi



### === 5.Mount MOL+ System === ###
#:Summary
#  Mount MOL+ System (MOL+ System is mostly used as Readable-Writable Device).
#:Output
#  MOL+ System on /MOL+ directory.
#  /etc/mtab
#
echo "${BLUE}Mounting ${MAGENTA}MOL+ ${BLUE}System${NORMAL}"
cat ${sysdir}/etc/ld.so.cache > /etc/ld.so.cache # copy library cache
FLAG=""

##< Mount MOL+ System on /MOL+ directory >##
## for writable device
if [ -z "${FLAG}" ]
then
	if pmount "${MOLPLUS_DEV}" /MOL+ "-o rw" >/dev/null 2>&1
	then
		FLAG="on"
		MOLPLUS_FS="${FOUND_FS}"
		echo " ${CRE}${BLUE}Created ${YELLOW}/MOL+ ${BLUE}on ${MAGENTA}${MOLPLUS_FS} writable device${NORMAL}"
	fi
fi

## for writable image file
if [ -z "${FLAG}" -a "${MODE}" = "rwmode" ]
then
	MOLPLUS_DEV="/BASE${MOLPLUS_DEV}"
	if pmount "${MOLPLUS_DEV}" /MOL+ "-o loop,rw" >/dev/null 2>&1
	then
		FLAG="on"
		MOLPLUS_FS="${FOUND_FS}"
		echo " ${CRE}${BLUE}Created ${YELLOW}/MOL+ ${BLUE}on ${MAGENTA}${MOLPLUS_FS} writable image file${NORMAL}"
	fi
fi

## for ramdisk
if [ -z "${FLAG}" ]
then
	# memory information
	FOUNDMEM="$(awk '/MemTotal/{print $2}' /proc/meminfo)"
	TOTALMEM="$(awk 'BEGIN{m=0};/MemFree|Cached/{m+=$2};END{print m}' /proc/meminfo)"
	echo "${CRE} ${BLUE}Total memory found: ${YELLOW}${FOUNDMEM} ${BLUE}KB${NORMAL}"

	# calc ram size
	MINSIZE=2000  # Minimum size of additional ram  partitions
	MINLEFT=16000 # At least this much memory minus 30% should remain when home and var are full.
	MAXSIZE="$((${TOTALMEM} - ${MINLEFT}))" # Maximum ramdisk size
	RAMSIZE="$((${TOTALMEM} / 5))"        # Default ramdisk size

	# create ramdisk
	if [ -n "${TOTALMEM}" -a "${TOTALMEM}" -gt "${MINLEFT}" ]
	then
		[ -z "${RAMSIZE}" ] && RAMSIZE=100000
		RAMSIZE=$((${RAMSIZE} * 4))
		echo -n "${CRE} ${BLUE}Creating ${YELLOW}/MOL+ ${BLUE}(dynamic size=${YELLOW}${RAMSIZE}${BLUE}kb) on ${MAGENTA}shared memory${BLUE}...${NORMAL}"
		/bin/mount -t tmpfs -o "size=${RAMSIZE}k" /dev/shm /MOL+
		MOLPLUS_DEV="/dev/shm"
		MOLPLUS_FS="tmpfs"
		echo "${GREEN}Done${NORMAL}"
	elif [ -z "${TOTALMEM}" ]
	then
		echo "${CRE}${RED}<< Can't Get Memory Information. Failed !! >>${NORMAL}"
		echo
		echo "${CRE}${CYAN}Please hit Return to start ash.${NORMAL}"
		read a
		startash
	else
		echo "${CRE}${RED}<< Your Computer does NOT have enough Memory Size. Failed !! >>${NORMAL}"
		echo
		echo "${CRE}  Detected Memory Size : ${TOTALMEM}kb"
		echo "${CRE}  Required Memory Size : ${MINLEFT}kb"
		echo
		echo "${CRE}${CYAN}Please hit Return to start ash.${NORMAL}"
		read a
		startash
	fi
fi
unset FLAG

## fstab and mtab for MOL+ System ##
#echo -e "${MOLPLUS_DEV}\t\t\t/MOL+\t\t\t${MOLPLUS_FS}\t\tdefaults\t\t\t0 0" >> /etc/fstab.in
grep "/MOL+" /etc/mtab >> /etc/mtab.in



### === 6.Setup MOL+ System === ###
#:Summary
#  Make MOL+ System available.
#:Output
#  Structure Filesystem Hierarchy on MOL+ System (Selection System),
#  or Overlay MOL and MOL+ System (Overlay System).
#
chmod 1755 /MOL+ # for sendmail

##< Setup Overlay System >##
if [ "${MODE}" = "overlay" ]
then
	echo "${BLUE}Setup ${MAGENTA}Overlay ${BLUE}System${NORMAL}"
	mkdir /UNIONFS
	modprobe unionfs
	/bin/mount -t unionfs -o dirs=/MOL+=rw:"${sysdir}"=ro none /UNIONFS

	#echo -e "none\t\t\t\t/UNIONFS\t\tunionfs\t\tdefaults\t\t\t0 0" >> /etc/fstab.in
	echo "none /UNIONFS unionfs rw,dirs=/MOL+=rw:${sysdir}=ro,debug=0,err=tryleft,delete=all,copyup=preserve,setattr=left 0 0" >> /etc/mtab.in
	#echo "none /UNIONFS unionfs rw,dirs=/MOL+=rw:${sysdir}=ro,debug=0,err=tryleft,delete=all,copyup=preserve,setattr=left 0 0" >> /etc/mtab

	# for yum bug
	if [ -d /UNIONFS/var/cache/yum ]; then
		rm -rf /UNIONFS/var/cache/yum
		rm -f /MOL+/var/cache/.wh.yum
		mkdir -p /MOL+/yum
		ln -sf /MOL+/yum /MOL+/var/cache/yum
	fi
	# creating some directories which was removed when MOL system created
	mkdir -p /UNIONFS/tmp /UNIONFS/var/tmp
	ln -snf ../var/tmp /UNIONFS/usr/tmp
	# change original files to this
	cp -a /etc/rc.d/rc.sysinit /UNIONFS/etc/rc.d/
	cp -a /etc/rc.d/init.d/halt /UNIONFS/etc/rc.d/init.d/
	[ -f /etc/rc.d/init.d/netfs ] && cp -a /etc/rc.d/init.d/netfs /UNIONFS/etc/rc.d/init.d/
	[ -f /etc/rc.d/init.d/functions ] && cp -a /etc/rc.d/init.d/functions /UNIONFS/etc/rc.d/init.d/
	cp -a /etc/fstab /UNIONFS/etc/
	cp -a /etc/mtab /UNIONFS/etc/
	# change original file to this after finishing linuxrc script
	cp -a /etc/X11/XF86Config-4.in /UNIONFS/etc/X11/
	cp -a /etc/X11/XF86Config.in /UNIONFS/etc/X11/
	cp -a /etc/X11/xorg.conf.in /UNIONFS/etc/X11/
	cp -a /etc/modprobe.conf.in /UNIONFS/etc/
	#cp -a /etc/fstab.in /UNIONFS/etc/
	cp -a /etc/mtab.in /UNIONFS/etc/

	# make links to unionfs
	for i in bin boot etc home lib opt root sbin srv tmp usr var; do
		rm -rf /$i
		${sysdir}/lib/ld-linux.so.2 --library-path ${sysdir}/lib ${sysdir}/bin/ln -snf /UNIONFS/$i /$i
	done

	chmod 1777 /tmp /var/tmp
	chmod 0755 /etc/sysconfig
	chmod 1755 /UNIONFS # for sendmail


##< Setup Selection System >##
else
	echo "${BLUE}Setup ${MAGENTA}Selection ${BLUE}System${NORMAL}"
	## Initialize MOL+ System ##
	# for /home and /root
	echo -n "${CRE} ${BLUE}Creating ${YELLOW}/home ${BLUE}and ${YELLOW}/root ${BLUE}directories...${NORMAL}"
	for u in $(echo ${sysdir}/home/* ${sysdir}/home/.[^.]* ${sysdir}/root)
	do
		u="${u#${sysdir}}"
		if grep "$u" /etc/passwd >/dev/null 2>&1
		then
			USER_D=$(grep "$u" /etc/passwd | head -n 1 | cut -d: -f1 2>/dev/null)
			GROUP_D=$(id -gn ${USER_D} 2>/dev/null)
			mkdir -p -m 0700 /MOL+$u
			case "$u" in */root) chmod 0750 /MOL+$u ;; esac
			if [ "${HOME}" = "skel" ]
			then
				cp -a ${sysdir}/etc/skel/.[^.]* /MOL+$u/ 2>/dev/null
			elif [ "${HOME}" = "myconf" ]
			then
				cp -a ${sysdir}$u/.[^.]* /MOL+$u/ 2>/dev/null
			else
				cp -a ${sysdir}$u/.[^.]* /MOL+$u/ 2>/dev/null
				cp -a ${sysdir}$u/*      /MOL+$u/ 2>/dev/null
			fi
			chown -Rf ${USER_D}:${GROUP_D} /MOL+$u
		else
			ln -s ${sysdir}$u /MOL+$u
		fi
	done
	echo "${GREEN}Done${NORMAL}"

	# for /etc, /var, and /tmp
	echo -n "${CRE} ${BLUE}Creating ${YELLOW}/etc ${BLUE}, ${YELLOW}/var ${BLUE}and ${YELLOW}/tmp ${BLUE}directories...${NORMAL}"
	mv /etc /MOL+/
	cp -aus ${sysdir}/etc/* /MOL+/etc/ 2>/dev/null
	mkdir -p /MOL+/var
	cp -aus ${sysdir}/var/* /MOL+/var/ 2>/dev/null
	mkdir -p -m 1777 /MOL+/tmp
	echo "${GREEN}Done${NORMAL}"

	# for others
	echo -n "${CRE} ${BLUE}Creating ${YELLOW}others ${BLUE}directories...${NORMAL}"
	for i in $(echo ${sysdir}/*); do
		[ -e "${i}" ] || continue
		ln -sn ${i} /MOL+/ 2>/dev/null
	done
	echo "${GREEN}Done${NORMAL}"

	## Make links to MOL+ System ##
	for i in $(echo /MOL+/*); do
		[ -e "${i}" ] || continue
		${sysdir}/lib/ld-linux.so.2 --library-path ${sysdir}/lib ${sysdir}/bin/ln -snf ${i} ${i#/MOL+} 2>/dev/null
	done

	## Create File System Structure ##
	echo -n "${CRE} ${BLUE}Creating File System Structure...${NORMAL}"
	# make writable files and dirs
	mkhierarchy make_writable /etc/mol/hierarchy.list
	# make empty dirs and files
	mkhierarchy make_empty_dir /etc/mol/hierarchy.list
	mkhierarchy make_empty_file /etc/mol/hierarchy.list
	# change mode
	mkhierarchy change_mode /etc/mol/hierarchy.list
	# delete links
	mkhierarchy del_link /etc/mol/hierarchy.list
	# fix broken links
	mkhierarchy fix_link /etc/mol/hierarchy.list
	# remove unnecessary files
	mkhierarchy remove /etc/mol/hierarchy.list
	echo "${GREEN}Done${NORMAL}"
fi



### === 7.Setup All Devices === ###
#:Summary
#  Make available all devices.
#:Output
#  Make Device Files.
#  Exec Auto Setup Mode (or Interactive Setup Mode).
#

##< Make Device Files >##
mount -t sysfs none /sys
mount -o mode=0755 -t tmpfs none /dev
mknod /dev/console c 5 1
mknod /dev/null c 1 3
mknod /dev/zero c 1 5
mkdir /dev/pts
mkdir /dev/shm
#/sbin/udevstart
/sbin/start_udev >/dev/null 2>&1
echo -n "/sbin/hotplug" > /proc/sys/kernel/hotplug
umount /sys

# device check
#DEVNAME="/dev/snd /dev/audio /dev/dsp /dev/mixer"
#for chkdev in ${DEVNAME}
#do
#	if [ ! -e ${chkdev} ]
#	then	
#		/sbin/MAKEDEV -d /dev ${chkdev#/dev/}
#	fi
#done


##< Exec Auto Setup Mode >##
## Executing hwsetup-mol
# check /etc/sysconfig/mol directory
if [ -d /etc/sysconfig/mol ]
then
	for f in $(echo /etc/sysconfig/mol/*)
	do
		for conf in floppy mol mouse netcard sound xserver
		do
			[ "${f#/etc/sysconfig/mol/}" == "${conf}" ] && mv -f /etc/sysconfig/mol/${f} /etc/sysconfig/mol/${f}.backup
		done
	done
else
	mkdir -p /etc/sysconfig/mol
fi

# check boot parameter
[ -n "${NOAUDIO}" ] && { HWSETUP_OPT="-a ${HWSETUP_OPT}"; }
[ -n "${NOSCSI}" ] && { HWSETUP_OPT="-s ${HWSETUP_OPT}"; }
[ -n "${NONETWORK}" ] && { HWSETUP_OPT="-net ${HWSETUP_OPT}"; }
[ -n "${NOVIDEO}" ] && { HWSETUP_OPT="-video ${HWSETUP_OPT}"; }
[ -n "${NORAID}" ] && { HWSETUP_OPT="-raid ${HWSETUP_OPT}"; }

/exec/hwsetup-mol -p ${HWSETUP_OPT}
[ -f /etc/sysconfig/mol/mol ] && . /etc/sysconfig/mol/mol

## XkbLayout and Videocard
if [ -z "${NOVIDEO}" -a -f /etc/X11/xorg.conf ]
then
	case $(grep -v "^#" /etc/X11/xorg.conf | awk '/XkbLayout/{print $3}') in
		*jp*) echo "XKEYBOARD=\"jp\"" >> /etc/sysconfig/mol/keyboard ;;
	esac
	mv /etc/X11/xorg.conf /etc/X11/xorg.conf.backup
	xsetup-mol
else
	echo " ${BLUE}Skipping X Configuration.${NORMAL}"
fi

## Soundcard ##
if [ -z "${NOAUDIO}" ] && [ -n "${SOUND_FULLNAME}" -a -n "${SOUND_DRIVER}" ]
then
	echo " ${GREEN}Soundcard is ${YELLOW}${SOUND_FULLNAME}${GREEN} driver=${MAGENTA}${SOUND_DRIVER}${NORMAL}"
	sed -e 's|@@SOUND@@|'"${SOUND_DRIVER}"'|g' /etc/modprobe.conf.in > /etc/modprobe.conf.tmp
	mv -f /etc/modprobe.conf.tmp /etc/modprobe.conf.in
else
	echo " ${BLUE}Skipping Soundcard detection.${NORMAL}"
	sed -e 's|@@SOUND@@||g' /etc/modprobe.conf.in > /etc/modprobe.conf.tmp
	mv -f /etc/modprobe.conf.tmp /etc/modprobe.conf.in
fi

## NIC ##
if [ -z "${NONETWORK}" ] && [ -n "${NETCARD_FULLNAME}" -a -n "${NETCARD_DRIVER}" ]
then
	NETCARD_DEVICE="$(awk -F: '/eth.:|tr.:/{print $1}' /proc/net/dev | awk -F' ' '{print $1}' 2>/dev/null)"
	echo " ${GREEN}Netcard is ${YELLOW}${NETCARD_FULLNAME}${GREEN} at ${MAGENTA}${NETCARD_DEVICE} ${GREEN}driver=${MAGENTA}${NETCARD_DRIVER}${NORMAL}"
	sed -e 's|@@NET@@|'"alias ${NETCARD_DEVICE} ${NETCARD_DRIVER}"'|g' /etc/modprobe.conf.in > /etc/modprobe.conf.tmp
	mv -f /etc/modprobe.conf.tmp /etc/modprobe.conf.in

	# for Static IP address
	if [ -n "${IPADDR}" ]
	then
		[ -n "${NETMASK}" ] || NETMASK="255.255.255.0"
		[ -n "${GATEWAY}" ] || GATEWAY="${IPADDR%.*}.1"
		[ -n "${NETDEV}" ]  || NETDEV="eth0"
		STATIC_NETDEV="${NETDEV}"
		echo " ${GREEN}Network device ${MAGENTA}${STATIC_NETDEV} ${GREEN}detected, IP is ${YELLOW}${IPADDR}${GREEN}.${NORMAL}"
#-----------------------------------------------------------------
cat > /etc/sysconfig/network-scripts/ifcfg-${STATIC_NETDEV} << EOF
DEVICE=${STATIC_NETDEV}
ONBOOT=yes
BOOTPROTO=static
IPADDR=${IPADDR}
NETMASK=${NETMASK}
EOF
#-----------------------------------------------------------------
		sed "s/GATEWAY=/#GATEWAY=/g" /etc/sysconfig/network > /tmp/network && echo "GATEWAY=${GATEWAY}" >> /tmp/network && mv -f /tmp/network /etc/sysconfig/network
	fi

	# for DHCP
	if [ -n "${NODHCP}" ]
	then
		echo " ${BLUE}Skipping DHCP broadcast/network detection as requested on boot commandline.${NORMAL}"
	else
		for DHCP_NETDEV in ${NETCARD_DEVICE}
		do
			[ "${STATIC_NETDEV}" = "${DHCP_NETDEV}" ] && continue
			echo " ${GREEN}Network device ${MAGENTA}${DHCP_NETDEV} ${GREEN}detected, DHCP broadcasting for IP.${NORMAL}"
#---------------------------------------------------------------
cat > /etc/sysconfig/network-scripts/ifcfg-${DHCP_NETDEV} << EOF
DEVICE=${DHCP_NETDEV}
ONBOOT=yes
BOOTPROTO=dhcp
EOF
#---------------------------------------------------------------
		done
	fi
else
	echo " ${BLUE}Skipping Netcard detection.${NORMAL}"
	sed -e 's|@@NET@@||g' /etc/modprobe.conf.in > /etc/modprobe.conf.tmp
	mv -f /etc/modprobe.conf.tmp /etc/modprobe.conf.in
fi

## Mouse
if [ -n "${MOUSE_FULLNAME}" -o -n "${MOUSE_DEVICE}" ]
then
	echo " ${GREEN}Mouse is ${YELLOW}${MOUSE_FULLNAME}${GREEN} at ${MAGENTA}${MOUSE_DEVICE}${NORMAL}"
fi

## Set /etc/modprobe.conf for MOL
cp -a /etc/modprobe.conf.in /etc/modprobe.conf
for tmpf in $(ls /etc/sysconfig/mol/*.tmp 2>/dev/null)
do
	mv -f ${tmpf%%.tmp} ${tmpf%%.tmp}.mol
	mv -f ${tmpf} ${tmpf%%.tmp}
done



### === 8.Pre-Init === ###
#:Summary
#  Next these commands necesarry executed before init process.
#:Output
#  Enable kernel messages
#  Make utmp and wtmp
#  Make Permanent Home Directory.
#  Make function of ejecting disk when halt.
#  Exec final script.
#  Tell kernel where the real modprobe lives
#  Change root device from /dev/fd0 to /dev/ram0
#  Type Finishing Message.
#

##< Enable kernel messages >##
echo "6" > /proc/sys/kernel/printk

##< Make utmp and wtmp >##
[ -e /var/run/utmp ] && rm -f /var/run/utmp
:> /var/run/utmp
[ -e /var/log/wtmp ] && rm -f /var/log/wtmp
:> /var/log/wtmp

##< Make Permanent Home Directory >##

##< Make function of ejecting disk when halt >##
if [ -z "${NOEJECT}" ]; then
	RWDIR="/MOL+"
	[ "${MODE}" == "overlay" ] && RWDIR="/UNIONFS"
	ln -snf /MOL/lib /lib
	if [ ! -d "${RWDIR}/lib" ]; then
		rm -rf ${RWDIR}/lib 
		[ "${MODE}" == "overlay" ] && rm -f /MOL+/.wh.lib # 0.6.2
		mkdir -p ${RWDIR}/lib
	fi
	ln -sn /MOL/lib/* ${RWDIR}/lib/ 2>/dev/null
	for lib in $(ldd /exec/init | awk '{if($(NF-1)!="dynamic"){print $(NF-1)}}'); do
		[ -e "${RODIR}${lib}" ] || continue
		#RWDIR="/MOL+"
		RODIR="${sysdir}"
	
		## start
		cd "${RWDIR}"
		CURRENT_TARGET="${RODIR}"
		current=""
	
		for current in $(echo "${lib}" | sed -e 's@/@ @g'); do
			# for target
			if [ "${current}" == "${lib##*/}" ]; then
				rm -rf "${current}"
				cp -a "${CURRENT_TARGET}/${current}" .
				if [ -L "${current}" ]; then
					LINKNAME="$(readlink ${current})"
					[ -e "${LINKNAME}" ] && rm -rf "${LINKNAME}"
					cp -a "${CURRENT_TARGET}/${LINKNAME}" ${LINKNAME}
				fi
				break
			# for not exist
			elif [ ! -e "${current}" ]; then
				mkdir -p "${RWDIR}${lib}"
			# for symlink
			elif [ -L "${current}" ]; then
				rm -f "${current}"
				mkdir -p "${RWDIR}${lib}"
			# for directory
			elif [ -d "${current}" ]; then
				:
			# error
			else
				echo "Bad Destination."
			fi
			cd "${current}"
			CURRENT_TARGET="${CURRENT_TARGET}/${current}"
			ln -sn ${CURRENT_TARGET}/* . >/dev/null 2>&1
		done
	done
#	for lib in $(ldd /exec/init | awk '{if($(NF-1)!="dynamic"){print $(NF-1)}}'); do
#		BASENAME="$(basename ${lib})"
#		BASEDIR="${lib%${BASENAME}}"
#		[ -z "${BASENAME}" -o -z "${BASEDIR}" ] && continue
#		[ -L "${RWDIR}${BASEDIR%/}" ] && rm -f ${RWDIR}${BASEDIR%/}
#		mkdir -p ${RWDIR}${BASEDIR}
#		[ -f "${RWDIR}${BASEDIR}${BASENAME}" ] && rm -f ${RWDIR}${BASEDIR}${BASENAME}
#		cp -a /MOL${BASEDIR}${BASENAME} ${RWDIR}${BASEDIR}
#	
#		if [ -L "${lib}" ]; then
#			LINKNAME="$(readlink ${lib})"
#			[ -z "${LINKNAME}" ] && continue
#			[ -f ${RWDIR}${BASEDIR}${LINKNAME} ] && rm -f ${RWDIR}${BASEDIR}${LINKNAME}
#			cp -a /MOL${BASEDIR}${LINKNAME} ${RWDIR}${BASEDIR}
#		fi
#	
#		ln -sn /MOL${BASEDIR}* ${RWDIR}${BASEDIR} 2>/dev/null
#	done
	ln -snf ${RWDIR}/lib /lib
else
	echo "NOEJECT=\"yes\"" >> /molinfo # temporary fix
fi

##< Execute Final Script >##
if [ -x "/exec/final.sh" ]
then
	echo "${BLUE}Executing Final Script${NORMAL}"
	/exec/final.sh
fi

##< Tell kernel where the real modprobe lives >##
echo "/sbin/modprobe" > /proc/sys/kernel/modprobe

##< Change root device from /dev/fd0 to /dev/ram0 >##
echo "0x100" > /proc/sys/kernel/real-root-dev

##< Unmount procfs >##
umount /proc

echo "${CRE}${BLUE}Starting init process.${NORMAL}"
## give control to the init process ##
#rm -f /linuxrc
#rm -rf ${moddir} /exec
exit 0


