package monalipse.views;

import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import monalipse.MonalipsePlugin;
import monalipse.part.CancelableRunner;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSServer;
import monalipse.server.IBoardListContentProvider;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.XMLMemento;
import org.eclipse.ui.part.ViewPart;

public class BoardListView extends ViewPart implements IResourceChangeListener
{
	public static final String ID_BOARD_LIST = BoardListView.class.getName();

	private static final String TAG_SELECTION = "selection";
	private static final String TAG_ELEMENT = "element";
	private static final String TAG_EXPANDED = "expanded";
	private static final String TAG_LABEL = "label";

	private static final String TAG_VERTICAL_POSITION = "verticalPosition";
	private static final String TAG_HORIZONTAL_POSITION = "horizontalPosition";
	private static final String TAG_FILTERS = "filters";
	private static final String TAG_FILTER = "filter";

	private TreeViewer viewer;
	private IBBSServer server;
	private IBoardListContentProvider provider;
	private CancelableRunner cancelable;
//	private DrillDownAdapter drillDownAdapter;
	private IAction reloadAction;
	private IAction abortAction;
//	private Action doubleClickAction;
	private IMemento memento;

	public BoardListView()
	{
	}
	
	public void init(IViewSite site, IMemento memento) throws PartInitException
	{
		super.init(site, memento);
		this.memento = memento;
	}

	public void createPartControl(Composite parent)
	{
		String target = getConfigurationElement().getAttribute("target");
		server = BBSServerManager.getInstanceOf(target, getSite().getWorkbenchWindow());
		viewer = new TreeViewer(parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL);
//		drillDownAdapter = new DrillDownAdapter(viewer);
		setContent();
//		viewer.setSorter(new NameSorter());
		makeActions();
		hookContextMenu();
//		hookDoubleClickAction();
		contributeToActionBars();
		getSite().setSelectionProvider(viewer);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		if(memento != null)
			restoreState(memento);
		memento = null;
	}
	
	public IBBSServer getServer()
	{
		return server;
	}
	
	public String getBoardNameOf(Object board)
	{
		if(provider != null)
			return provider.getText(board);
		else
			return null;
	}

	public void resourceChanged(IResourceChangeEvent event)
	{
		if(provider != null && provider.resourceChanged(event))
		{
			IMemento memento = XMLMemento.createWriteRoot(BoardListView.class.getName());
			saveState(memento);
			viewer.setInput(viewer.getInput());
			restoreState(memento);
		}
	}

	public void saveState(IMemento memento)
	{
		if(viewer == null)
		{
			if(this.memento != null)
				memento.putMemento(this.memento);
			return;
		}
		saveExpansionState(memento);
		saveSelectionState(memento);
	}

	private void saveSelectionState(IMemento memento)
	{
		setElements(memento.createChild(TAG_SELECTION), ((IStructuredSelection)viewer.getSelection()).toArray());
	}

	private void saveExpansionState(IMemento memento)
	{
		setElements(memento.createChild(TAG_EXPANDED), viewer.getVisibleExpandedElements());
	}
	
	private void setElements(IMemento memento, Object[] elements)
	{
		if(provider == null)
			return;

		if(elements.length > 0)
		{
			for (int i = 0; i < elements.length; i++)
			{
				Object o = elements[i];
				IMemento elementMem = memento.createChild(TAG_ELEMENT);
				while(provider.getParent(o) != null)
				{
					IMemento labelMem = elementMem.createChild(TAG_LABEL);
					labelMem.putString(TAG_LABEL, provider.getText(o));
					o = provider.getParent(o);
				}
			}
		}
	}

	private void restoreState(IMemento memento)
	{
		restoreExpansionState(memento);
		restoreSelectionState(memento);
	}

	private void restoreSelectionState(IMemento memento)
	{
		IMemento childMem;
		childMem = memento.getChild(TAG_SELECTION);
		if (childMem != null)
			viewer.setSelection(new StructuredSelection(getElements(childMem)));
	}
	
	private void restoreExpansionState(IMemento memento)
	{
		IMemento childMem = memento.getChild(TAG_EXPANDED);
		if (childMem != null)
			viewer.setExpandedElements(getElements(childMem).toArray());
	}
	
	private List getElements(IMemento memento)
	{
		ArrayList list = new ArrayList();

		IMemento[] elementMem = memento.getChildren(TAG_ELEMENT);
		for (int i = 0; i < elementMem.length; i++)
		{
			IMemento[] labelMem = elementMem[i].getChildren(TAG_LABEL);
			Object[] children = provider.getElements(viewer.getInput());
			Object e = null;
			for(int j = labelMem.length - 1; 0 <= j; j--)
			{
				e = findElement(children, labelMem[j].getString(TAG_LABEL));
				if(e == null)
					break;
				else
					children = provider.getChildren(e);
			}
			if(e != null)
				list.add(e);
		}
		
		return list;
	}
	
	private Object findElement(Object[] list, String label)
	{
		if(list == null)
			return null;
		for(int i = 0; i < list.length; i++)
		{
			if(provider.getText(list[i]).equals(label))
				return list[i];
		}
		return null;
	}

	private void setContent()
	{
		if(server == null)
		{
			provider = null;
			viewer.setContentProvider(new NullTreeContentProvider());
			viewer.setLabelProvider(new LabelProvider());
			viewer.setInput(ResourcesPlugin.getWorkspace());
		}
		else
		{
			provider = server.getBoardListContentProvider();
			viewer.setContentProvider(provider);
			viewer.setLabelProvider(provider);
			viewer.setInput(server.getProject());

			if(provider.getElements(viewer.getInput()).length == 0)
			{
				getSite().getShell().getDisplay().asyncExec(new Runnable()
					{
						public void run()
						{
							getSite().getPage().activate(BoardListView.this);
							reloadAction.run();
						}
					});
			}
		}
	}

	private void hookContextMenu()
	{
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager manager)
				{
					BoardListView.this.fillContextMenu(manager);
				}
			});
		Menu menu = menuMgr.createContextMenu(viewer.getControl());
		viewer.getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, viewer);
	}

	private void contributeToActionBars()
	{
		IActionBars bars = getViewSite().getActionBars();
//		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
		fillLocalStatuBar(bars.getStatusLineManager());
	}

	private void fillLocalStatuBar(IStatusLineManager manager)
	{
		cancelable = new CancelableRunner(manager, getSite().getShell().getDisplay(), abortAction);
	}

	private void fillLocalPullDown(IMenuManager manager)
	{
		manager.add(reloadAction);
		manager.add(new Separator());
		manager.add(abortAction);
	}

	private void fillContextMenu(IMenuManager manager)
	{
		manager.add(reloadAction);
		manager.add(abortAction);
//		manager.add(new Separator());
//		drillDownAdapter.addNavigationActions(manager);
		// Other plug-ins can contribute there actions here
		manager.add(new Separator("Additions"));
	}

	private void fillLocalToolBar(IToolBarManager manager)
	{
		manager.add(reloadAction);
		manager.add(abortAction);
//		manager.add(new Separator());
//		drillDownAdapter.addNavigationActions(manager);
	}

	private void makeActions()
	{
		String iconPath = "icons/"; //$NON-NLS-1$		
		URL installURL = Platform.getPlugin(MonalipsePlugin.PLUGIN_ID).getDescriptor().getInstallURL();

		reloadAction = new Action()
			{
				public void run()
				{
					cancelable.run(cancelable, new IRunnableWithProgress()
						{
							public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException
							{
								if(provider != null)
									provider.updateBoardList();
							}
						});
				}
			};
		reloadAction.setText("Reload");
		reloadAction.setToolTipText("Reload board list");
		try
		{
			reloadAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "refresh_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

		abortAction = new Action()
			{
				public void run()
				{
					cancelable.cancel();
				}
			};
		abortAction.setText("Abort");
		abortAction.setToolTipText("Abort downloading board list");
		abortAction.setEnabled(false);
		try
		{
			abortAction.setImageDescriptor(ImageDescriptor.createFromURL(new URL(installURL, iconPath + "stop_nav.gif"))); //$NON-NLS-1$
		}
		catch (MalformedURLException e)
		{
		}

/*
		doubleClickAction = new Action()
			{
				public void run()
				{
					ISelection selection = viewer.getSelection();
					Object obj = ((IStructuredSelection) selection).getFirstElement();
					showMessage("Double-click detected on " + obj.toString());
				}
			};
*/
	}
/*
	private void hookDoubleClickAction()
	{
		viewer.addDoubleClickListener(new IDoubleClickListener()
		{
			public void doubleClick(DoubleClickEvent event)
			{
				doubleClickAction.run();
			}
		});
	}
*/
	public void setFocus()
	{
		viewer.getControl().setFocus();
	}

	private class NameSorter extends ViewerSorter
	{
	}

}