<?php
/**
 * コントローラの起動を行うためのクラスです。
 * クエリー文字列を解析、前処理を行います。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @license http://www.opensource.org/licenses/bsd-license.php The BSD License
 */
class Moony
{
    /** @var string アクション名称 */
    var $_action_name;

    /** @var array PATH_INFOの値の配列 */
    var $_path_info;

    /**
     * コンストラクタです。
     * 動作設定条件を設定した連想配列を引数として要求します。
     * 設定されるべきキーは以下の通りです。
     * <ul>
     *   <li>action_dir ... アクションクラスの配置ディレクトリ （必須）</li>
     *   <li>template_dir ... テンプレートファイルの配置ディレクトリ （必須）</li>
     *   <li>compile_dir ... コンパイル済みテンプレートファイルの格納先ディレクトリ （必須）</li>
     *   <li>log_dir ... 実行／エラーログの出力先ディレクトリ （必須）</li>
     *   <li>smarty_class ... Smarty.class.phpへのパス （必須）</li>
     *   <li>cache_dir ... テンプレートファイルのキャッシュ格納先ディレクトリ （任意）</li>
     *   <li>config_dir ... テンプレートファイル設定格納先ディレクトリ （任意）</li>
     *   <li>compile_check ... テンプレートファイルの更新をチェックするかどうか （任意）</li>
     *   <li>force_compile ... 常にテンプレートファイルをコンパイルするかどうか （任意）</li>
     *   <li>default_modifiers ... 全てのテンプレートでデフォルトで使用される修飾子（任意）</li>
     *   <li>use_db ... PEAR::DBを利用したデータベースアクセスクラス（Moony_DB）を利用するかどうか（任意）</li>
     * </ul>
     * 引数の連想配列に必要なキーは以下の通りです。
     * 
     * @access public
     * @param array $settings Moonyの動作設定を記述した連想配列
     */
    function Moony($settings)
    {
        // 設定チェック
        if (is_null($settings) || !is_array($settings) ||
            !isset($settings['action_dir']) ||
            !isset($settings['template_dir']) ||
            !isset($settings['compile_dir']) ||
            !isset($settings['log_dir']) ||
            !isset($settings['smarty_class'])) {
            trigger_error('Moony configuration failed.', E_USER_ERROR);
        }

        // 定義
        define('MOONY_ACTION_DIR', $settings['action_dir']);
        define('MOONY_SMARTY_TEMPLATE_DIR', $settings['template_dir']);
        define('MOONY_SMARTY_COMPILE_DIR', $settings['compile_dir']);
        define('MOONY_LOG_DIR', $settings['log_dir']);

        // Smarty
        require_once $settings['smarty_class'];

        // オプション指定
        if (isset($settings['cache_dir'])) {
            define('MOONY_SMARTY_CACHE_DIR', $settings['cache_dir']);
        }
        if (isset($settings['config_dir'])) {
            define('MOONY_SMARTY_CONFIG_DIR', $settings['config_dir']);
        }
        if (isset($settings['compile_check'])) {
            define('MOONY_SMARTY_COMPILE_CHECK', $settings['compile_check']);
        }
        if (isset($settings['force_compile'])) {
            define('MOONY_SMARTY_FORCE_COMPILE', $settings['force_compile']);
        }
        if (isset($settings['default_modifiers'])) {
            define('MOONY_SMARTY_DEFAULT_MODIFIERS', $settings['default_modifiers']);
        }
        if (isset($settings['use_db'])) {
            define('MOONY_USE_DB', $settings['use_db']);
        }
    }

    /**
     * 必要なファイルのincludeを行った後、
     * コントローラを起動します。
     * 
     * @access public
     */
    function activate()
    {
        // 設定
        $root_dir = dirname(__FILE__);
        require_once $root_dir . '/Moony_Config.php';
        require_once $root_dir . '/util/Moony_Error.php';
        require_once $root_dir . '/util/Moony_Logger.php';
        require_once $root_dir . '/util/Moony_Utils.php';
        require_once $root_dir . '/util/Moony_Uploader.php';
        require_once $root_dir . '/util/Moony_Downloader.php';
        require_once $root_dir . '/core/Moony_View.php';
        require_once $root_dir . '/core/Moony_Action.php';
        require_once $root_dir . '/core/Moony_Controller.php';
        require_once $root_dir . '/core/Moony_Request.php';
        require_once $root_dir . '/core/Moony_Response.php';
        require_once $root_dir . '/core/Moony_Session.php';
        if (MOONY_USE_DB === true) {
            require_once $root_dir . '/util/Moony_DB.php';
        }

        // エラーハンドラ設定
        set_error_handler(array('Moony_Error', 'handle'));

        // NULLバイト文字列除去
        $_POST   = Moony_Utils::removeNullString($_POST);
        $_GET    = Moony_Utils::removeNullString($_GET);
        $_FILES  = Moony_Utils::removeNullString($_FILES);
        $_COOKIE = Moony_Utils::removeNullString($_COOKIE);
        $_SERVER = Moony_Utils::removeNullString($_SERVER);

        // PATH_INFO解析
        $this->_parse();

        // 起動
        $controller =& new Moony_Controller();
        $controller->process($this->_action_name, $this->_path_info);
    }

    /**
     * PATH_INFOを解析、
     * アクションに関する設定を取得します。
     * 
     * @access private
     */
    function _parse()
    {
        // PATH_INFO取得
        $path_info = '';
        if (isset($_SERVER['PATH_INFO'])) {
            $path_info = $_SERVER['PATH_INFO'];
        }

        // ディレクトリが指定されていた場合、'index'を末尾に付与
        if (substr($path_info, -1) == '/') {
            $path_info .= 'index';
        }

        // スラッシュ区切りで分割
        $this->_path_info = explode('/', $path_info);

        // 最初の空要素を除去
        array_shift($this->_path_info);

        // 最後の要素から拡張子を除去
        $last = array_pop($this->_path_info);
        $last = Moony_Utils::removeFileExtension($last);
        $this->_path_info[] = $last;

        // アクション名称組み立て
        $this->_action_name = '';
        for ($i = 0; $i < MOONY_EVAL_AS_ACTION_SETTING; $i++) {
            if (isset($this->_path_info[$i])) {
                $this->_action_name .= ucfirst($this->_path_info[$i]);
            }
        }

        // 設定されていない場合、デフォルトアクション
        if (strlen($this->_action_name) == 0) {
            $this->_action_name = MOONY_DEFAULT_ACTION_NAME;
        }
    }
}
?>
