<?php
/**
 * リクエスト情報を格納するクラスです。
 * 
 * @package Moony
 * @subpackage core
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @license http://www.opensource.org/licenses/bsd-license.php The BSD License
 */
class Moony_Request
{
    /** @var array 実際のパラメータを格納する配列 */
    var $_params;

    /** @var object Moony_Sessionのインスタンス */
    var $_session;

    /** @var array PATH_INFOの値の配列 */
    var $_path_info;

    /**
     * コンストラクタです。
     * 
     * @access public
     */
    function Moony_Request()
    {
        $this->_params = ($_SERVER['REQUEST_METHOD'] == 'POST') ? $_POST : $_GET;
        $this->_params = $this->_filter($this->_params);
        if (!ini_get('mbstring.encoding_translation') &&
            MOONY_REQUEST_PARAMETER_CONVERT_ENCODING &&
            MOONY_OUTPUT_ENCODING != MOONY_INTERNAL_ENCODING) {
            // エンコーディングの変換
            mb_convert_variables(MOONY_INTERNAL_ENCODING, MOONY_OUTPUT_ENCODING, &$this->_params);
        }
    }

    /**
     * パラメータ値を取得します。
     * 
     * @access public
     * @param string $name パラメータキー
     * @param mixed $alt パラメータが存在しない場合の代替値（（デフォルト<code>null</code>））
     * @return mixed パラメータ値
     */
    function get($name, $alt = null)
    {
        if (!$this->exists($name)) {
            return $alt;
        }
        return $this->_params[$name];
    }

    /**
     * 全てのパラメータを連想配列として返します。
     *
     * @access public
     * @return array 全てのパラメータの連想配列
     */
    function getAll()
    {
        return $this->_params;
    }

    /**
     * 指定された名称のリクエストパラメータが
     * 存在するかどうかを返します。
     *
     * @access public
     * @param string $name パラメータキー
     * @return boolean 存在する場合<code>true</code>
     */
    function exists($name)
    {
        return isset($this->_params[$name]);
    }

    /**
     * HTTPセッション情報を取得します。
     *
     * @access public
     * @return object HTTPセッション情報
     */
    function getSession()
    {
        if (isset($this->_session)) {
            return $this->_session;
        }
        return null;
    }

    /**
     * HTTPセッション情報を設定します。
     *
     * @access public
     * @param object $session HTTPセッション情報
     */
    function setSession(&$session)
    {
        $this->_session = $session;
    }

    /**
     * 指定されたインデックスのPATH_INFOの値を返します。
     * 例えば、PATH_INFOが/foo/bar.htmlの場合、
     * getPathInfo(0)の戻り値は'foo'、
     * getPathInfo(1)の戻り値は'bar'になります。
     *
     * @access public
     * @param integer $index PATH_INFOの値の配列のインデックス
     * @param string $alt 該当のインデックスが存在しない場合の代替値
     */
    function getPathInfo($index, $alt = null)
    {
        if (isset($this->_path_info[$index])) {
            return $this->_path_info[$index];
        }
        return $alt;
    }

    /**
     * PATH_INFOの値の配列を設定します。
     *
     * @access public
     * @param array $path_info PATH_INFOの値の配列
     */
    function setPathInfo($path_info)
    {
        $this->_path_info = $path_info;
    }

    /**
     * リクエストパラメータを処理します。
     * magic_quotes_gpcがONの場合、エスケープ文字を除去します。
     * また、NULLバイト文字列を除去します。
     *
     * @access private
     * @param mixed $data 処理対象のデータ
     * @return string 処理済の文字列
     */
    function _filter($data)
    {
        if (is_array($data)) {
            return array_map(array(&$this, '_filter'), $data);
        }
        if (get_magic_quotes_gpc()) {
            $data = stripslashes($data);
        }
        return $data;
    }
}
?>
