<?php
/**
 * レスポンス情報を格納するクラスです。
 *
 * @package Moony
 * @subpackage core
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @license http://www.opensource.org/licenses/bsd-license.php The BSD License
 */
class Moony_Response
{
    /** @var array テンプレートに設定する値 */
    var $_attributes = array();

    /** @var string テンプレートファイル名 */
    var $_template;

    /** @var string リダイレクト先のURL */
    var $_redirect_url;

    /** @var boolean キャッシュするかどうか */
    var $_cache = false;

    /** @var ファイルを添付する場合の情報 */
    var $_attachment_file = array();

    /** @var array HTTPヘッダの配列 */
    var $_headers = array();

    /** @var integer HTTPステータス */
    var $_status;

    /**
     * コンストラクタです。
     *
     * @access public
     */
    function Moony_Response()
    {
    }

    /**
     * テンプレートに設定する値を設定します。
     *
     * @access public
     * @param string $key テンプレートの変数名
     * @param mixed $value 割り当てる値
     * @param boolean $escape 値をエスケープ処理するかどうか
     */
    function set($key, $value, $escape = true)
    {
        if ($escape) {
            $value = Moony_Utils::escape($value);
        }
        $this->_attributes[$key] = $value;
    }

    /**
     * テンプレートに設定する値を設定します。
     *
     * @access public
     * @param array $attributes テンプレートの変数名をキーに、割り当てる値を値に持つ連想配列
     * @param boolean $escape 値をエスケープ処理するかどうか
     */
    function setAll($attributes, $escape = true)
    {
        foreach ($attributes as $key => $value) {
            $this->set($key, $value, $escape);
        }
    }

    /**
     * テンプレートに設定する値を全て取得します。
     *
     * @access public
     * @return array テンプレートに設定する全ての値
     */
    function getAttributes()
    {
        return $this->_attributes;
    }

    /**
     * テンプレートファイル名を取得します。
     *
     * @access public
     * @return string テンプレートファイル名
     */
    function getTemplate()
    {
        return $this->_template;
    }

    /**
     * テンプレートファイル名を設定します。
     *
     * @access public
     * @param string $template テンプレートファイル名
     */
    function setTemplate($template)
    {
        $this->_template = $template;
    }

    /**
     * HTTPステータスを設定します。
     *
     * @access public
     * @param integer $status HTTPステータス
     */
    function setStatus($status)
    {
        if (is_int($status)) {
            $this->_status = $status;
        }
    }

    /**
     * 設定されたHTTPステータスを取得します。
     *
     * @access public
     * @return integer HTTPステータス
     */
    function getStatus()
    {
        return $this->_status;
    }

    /**
     * リダイレクト先のURLを取得します。
     *
     * @access public
     * @return string リダイレクト先のURL
     */
    function getRedirectUrl()
    {
        return $this->_redirect_url;
    }

    /**
     * リダイレクト先のURLを設定します。
     *
     * @access public
     * @param string $redirect_url リダイレクト先のURL
     */
    function setRedirectUrl($redirect_url)
    {
        $this->_redirect_url = $redirect_url;
    }

    /**
     * ファイルを添付します。
     * 添付するファイルが指定されている場合、
     * 画面描画処理を行いません。
     *
     * @access public
     * @param string $file 添付するファイルのパス
     * @param string $name 添付するファイルにつける名前
     */
    function setAttachmentFile($file, $name = null)
    {
        $this->_attachment_file[0] = $file;
        $this->_attachment_file[1] = $name;
    }

    /**
     * 添付するファイル名を取得します。
     *
     * @access public
     * @return array 添付するファイル名（[0]=>パス、[1]=>名前）
     */
    function getAttachmentFile()
    {
        return $this->_attachment_file;
    }

    /**
     * HTTPヘッダにヘッダ情報を追加します。
     *
     * @access public
     * @param string $header ヘッダ情報
     */
    function addHeader($header)
    {
        $this->_headers[] = $header;
    }

    /**
     * HTTPヘッダに設定したヘッダ情報を全て取得します。
     *
     * @access public
     * @return array HTTPヘッダに設定したヘッダ情報
     */
    function getHeaders()
    {
        return $this->_headers;
    }

    /**
     * テンプレートファイルのコンパイル結果を
     * キャッシュするかどうかを設定、結果を返します。
     * 引数が指定されない場合は、単純に現在の設定を返します。
     *
     * @access public
     * @param boolean $cache キャッシュするかどうか
     * @return boolean キャッシュするかどうか
     */
    function cache($cache = null)
    {
        if ($cache === true) {
            $this->_cache = true;
        } else if ($cache === false) {
            $this->_cache = false;
        }
        return $this->_cache;
    }
}
?>
