<?php
/**
 * セッションを管理するクラスです。
 * 
 * @package Moony
 * @subpackage core
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @license http://www.opensource.org/licenses/bsd-license.php The BSD License
 */
class Moony_Session
{
    /** @var boolean セッションが開始しているかどうか */
    var $_has_started = false;

    /**
     * コンストラクタです。
     * 
     * @access public
     */
    function Moony_Session()
    {
    }

    /**
     * セッションを開始します。
     * セッションIDを再生成し、
     * 古いセッションファイルを削除します。
     * 
     * @access public
     */
    function start()
    {
        if (MOONY_REGENERATE_SESSION_ID) {
            // 開始、セッション値を保存
            session_start();
            $stored = serialize($_SESSION);

            // 破棄、新セッションID生成
            session_destroy();
            session_id(md5(uniqid(rand(), true)));

            // 再開、セッション値復元
            session_start();
            $_SESSION = unserialize($stored);
        } else {
            // 開始
            session_start();
        }

        $this->_has_started = true;
    }

    /**
     * セッションを終了します。
     * 
     * @access public
     * @return boolean 正常に終了すれば<code>true</code>
     */
    function destroy()
    {
        if (!$this->_has_started) {
            return true;
        }

        $_SESSION = array();

        if (isset($_COOKIE[session_name()])) {
            setcookie(session_name(), '', time() - 42000, '/');
        }

        session_destroy();
        $this->_has_started = false;

        return true;
    }

    /**
     * セッションIDを返します。
     * セッションが開始されていない場合、
     * 無条件に<code>null</code>を返します。
     *
     * @access public
     * @return string セッションID
     */
    function getId()
    {
        if (!$this->_has_started) {
            return null;
        }
        return session_id();
    }

    /**
     * セッションに格納された値を取得します。
     * 指定されたキーに紐付く値が設定されていない場合、
     * 代替値を返します。
     * 
     * @access public
     * @param string $key 格納キー
     * @param mixed $alt 代替値
     * @return mixed 格納値
     */
    function get($key, $alt = null)
    {
        if (!$this->_has_started || !$this->exists($key)) {
            return $alt;
        }
        return $_SESSION[$key];
    }

    /**
     * セッションに値を設定します。
     * 
     * @access public
     * @param string $key 格納キー
     * @param mixed $value 格納値
     * @return boolean 正常に終了すれば<code>true</code>
     */
    function set($key, $value)
    {
        if (!$this->_has_started) {
            return false;
        }
        $_SESSION[$key] = $value;
        return true;
    }

    /**
     * 指定されたキーにセッション値が
     * 格納されているかどうかを返します。
     *
     * @access public
     * @param string $key セッションキー
     * @return boolean 存在する場合<code>true</code>
     */
    function exists($key)
    {
        return isset($_SESSION[$key]);
    }

    /**
     * セッションから値を削除します。
     * 
     * @access public
     * @param string $key 格納キー
     * @return boolean 正常に終了すれば<code>true</code>
     */
    function remove($key)
    {
        if (!$this->_has_started) {
            return false;
        }
        unset($_SESSION[$key]);
        return true;
    }
}
?>
