<?php
require_once 'DB.php';   // PEAR::DB

/**
 * PEAR::DBを利用したシンプルな
 * データベースアクセスを提供するクラスです。
 * 
 * @package Moony
 * @subpackage util
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @license http://www.opensource.org/licenses/bsd-license.php The BSD License
 */
class Moony_DB
{
    /** @var object PEAR::DBのインスタンス */
    var $_db;

    /** @var string データベースの文字エンコーディング */
    var $_db_encoding;

    /** @var boolean トランザクション内かどうか */
    var $_in_trans;

    /**
     * データベースに接続し、PEAR::DBのインスタンスを生成します。
     * 引数としてデータベースの文字エンコーディングが指定された場合、
     * 検索結果の文字エンコーディングを内部エンコーディングに自動的に変換します。
     *
     * @access public
     * @param string $dsn データベース接続情報文字列
     * @param string $db_encoding データベースの文字エンコーディング
     */
    function Moony_DB($dsn, $db_encoding = null)
    {
        $this->_db =& DB::connect($dsn);
        if (DB::isError($this->_db)) {
            Moony_Error::raise($this->_db->getMessage(), __FILE__, __LINE__);
        }
        if ($db_encoding != null) {
            $this->_db_encoding = $db_encoding;
        }
    }

    /**
     * 検索SQLを実行します。
     * 検索結果は項目名をキーとした連想配列の配列として返されます。
     * インスタンス生成時にデータベースの文字エンコーディングが指定されている場合、
     * 検索結果のエンコーディングを内部エンコーディングに自動的に変換します。
     *
     * @access public
     * @param string $sql リプレースホルダーを含むSQL
     * @param array $params リプレースホルダーに設定するパラメータ
     * @param integer $log_level 実行SQLのログ出力レベル
     * @return array 検索結果（項目名をキーとした連想配列の配列）
     */
    function query($sql, $params = null, $log_level = MOONY_LOG_LEVEL_NONE)
    {
        $result = array();
        Moony_Logger::log($log_level, $this->_getLog($sql, $params), __FILE__, __LINE__);
        $result = $this->_db->getAll($sql, $params, DB_FETCHMODE_ASSOC);
        if (DB::isError($result)) {
            if ($this->_in_trans) {
                $this->rollback();
            }
            Moony_Error::raise($result->getMessage(), __FILE__, __LINE__);
        }
        if (isset($this->_db_encoding)) {
            // エンコーディングの変換
            mb_convert_variables(MOONY_INTERNAL_ENCODING, $this->_db_encoding, &$result);
        }
        return $result;
    }

    /**
     * 更新SQLを実行します。
     * 影響を与えたレコード数を結果として返します。
     *
     * @access public
     * @param string $sql リプレースホルダーを含むSQL
     * @param array $params リプレースホルダーに設定するパラメータ
     * @param integer $log_level 実行SQLのログ出力レベル
     * @return integer 影響を与えたレコード数
     */
    function execute($sql, $params = null, $log_level = MOONY_LOG_LEVEL_NONE)
    {
        $result = 0;
        Moony_Logger::log($log_level, $this->_getLog($sql, $params), __FILE__, __LINE__);
        $result = $this->_db->query($sql, $params);
        if (DB::isError($result)) {
            if ($this->_in_trans) {
                $this->rollback();
            }
            Moony_Error::raise($result->getMessage(), __FILE__, __LINE__);
        }
        return $this->_db->affectedRows();
    }

    /**
     * トランザクションを開始します。
     * AUTO COMMITをOFFに設定します。
     *
     * @access public
     */
    function begin()
    {
        $this->_db->autoCommit(false);
        $this->_in_trans = true;
    }

    /**
     * コミット処理を行います。
     *
     * @access public
     */
    function commit()
    {
        $this->_db->commit();
    }

    /**
     * ロールバック処理を行います。
     *
     * @access public
     */
    function rollback()
    {
        $this->_db->rollback();
    }

    /**
     * データベース接続を切断します。
     *
     * @access public
     */
    function disconnect()
    {
        $this->_db->disconnect();
    }

    /**
     * ログ出力用のSQL文字列を取得します。
     *
     * @access private
     * @param string $sql リプレースホルダーを含むSQL
     * @param array $params リプレースホルダーに設定するパラメータ配列
     * @return string ログ出力用のSQL文字列
     */
    function _getLog($sql, $params = null)
    {
        $message = '<Query>=' . $sql;
        if ($params != null) {
            $message .= '; <Parameters>=';
            foreach ($params as $param) {
                $message .= $param . ',';
            }
            $message = rtrim($message, ' ,');
        }
        return $message;
    }
}
?>
