<?php
/**
 * エラーを処理するクラスです。
 * スタティックなメソッドで構成されます。
 * 
 * @package Moony
 * @subpackage util
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @license http://www.opensource.org/licenses/bsd-license.php The BSD License
 */
class Moony_Error
{
    /**
     * 発生したエラーのハンドリングを行います。
     * エラーログを出力、HTTPのステータス500を返し、処理を終了します。
     * 
     * @access public
     * @static
     * @param integer $code エラーコード
     * @param string $message メッセージ
     * @param string $file 呼び出し元のファイル名
     * @param integer $line 呼び出し元の行番号
     */
    function handle($code, $message, $file, $line)
    {
        Moony_Logger::error($message, $file, $line);

        switch ($code) {
            case E_ERROR:
            case E_PARSE:
            case E_NOTICE:
            case E_USER_ERROR:
            case E_USER_WARNING:
            case E_USER_NOTICE:

                Moony_Error::internalServerError();
                exit(1);
        }
    }

    /**
     * エラーを発生させます。
     * エラーコードにはE_USER_ERRORを使用します。
     *
     * @access public
     * @static
     * @param string $message メッセージ
     * @param string $file 呼び出し元のファイル名
     * @param integer $line 呼び出し元の行番号
     */
    function raise($message, $file, $line)
    {
        Moony_Error::handle(E_USER_ERROR, $message, $file, $line);
    }

    /**
     * テンプレートファイルが見つからない場合のエラー処理です。
     * httpのステータスで404を返します。
     * Moony_Config.phpにてエラーメッセージが設定されている場合は
     * そのメッセージをテキストとして表示します。
     *
     * @access public
     * @static
     */
    function notFound()
    {
        Moony_Utils::sendStatus(404);
        if (strlen(MOONY_404_MESSAGE) > 0) {
            header('Content-Type: text/plain');
            echo MOONY_404_MESSAGE;
        }
    }

    /**
     * 動作エラーが発生した場合のエラー処理です。
     * httpのステータスで500を返します。
     * Moony_Config.phpにてエラーメッセージが設定されている場合は
     * そのメッセージをテキストとして表示します。
     *
     * @access public
     * @static
     */
    function internalServerError()
    {
        Moony_Utils::sendStatus(500);
        if (strlen(MOONY_500_MESSAGE) > 0) {
            header('Content-Type: text/plain');
            echo MOONY_500_MESSAGE;
        }
    }
}
?>
