<?php
/**
 * ファイルのアップロードを行うためのクラスです。
 * 
 * @package Moony
 * @subpackage util
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @license http://www.opensource.org/licenses/bsd-license.php The BSD License
 */
class Moony_Uploader
{
    /** @var string 処理対象のファイルに付けられた名前 */
    var $_name;

    /**
     * コンストラクタです。
     *
     * @access public
     * @param string $name 処理対象のファイルに付けられた名前
     */
    function Moony_Uploader($name)
    {
        $this->_name = $name;
    }

    /**
     * クライアント側の元のファイル名を返します。
     * パラメータが配列になっている場合、配列で返されます。
     *
     * @access public
     * @return string|array クライアント側の元のファイル名
     */
    function getOriginalName()
    {
        return $_FILES[$this->_name]['name'];
    }

    /**
     * ファイルのMIMEタイプを返します。
     * パラメータが配列になっている場合、配列で返されます。
     *
     * @access public
     * @return string|array ファイルのMIME型
     */
    function getMimeType()
    {
        return $_FILES[$this->_name]['type'];
    }

    /**
     * ファイルサイズを返します。
     * パラメータが配列になっている場合、配列で返されます。
     *
     * @access public
     * @return integer|array ファイルサイズ
     */
    function getSize()
    {
        return $_FILES[$this->_name]['size'];
    }

    /**
     * サーバ側に保存されているテンポラリファイル名を返します。
     * パラメータが配列になっている場合、配列で返されます。
     *
     * @access public
     * @return integer|array サーバ側に保存されているテンポラリファイル名
     */
    function getTmpName()
    {
        return $_FILES[$this->_name]['tmp_name'];
    }

    /**
     * アップロードのエラーコードを返します。
     * パラメータが配列になっている場合、配列で返されます。
     *
     * @access public
     * @return integer|array アップロードのエラーコード
     */
    function getErrorCode()
    {
        return $_FILES[$this->_name]['error'];
    }

    /**
     * ファイルのアップロード処理を行います。
     * 何かエラーが発生している場合、処理を一切行わずにfalseを返します。
     *
     * @access public
     * @param string $save_dir 保存先のディレクトリ名
     * @param integer $file_mode 保存したファイルのパーミッション（8進数で指定）
     * @param boolean $on_error_remove 処理途中でエラーが発生した場合、アップロード済みのファイルを削除するかどうか
     * @return boolean 全てのアップロードが成功した場合true、そうでない場合false
     */
    function upload($save_dir, $file_mode = 0644, $on_error_remove = false)
    {
        if (!file_exists($save_dir)) {
            Moony_Logger::warn('Directory not found: ' . $save_dir,  __FILE__, __LINE__);
            return false;
        }
        if (!isset($_FILES[$this->_name])) {
            Moony_Logger::warn('File name not found: ' . $this->_name,  __FILE__, __LINE__);
            return false;
        }
        if ($this->hasError()) {
            Moony_Logger::warn('Upload prepare failed: ' . $this->_name,  __FILE__, __LINE__);
            return false;
        }
        if (is_array($_FILES[$this->_name]['name'])) {
            $paths = array();
            foreach ($_FILES[$this->_name]['name'] as $key => $name) {
                $org_name = basename($name);
                $tmp_name = $_FILES[$this->_name]['tmp_name'][$key];
                $save_path = Moony_Utils::buildPath($save_dir, $org_name);
                if (move_uploaded_file($tmp_name, $save_path)) {
                    chmod($save_path, $file_mode);
                    $paths[] = $save_path;
                } else {
                    Moony_Logger::warn('File upload failed: ' . $this->_name,  __FILE__, __LINE__);
                    if ($on_error_remove) {
                        foreach ($paths as $path) {
                            @unlink($path);
                        }
                    }
                    return false;
                }
            }
        } else {
            $org_name = basename($_FILES[$this->_name]['name']);
            $tmp_name = $_FILES[$this->_name]['tmp_name'];
            $save_path = Moony_Utils::buildPath($save_dir, $org_name);
            if (move_uploaded_file($tmp_name, $save_path)) {
                chmod($save_path, $file_mode);
            } else {
                Moony_Logger::warn('File upload failed: ' . $this->_name . "[${key}]",  __FILE__, __LINE__);
                return false;
            }
        }
        return true;
    }

    /**
     * ファイルが画像であることが期待される場合の
     * MIMEタイプをチェックします。
     * テンポラリディレクトリに保存されたファイルに対して
     * getimagesize関数を用いて正確なMIMEタイプを取得、
     * $_FILESに格納されているMIMEタイプと合致するかどうか確認します。
     *
     * @access public
     * @return boolean 画像として正常なMIMEタイプであればtrue
     */
    function checkImageMimeType()
    {
        if (is_array($_FILES[$this->_name]['name'])) {
            foreach ($_FILES[$this->_name]['type'] as $key => $mime_type) {
                if (strlen($_FILES[$this->_name]['tmp_name'][$key]) == 0) {
                    continue;
                }
                $tmp_name = $_FILES[$this->_name]['tmp_name'][$key];
                $attributes = getimagesize($tmp_name);
                if (!$attributes) {
                    // 画像でない
                    return false;
                }
                if ($attributes['mime'] != $mime_type) {
                    // MIMEタイプが一致しない
                    return false;
                }
            }
        } else {
            if (strlen($_FILES[$this->_name]['tmp_name']) == 0) {
                return true;
            }
            $tmp_name = $_FILES[$this->_name]['tmp_name'];
            $attributes = getimagesize($tmp_name);
            if (!$attributes) {
                // 画像でない
                return false;
            }
            if ($attributes['mime'] != $_FILES[$this->_name]['type']) {
                // MIMEタイプが一致しない
                return false;
            }
        }
        return true;
    }

    /**
     * 対象のファイルが間違いなく
     * アップロード準備されているかどうか調べます。
     *
     * @access public
     * @return boolean 何かエラーがある場合true
     */
    function hasError()
    {
        $error_code = $this->getErrorCode();
        if (is_array($error_code)) {
            foreach ($error_code as $code) {
                if ($code != UPLOAD_ERR_OK) {
                    return true;
                }
            }
        } else {
            if ($error_code != UPLOAD_ERR_OK) {
                return true;
            }
        }
        return false;
    }
}
?>
