<?php
/**
 * ユーティリティを提供するクラスです。
 * 
 * @package Moony
 * @subpackage util
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @license http://www.opensource.org/licenses/bsd-license.php The BSD License
 */
class Moony_Utils
{
    /**
     * 指定された変数をhtmlに埋め込むための
     * エスケープ処理を行います。
     *
     * @access public
     * @static
     * @param mixed $src エスケープ対象の変数
     * @return mixed エスケープ済みの変数
     */
    function escape($src)
    {
        if (is_array($src)) {
            return array_map(array('Moony_Utils', 'escape'), $src);
        }
        return htmlspecialchars($src, ENT_QUOTES);
    }

    /**
     * NULLバイト文字列を除去します。
     *
     * @access public
     * @static
     * @param mixed $src 処理対象の変数
     * @return mixed NULLバイトを除去した変数
     */
    function removeNullString($src)
    {
        if (is_array($src)) {
            return array_map(array('Moony_Utils', 'removeNullString'), $src);
        }
        return str_replace("\0", '', $src);
    }

    /**
     * パスを組み立てます。
     *
     * @access public
     * @static
     * @param string $dir ディレクトリ名
     * @param string $file ファイル名
     * @param string $ext ファイルの拡張子
     * @return string それぞれを連結したパス文字列
     */
    function buildPath($dir, $file, $ext = null)
    {
        if (strlen($dir) > 0 && substr($dir, -1, 1) == DIRECTORY_SEPARATOR) {
            $dir = substr($dir, 0, -1);
        }
        if (strlen($file) > 0 && substr($file, 0, 1) == DIRECTORY_SEPARATOR) {
            $file = substr($file, 1);
        }
        if (strlen($dir) > 0) {
            $file = $dir . DIRECTORY_SEPARATOR . $file;
        }
        if (strlen($ext) == 0 || strlen($file) == 0 || substr($file, -1, 1) == DIRECTORY_SEPARATOR) {
            return $file;
        }
        return $file . '.' . $ext;
    }

    /**
     * 指定されたURLにリダイレクトします。
     *
     * @access public
     * @static
     * @param string $url リダイレクト先のURL
     */
    function redirect($url)
    {
        if (!headers_sent()) {
            if (ob_get_length() !== false) {
                ob_end_clean();
            }
            header("Location: ${url}");
        }
    }

    /**
     * 指定されたステータス番号に対応した
     * HTTPのヘッダ情報を送出します。
     *
     * @access public
     * @static
     * @param integer $number 送出するhttpのステータス番号
     */
    function sendStatus($number)
    {
        static $status = array(
            100 => 'HTTP/1.0 100 Continue',
            101 => 'HTTP/1.0 101 Switching Protocols',
            200 => 'HTTP/1.0 200 OK',
            201 => 'HTTP/1.0 201 Created',
            202 => 'HTTP/1.0 202 Accepted',
            203 => 'HTTP/1.0 203 Non-Authoritative Information',
            204 => 'HTTP/1.0 204 No Content',
            205 => 'HTTP/1.0 205 Reset Content',
            206 => 'HTTP/1.0 206 Partial Content',
            300 => 'HTTP/1.0 300 Multiple Choices',
            301 => 'HTTP/1.0 301 Moved Permanently',
            302 => 'HTTP/1.0 302 Found',
            303 => 'HTTP/1.0 303 See Other',
            304 => 'HTTP/1.0 304 Not Modified',
            305 => 'HTTP/1.0 305 Use Proxy',
            307 => 'HTTP/1.0 307 Temporary Redirect',
            400 => 'HTTP/1.0 400 Bad Request',
            401 => 'HTTP/1.0 401 Unauthorized',
            402 => 'HTTP/1.0 402 Payment Required',
            403 => 'HTTP/1.0 403 Forbidden',
            404 => 'HTTP/1.0 404 Not Found',
            405 => 'HTTP/1.0 405 Method Not Allowed',
            406 => 'HTTP/1.0 406 Not Acceptable',
            407 => 'HTTP/1.0 407 Proxy Authentication Required',
            408 => 'HTTP/1.0 408 Request Time-out',
            409 => 'HTTP/1.0 409 Conflict',
            410 => 'HTTP/1.0 410 Gone',
            411 => 'HTTP/1.0 411 Length Required',
            412 => 'HTTP/1.0 412 Precondition Failed',
            413 => 'HTTP/1.0 413 Request Entity Too Large',
            414 => 'HTTP/1.0 414 Request-URI Too Large',
            415 => 'HTTP/1.0 415 Unsupported Media Type',
            416 => 'HTTP/1.0 416 Requested range not satisfiable',
            417 => 'HTTP/1.0 417 Expectation Failed',
            500 => 'HTTP/1.0 500 Internal Server Error',
            501 => 'HTTP/1.0 501 Not Implemented',
            502 => 'HTTP/1.0 502 Bad Gateway',
            503 => 'HTTP/1.0 503 Service Unavailable',
            504 => 'HTTP/1.0 504 Gateway Time-out'
        );
        if (isset($status[$number])) {
            header($status[$number]);
        }
    }

    /**
     * パスからファイル拡張子を取り除いた文字列を返します。
     * パスがDIRECTORY_SEPARATORで終了している場合、何も処理を行いません。
     *
     * @access public
     * @static
     * @param string $path 処理対象のパス
     * @return string 拡張子を取り除いた文字列
     */
    function removeFileExtension($path)
    {
        if (strlen($path) == 0 || substr($path, -1, 1) == DIRECTORY_SEPARATOR) {
            return $path;
        }
        $pos = strrpos($path, '.');
        if ($pos !== false) {
            $path = substr($path, 0, $pos);
        }
        return $path;
    }
}
?>
