<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * アクションクラスに
 * インスタンスインジェクションを行うためのクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Injector
{
    /** @var object アクションクラスのインスタンス */
    var $action;

    /** @var array 設定ファイルから取得したインジェクション定義 */
    var $injections;

    /**
     * コンストラクタです。
     *
     * @access public
     * @param object $action アクションクラスのインスタンス
     */
    function Moony_Injector(&$action)
    {
        $this->action = &$action;
        $this->injections = array(
            'request'  => 'Moony_Request',
            'response' => 'Moony_Response',
            'session'  => 'Moony_Session',
            'flash'    => 'Moony_Flash',
            'smarty'   => 'Smarty'
        );
    }

    /**
     * 設定ファイルを読み込みます。
     *
     * @access public
     * @param array $config_files 設定ファイル名の配列
     */
    function loadAll($config_files)
    {
        foreach ($config_files as $file) {
            // 読み込み
            if ($settings = Moony_Loader::loadIniFile($file)) {
                $injections = Moony_Utils::getArrayValue('Injections', $settings);
                foreach ($injections as $prop_name => $class_name) {
                    $this->injections[$prop_name] = $class_name;
                }
            }
        }
    }

    /**
     * Moony_Requestのインスタンスを
     * アクションクラスにインジェクションします。
     *
     * @access public
     * @param object $request Moony_Requestのインスタンス
     */
    function injectRequest(&$request)
    {
        while (array_search('Moony_Request', $this->injections) !== false) {
            $key = $this->inject('Moony_Request', &$request);
            if ($key !== null) {
                unset($this->injections[$key]);
            }
        }
    }

    /**
     * Moony_Responseのインスタンスを
     * アクションクラスにインジェクションします。
     *
     * @access public
     * @param object $response Moony_Responseのインスタンス
     */
    function injectResponse(&$response)
    {
        while (array_search('Moony_Response', $this->injections) !== false) {
            $key = $this->inject('Moony_Response', &$response);
            if ($key !== null) {
                unset($this->injections[$key]);
            }
        }
    }

    /**
     * Moony_Sessionのインスタンスを
     * アクションクラスにインジェクションします。
     *
     * @access public
     * @param object $session Moony_Sessionのインスタンス
     */
    function injectSession(&$session)
    {
        while (array_search('Moony_Session', $this->injections) !== false) {
            $key = $this->inject('Moony_Session', &$session);
            if ($key !== null) {
                unset($this->injections[$key]);
            }
        }
    }

    /**
     * Moony_Flashのインスタンスを
     * アクションクラスにインジェクションします。
     *
     * @access public
     * @param object $flash Moony_Flashのインスタンス
     */
    function injectFlash(&$flash)
    {
        while (array_search('Moony_Flash', $this->injections) !== false) {
            $key = $this->inject('Moony_Flash', &$flash);
            if ($key !== null) {
                unset($this->injections[$key]);
            }
        }
    }

    /**
     * Smartyのインスタンスを
     * アクションクラスにインジェクションします。
     *
     * @access public
     * @param object $smarty Smartyのインスタンス
     */
    function injectSmarty(&$smarty)
    {
        while (array_search('Smarty', $this->injections) !== false) {
            $key = $this->inject('Smarty', &$smarty);
            if ($key !== null) {
                unset($this->injections[$key]);
            }
        }
    }

    /**
     * Moony_DBのインスタンスを
     * アクションクラスにインジェクションします。
     *
     * @access public
     */
    function injectDB()
    {
        while (array_search('Moony_DB', $this->injections) !== false) {
            $db =& new Moony_DB();
            $key = $this->inject('Moony_DB', &$db);
            if ($key !== null) {
                unset($this->injections[$key]);
            }
        }
    }

    /**
     * Moony_Uploaderのインスタンスを
     * アクションクラスにインジェクションします。
     *
     * @access public
     */
    function injectUploader()
    {
        while (array_search('Moony_Uploader', $this->injections) !== false) {
            $uploader =& new Moony_Uploader();
            $key = $this->inject('Moony_Uploader', &$uploader);
            if ($key !== null) {
                unset($this->injections[$key]);
            }
        }
    }

    /**
     * Moony_Request、Moony_Response、
     * Moony_Session、Smarty以外のクラスのインスタンスを
     * アクションクラスにインジェクションします。
     * パラメータなしのコンストラクタを使用してインスタンスを生成します。
     *
     * @access public
     */
    function injectOthers()
    {
        foreach ($this->injections as $property_name => $class_name) {
            if (Moony_Loader::loadClass($class_name)) {
                $instance =& new $class_name;
                $this->inject($class_name, &$instance);
            }
        }
    }

    /**
     * アクションクラスに指定されたクラスを
     * インスタンスインジェクションします。
     *
     * @access public
     * @param string $class_name クラス名
     * @param object $instance インスタンス
     * @return string インジェクションされたアクションクラスのインスタンス変数名
     */
    function inject($class_name, &$instance)
    {
        if (!is_null($this->action)) {
            $property_name = array_search($class_name, $this->injections);
            if ($property_name !== false) {
                $this->action->$property_name = &$instance;
                return $property_name;
            }
        }
        return null;
    }
}
?>
