<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * ファイル、クラスなどをロードするためのクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Loader
{
    /**
     * 指定された名称のクラスを読み込みます。
     * 定義されているファイル名はクラス名と同一であるものとします。
     * （クラス名がFooの場合、Foo.phpまたはFoo.class.php）
     * include_pathを考慮してクラスファイルを探します。
     * また、Aaa_Bb_Ccccのようなクラス名の場合、
     * Aaa/Bb/Cccc.phpもしくはAaa/BbCccc.class.phpという名称の
     * ファイルがあればそのファイルを読み込みます。
     *
     * @access public
     * @static
     * @param string $class_name 読み込むクラス
     * @param boolean $once include_onceを行う場合true、includeを行う場合false
     * @return boolean 読み込みに成功すればtrue
     */
    function loadClass($class_name)
    {
        if (class_exists($class_name)) {
            // 読み込み済み
            return true;
        }

        $extentions = array('php', 'class.php');

        foreach ($extentions as $extention) {
            $file_name = Moony_Utils::buildPath('', $class_name, $extention);
            if (Moony_Loader::loadFile($file_name)) {
                return true;
            }
        }

        $file_base = str_replace('_', DIRECTORY_SEPARATOR, $class_name);
        foreach ($extentions as $extention) {
            $file_name = Moony_Utils::buildPath('', $file_base, $extention);
            if (Moony_Loader::loadFile($file_name)) {
                if (class_exists($class_name)) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * 指定されたファイルをincludeします。
     * include_pathを考慮してファイルを探します。
     *
     * @access public
     * @static
     * @param string $file_name ファイル名称
     * @param boolean $once include_onceを行う場合true、includeを行う場合false
     * @return boolean 読み込みに成功すればtrue
     */
    function loadFile($file_name, $once = true)
    {
        if (Moony_Utils::isReadableFile($file_name)) {
            if ($once) {
                include_once $file_name;
            } else {
                include $file_name;
            }
            return true;
        }
        return false;
    }

    /**
     * 指定されたiniファイルを読み込みます。
     * セクションを意識して読み込みます。
     *
     * @access public
     * @static
     * @param string $ini_file iniファイル名称
     * @return array|boolean 読み込んだ内容（ファイルが存在しない場合はfalse）
     */
    function loadIniFile($ini_file)
    {
        if (!file_exists($ini_file)) {
            return false;
        }
        // 十分高速なのでキャッシュしない
        return parse_ini_file($ini_file, true);
    }
}
?>
