<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * コントローラクラスです。
 * アクションの設定・起動を行います。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Controller
{
    /**
     * コンストラクタです。
     * 
     * @access public
     */
    function Moony_Controller()
    {
    }

    /**
     * 実際の処理を行います。アクションを実行します。
     * また、指定されたアクションに対応するクラスが存在しない場合は、
     * 無条件にアクション名に紐付くテンプレートを表示させます。
     * 
     * @access public
     * @param array $config_files 設定ファイル名の配列
     */
    function process($config_files)
    {
        // アクション名決定
        $router = new Moony_Router();
        $router->route();
        $action_name = $router->getActionName();

        // Request生成
        $request =& new Moony_Request();

        // Response生成
        $response =& new Moony_Response();
        $response->setTemplate($action_name);

        // Session生成、設定に応じて開始
        $session =& new Moony_Session();
        if (MOONY_SESSION_AUTO_START) {
            $session->start();
        }

        // Flash生成、リストア
        $flash = new Moony_Flash(&$session);
        $flash->restore();

        // View生成
        $view = new Moony_View();

        // plugin登録
        Moony_Plugins::register(&$view->smarty);

        // アクション生成
        $action = null;
        if (Moony_Loader::loadClass($action_name, MOONY_ACTION_DIR)) {
            // インスタンス生成
            $action =& new $action_name;
        }

        // アクションの処理
        if (!is_null($action)) {

            // インジェクション
            $injector =& new Moony_Injector(&$action);
            $injector->loadAll($config_files);
            $injector->injectRequest(&$request);
            $injector->injectResponse(&$response);
            $injector->injectSession(&$session);
            $injector->injectFLash(&$flash);
            $injector->injectSmarty(&$view->smarty);
            $injector->injectDB();
            $injector->injectUploader();
            $injector->injectOthers();

            // フィルタ処理
            if (method_exists($action, 'filter')) {
                $filter =& new Moony_Filter(&$request);
                $action->filter(&$filter);
            }

            // 入力値検証
            if (method_exists($action, 'validate')) {

                // Validator生成
                $validator =& new Moony_Validator(&$request);
                $validator->setTemplate($action_name);

                // 検証
                $action->validate(&$validator);

                // 検証エラー
                if ($validator->hasError()) {

                    // 終了処理
                    $this->processEnd(&$response, &$session, &$flash);

                    // 前回テンプレート設定内容を設定
                    $prev_response = null;
                    if ($session->exists(MOONY_PREVIOUS_RESPONSE_NAME)) {
                        $prev_response = $session->get(MOONY_PREVIOUS_RESPONSE_NAME);
                    }
                    if (!is_null($prev_response)) {
                        // 再度エスケープする必要はない
                        $response->setAll($prev_response->getAll(), false);
                    }

                    // リクエストパラメータをレスポンス情報に設定
                    // エラーメッセージ設定、テンプレート設定
                    $response->setAll($request->getAll());
                    $response->set(MOONY_VALIDATION_ERROR_NAME, $validator->getErrors());
                    $response->setTemplate($validator->getTemplate());

                    // 画面描画
                    $view->render($response);
                    exit;
                }
            }

            // 実処理
            if (method_exists($action, 'execute')) {
                $action->execute();
            }

            // 終了処理
            $this->processEnd(&$response, &$session, &$flash);

            // テキスト出力がなされていた場合、
            // 出力して処理終了
            if (strlen($response->getOutput()) > 0) {
                if (!headers_sent()) {
                    // ヘッダ送出
                    header('Content-Type: text/plain');
                }
                echo $response->getOutput();
                exit;
            }

        } else {
            // 終了処理
            $this->processEnd(&$response, &$session, &$flash);
        }

        // 画面描画
        $view->render($response);
        $session->set(MOONY_PREVIOUS_RESPONSE_NAME, $response);
        exit;
    }

    /**
     * 終了処理を行います。
     * トランザクショントークンの保存を行い、
     * Flash変数の保存を行います。
     *
     * @access public
     * @param object $response Moony_Responseのインスタンス
     * @param object $session Moony_Sessionのインスタンス
     * @param object $flash Moony_Flashのインスタンス
     */
    function processEnd(&$response, &$session, &$flash)
    {
        if ($session->hasStarted()) {
            // トランザクショントークン保存
            $token = md5(uniqid(rand(), true));
            $session->set(MOONY_TRANSACTION_TOKEN_NAME, $token);
            $response->set(MOONY_TRANSACTION_TOKEN_NAME, $token);
        }
        // Flash保存
        $flash->save();
    }
}
?>
