<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * Flash変数をセッションに格納する際のキー
 */
define('MOONY_SESSION_FLASH_KEY', '_moony_flash');

/**
 * Flash変数を管理するクラスです。
 * Flash変数は次のアクションが実行されるまで
 * セッションに保管される一時的な変数です。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Flash
{
    /** @var object Moony_Sessionのインスタンス */
    var $session;

    /** @var array 新たにFlash変数に設定される値 */
    var $values;

    /** @var array Flash変数に保存されていた値 */
    var $stored;

    /**
     * コンストラクタです。
     *
     * @access public
     * @param object $session Moony_Sessionのインスタンス
     */
    function Moony_Flash(&$session)
    {
        $this->session = &$session;
        $this->values = array();
        $this->stored = null;
    }

    /**
     * Flash変数に値を設定、
     * 次のアクションに渡す値を設定します。
     * セッションが開始されていない場合、無条件にfalseを返します。
     * 
     * @access public
     * @param string $key 格納キー
     * @param mixed $value 格納値
     * @return boolean 正常に終了すればtrue
     */
    function set($key, $value)
    {
        if (!$this->session->hasStarted()) {
            return false;
        }
        $this->values[$key] = $value;
        return true;
    }

    /**
     * Flash変数に格納されていた
     * 前のアクションから渡された値を取得します。
     * 指定されたキーに紐付く値が設定されていない場合、代替値を返します。
     * 
     * @access public
     * @param string $key 格納キー
     * @param mixed $alt 代替値
     * @return mixed 格納値
     */
    function get($key, $alt = null)
    {
        return Moony_Utils::getArrayValue($key, $this->stored, $alt);
    }

    /**
     * Flash変数に格納されていた
     * 前のアクションから渡された全ての値を
     * 連想配列として取得します。
     * 
     * @access public
     * @return array 格納されていた全ての値
     */
    function getAll()
    {
        return $this->stored;
    }

    /**
     * 前のアクションから値が
     * Flash変数として渡されているかどうかを調べます。
     *
     * @access public
     * @param string $key 格納キー
     * @return boolean 渡されていればtrue
     */
    function exists($key)
    {
        return array_key_exists($key, $this->stored);
    }

    /**
     * Flash変数に設定、
     * 次のアクションに渡そうとしていた値を削除します。
     * 
     * @access public
     * @param string $key 格納キー
     * @return mixed|boolean 削除した値（削除に失敗した場合false）
     */
    function remove($key)
    {
        if (array_key_exists($key, $this->values)) {
            $value = $this->values[$key];
            unset($this->values[$key]);
            return $value;
        }
        return false;
    }

    /**
     * Flash変数の内容を次のアクションが実行されるまで持ち越します。
     * $keyが指定されない場合、全てのFlash変数の内容が持ち越されます。
     *
     * @access public
     * @param string $key 持ち越すFlash変数のキー
     */
    function keep($key = null)
    {
        if (is_null($key)) {
            foreach ($this->stored as $key => $value) {
                $this->set($key, $value);
            }
        } else {
            $this->set($key, $this->get($key));
        }
    }

    /**
     * セッションからFlash変数の内容を取り出します。
     * 取り出したFlash変数に関しては、セッションからその値を削除します。
     *
     * @access public
     */
    function restore()
    {
        if ($this->session->hasStarted() && $this->session->exists(MOONY_SESSION_FLASH_KEY)) {
            $this->stored = $this->session->get(MOONY_SESSION_FLASH_KEY);
            $this->session->remove(MOONY_SESSION_FLASH_KEY);
        }
    }

    /**
     * Flash変数の内容をセッションに保存します。
     *
     * @access public
     * @return boolean 保存に成功すればtrue
     */
    function save()
    {
        return $this->session->set(MOONY_SESSION_FLASH_KEY, $this->values);
    }
}
?>
