<?php
/**
 * Moony - a simple web application framework
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

/**
 * 検証タイプ: 必須項目
 */
define('MOONY_VALIDATE_REQUIRED', 1);

/**
 * 検証タイプ: フォーマット（正規表現、preg_matchを使用）
 */
define('MOONY_VALIDATE_FORMAT', 2);

/**
 * 検証タイプ: フォーマット（正規表現・マルチバイト対応、mb_eregを使用）
 */
define('MOONY_VALIDATE_FORMAT_MB', 3);

/**
 * 検証タイプ: 数値
 */
define('MOONY_VALIDATE_NUMBER', 4);

/**
 * 検証タイプ: アルファベット
 */
define('MOONY_VALIDATE_ALPHA', 5);

/**
 * 検証タイプ: アルファベットまたは数字
 */
define('MOONY_VALIDATE_ALNUM', 6);

/**
 * 検証タイプ: 数字
 */
define('MOONY_VALIDATE_DIGIT', 7);

/**
 * 検証タイプ: 全角カタカナ
 */
define('MOONY_VALIDATE_KATAKANA', 8);

/**
 * 検証タイプ: 全角カタカナ
 */
define('MOONY_VALIDATE_HIRAGANA', 9);

/**
 * 検証タイプ: Eメール
 */
define('MOONY_VALIDATE_EMAIL', 10);

/**
 * 検証タイプ: URL
 */
define('MOONY_VALIDATE_URL', 11);

/**
 * 検証タイプ: 文字列長
 */
define('MOONY_VALIDATE_LENGTH', 12);

/**
 * 検証タイプ: 範囲
 */
define('MOONY_VALIDATE_RANGE', 13);

/**
 * 検証タイプ: 日付
 */
define('MOONY_VALIDATE_DATE', 14);

/**
 * 検証タイプ: ユーザ独自関数を用いる
 */
define('MOONY_VALIDATE_USERFUNC', 15);

/**
 * 入力値の検証を行うためのクラスです。
 * 
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 * @access public
 */
class Moony_Validator
{
    /** @var object Moony_Requestのインスタンス */
    var $request;

    /** @var string エラー時に表示するテンプレートファイル名 */
    var $template;

    /** @var boolean 妥当かどうかの検証結果 */
    var $is_valid;

    /** @var array エラーメッセージの連想配列 */
    var $errors;

    /**
     * コンストラクタです。
     *
     * @access public
     * @param object $request Moony_Requestのインスタンス
     */
    function Moony_Validator(&$request)
    {
        $this->request = &$request;
        $this->template = null;
        $this->is_valid = true;
        $this->errors = array();
    }

    /**
     * 検証パターンを設定、実際の検証を行います。
     * 検証でエラーが発生した場合、エラーメッセージの追加を行います。
     *
     * @access public
     * @param integer $type 検証タイプ
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     * @param mixed $param1 パラメータ1（MOONY_VALIDATE_FORMATの場合preg_matchで使用可能な正規表現パターン、
     *                                   MOONY_VALIDATE_FORMAT_MBの場合mb_eregで使用可能な正規表現パターン、
     *                                   MOONY_VALIDATE_LENGTHの場合最小長、
     *                                   MOONY_VALIDATE_RANGEの場合最小値、
     *                                   MOONY_VALIDATE_DATEの場合日付フォーマット、
     *                                   MOONY_VALIDATE_USERFUNCの場合コールバック関数）
     * @param mixed $param2 パラメータ2（MOONY_VALIDATE_LENGTHの場合最大長、
     *                                   MOONY_VALIDATE_RANGEの場合最大値、
     *                                   MOONY_VALIDATE_USERFUNCの場合パラメータ配列）
     * @param mixed $param3 パラメータ3（MOONY_VALIDATE_LENGTHの場合最小長/最大長と同値を許容するかどうか、
     *                                   MOONY_VALIDATE_RANGEの場合最小値/最大値と同値を許容するかどうか）
     * @return boolean 検証結果
     */
    function set($type, $name, $message, $param1 = null, $param2 = null, $param3 = null)
    {
        // 入力値
        $value = $this->request->get($name);

        // タイプ別検証
        switch ($type) {
            case MOONY_VALIDATE_REQUIRED :
                if (Moony_Checker::isPresent($value)) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_FORMAT :
                if (Moony_Checker::isRegex($value, $param1)) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_FORMAT_MB :
                if (Moony_Checker::isRegexMb($value, $param1)) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_NUMBER :
                if (Moony_Checker::isValid($value, 'is_numeric')) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_ALPHA :
                if (Moony_Checker::isValid($value, 'ctype_alpha')) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_ALNUM :
                if (Moony_Checker::isValid($value, 'ctype_alnum')) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_DIGIT :
                if (Moony_Checker::isValid($value, 'ctype_digit')) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_KATAKANA :
                $value = Moony_Utils::toArray($value);
                if (MOONY_INTERNAL_ENCODING != 'EUC-JP') {
                    mb_convert_variables('EUC-JP', MOONY_INTERNAL_ENCODING, &$value);
                }
                if (Moony_Checker::isRegex($value, '/^(\xA5[\xA1-\xF6]|[\xA1\xBC])+$/')) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_HIRAGANA :
                $value = Moony_Utils::toArray($value);
                if (MOONY_INTERNAL_ENCODING != 'EUC-JP') {
                    mb_convert_variables('EUC-JP', MOONY_INTERNAL_ENCODING, &$value);
                }
                if (Moony_Checker::isRegex($value, '/^(\xA4[\xA1-\xF3]|[\xA1\xBC])+$/')) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_EMAIL :
                if (Moony_Checker::isRegex($value, '/^[a-z0-9\-\._]+@[a-z0-9]([0-9a-z\-]*[a-z0-9]\.){1,}[a-z]{1,4}$/i')) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_URL :
                if (Moony_Checker::isRegex($value, '/^https?:\/\/[-_.!~*\'()a-z0-9;\/?:\@&=+\$,%#]+$/i')) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_LENGTH :
                $param3 = is_null($param3) ? true : $param3;
                if (Moony_Checker::isLengthBetween($value, $param1, $param2, $param3)) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_RANGE :
                $param3 = is_null($param3) ? true : $param3;
                if (Moony_Checker::isBetween($value, $param1, $param2, $param3)) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_DATE :
                if (Moony_Checker::isDate($value, $param1)) {
                    return true;
                }
                break;
            case MOONY_VALIDATE_USERFUNC :
                if (Moony_Checker::isValid($value, $param1, $param2)) {
                    return true;
                }
                break;
            default :
                Moony_Logger::warn('Validation Type Not Defined: ' . $type, __FILE__, __LINE__);
                return true;
                break;
        }

        $this->addError($name, $message);
        $this->is_valid = false;

        return false;
    }

    /**
     * エラー時に使用する
     * テンプレートファイル名を設定します。
     * 拡張子を除いた名称を設定してください。
     *
     * @access public
     * @param string $template テンプレートファイル名
     */
    function setTemplate($template)
    {
        $this->template = $template;
    }

    /**
     * エラー時に使用する
     * テンプレートファイル名を取得します。
     *
     * @access public
     * @return string テンプレートファイル名
     */
    function getTemplate()
    {
        return $this->template;
    }

    /**
     * エラーメッセージを追加します。
     *
     * @access public
     * @param string $name 入力項目名称
     * @param string $message エラーメッセージ
     */
    function addError($name, $message)
    {
        $this->errors[$name][] = $message;
    }

    /**
     * エラーが存在するかどうかを返します。
     *
     * @access public
     * @return boolean 存在すればtrue
     */
    function hasError()
    {
        return !$this->is_valid || (count($this->errors) > 0);
    }

    /**
     * 登録された全てのエラーメッセージを
     * 連想配列として取得します。
     *
     * @access public
     * @return array 全てのエラーメッセージ
     */
    function getErrors()
    {
        return $this->errors;
    }
}
?>
