<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

define('MOONY_VERSION', '0.12.1');
define('MOONY_ROOT_DIR', dirname(__FILE__));

define('MOONY_ACTION_KEY', 'moony_action');
define('MOONY_VALIDATOR_KEY', 'moony_validator');
define('MOONY_TOKEN_KEY', 'moony_token');

require_once MOONY_ROOT_DIR . '/Moony/Processor.php';

/**
 * Moonyのフロントエンドクラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony
{
    /**
     * アクションクラスを格納するディレクトリ
     * @var string
     */
    var $action_dir = 'actions';

    /**
     * テンプレートファイルを格納するディレクトリ
     * @var string
     */
    var $template_dir = 'templates';

    /**
     * 入力エンコーディング
     * @var string
     */
    var $input_encoding;

    /**
     * 出力エンコーディング
     * @var string
     */
    var $output_encoding;

    /**
     * 内部エンコーディング
     * @var string
     */
    var $internal_encoding;

    /**
     * アクションクラスが見つからない場合に実行されるアクション名
     * @var string
     */
    var $default_action_name;

    /**
     * アクションクラスが見つからない場合のリダイレクト先
     * @var string
     */
    var $default_redirect_url;

    /**
     * セッションを自動的に開始するかどうか
     * @var bool
     */
    var $session_auto_start = true;

    /**
     * 実行するアクションを決定、処理を開始します。
     *
     * @access public
     */
    function start()
    {
        if (is_null($this->internal_encoding)) {
            // 内部エンコーディング
            $this->internal_encoding = mb_internal_encoding();
        }
        $processor =& new Moony_Processor();
        $processor->process($this);
    }

    /**
     * アクションクラスを格納するディレクトリを設定します。
     * 初期設定値は「actions」です。
     *
     * @access public
     * @param string $action_dir アクションクラスを格納するディレクトリ
     */
    function setActionDir($action_dir)
    {
        $this->action_dir = rtrim($action_dir, '/\\');
    }

    /**
     * テンプレートファイルを格納するディレクトリを設定します。
     * 初期設定値は「templates」です。
     *
     * @access public
     * @param string $template_dir テンプレートファイルを格納するディレクトリ
     */
    function setTemplateDir($template_dir)
    {
        $this->template_dir = rtrim($template_dir, '/\\');
    }

    /**
     * 入力エンコーディングを設定します。
     * 設定した場合、リクエストパラメータの値は
     * 内部エンコーディングに変換されます。
     *
     * @access public
     * @param string $input_encoding 入力エンコーディング
     */
    function setInputEncoding($input_encoding)
    {
        $this->input_encoding = $input_encoding;
    }

    /**
     * 出力エンコーディングを設定します。
     * 設定した場合、出力時のデフォルトエンコーディングとして使用されます。
     *
     * @access public
     * @param string $output_encoding 出力エンコーディング
     */
    function setOutputEncoding($output_encoding)
    {
        $this->output_encoding = $output_encoding;
    }

    /**
     * 内部エンコーディングを設定します。
     * 設定されない場合、mb_internal_encodingの戻り値を
     * 内部エンコーディングとして使用します。
     *
     * @access public
     * @param string $internal_encoding 内部エンコーディング
     */
    function setInternalEncoding($internal_encoding)
    {
        $this->internal_encoding = $internal_encoding;
    }

    /**
     * ルーティングの結果、アクションクラスが見つからなかった場合に
     * デフォルトで実行されるアクションを設定します。
     *
     * @access public
     * @param string $action_name アクション名
     */
    function setDefaultAction($action_name)
    {
        $this->default_action_name = $action_name;
    }

    /**
     * ルーティングの結果、アクションクラスが
     * 見つからなかった場合のリダイレクト先を設定します。
     *
     * @access public
     * @param string $action_name アクション名
     */
    function setDefaultRedirect($redirect_url)
    {
        $this->default_redirect_url = $redirect_url;
    }

    /**
     * セッションを自動的に開始するかどうかを設定します。
     * 初期設定値は「true」で、変更しない限り
     * 自動的にセッションが開始されます。
     *
     * @access public
     * @param bool $session_auto_start セッションを自動で開始するかどうか
     */
    function setSessionAutoStart($session_auto_start)
    {
        $this->session_auto_start = (bool) $session_auto_start;
    }
}
?>
