<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

/**
 * アクションクラスの基底クラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Action
{
    /**
     * Moonyのインスタンス（動作設定が格納されている）
     * @var object
     */
    var $moony;

    /**
     * Moony_Requestのインスタンス
     * @var object
     */
    var $request;

    /**
     * Moony_Sessionのインスタンス
     * @var object
     */
    var $session;

    /**
     * PTH_INFOとして渡されたパラメータの配列
     * @var array
     */
    var $args;

    /**
     * テンプレートの処理結果を取得します。
     * テンプレートファイルが存在しない場合、falseを返します。
     * テンプレートの名前空間はアクションクラスと同一なので、
     * アクションクラスのメンバ変数にそのままアクセス可能です。
     * $local_varsが設定されている場合、
     * 値はテンプレート内部で使用できるように展開されます。
     *
     * @access public
     * @param string $template テンプレートファイル名
     * @param array $local_vars テンプレートに展開する値の連想配列
     * @return string|bool テンプレートの処理結果
     */
    function fetch($template, $local_vars = null)
    {
        if (is_null($this->moony->template_dir)) {
            // ディレクトリが定義されていない
            trigger_error('Not defined: template_dir', E_USER_ERROR);
            header('HTTP/1.1 500 Internal Server Error');
            exit;
        }

        if (is_array($local_vars)) {
            // 値を展開
            extract($local_vars, EXTR_SKIP);
        }

        $path = $this->moony->template_dir . DIRECTORY_SEPARATOR . $template;
        if (!file_exists($path)) {
            return false;
        }

        ob_start();
        include $path;
        $result = ob_get_clean();

        return $result;
    }

    /**
     * テンプレートの処理結果を出力します。
     * テンプレートファイルが存在しない場合、
     * HTTPのステータスで404を送出して処理を終了します。
     * テンプレートの名前空間はアクションクラスと同一なので、
     * アクションクラスのメンバ変数にそのままアクセス可能です。
     * $local_varsが設定されている場合、
     * 値はテンプレート内部で使用できるように展開されます。
     * $convert_encodingがtrueで出力エンコーディングが指定されている場合、
     * 出力内容を内部エンコーディングから出力エンコーディングに変換します。
     * 出力エンコーディングはMoony::setOutputEncoding()で設定された値です。
     * 出力エンコーディングを変更する場合、
     * $this->moony->setOutputEncoding('foo')を実行してください。
     *
     * @access public
     * @param string $template テンプレートファイル名
     * @param array $local_vars テンプレートに展開する値の連想配列
     * @param bool $convert_encoding 出力時にエンコーディングを変換するか
     */
    function render($template, $local_vars = null, $convert_encoding = true)
    {
        $result = $this->fetch($template, $local_vars);
        if ($result === false) {
            header('HTTP/1.1 404 Not Found');
            exit;
        }
        $this->renderText($result, $convert_encoding);
    }

    /**
     * 検証エラーが存在する場合、
     * テンプレートの処理結果を出力して処理を終了します。
     * テンプレートファイルが存在しない場合、
     * HTTPのステータスで404を送出して処理を終了します。
     * テンプレートの名前空間はアクションクラスと同一なので、
     * アクションクラスのメンバ変数にそのままアクセス可能です。
     * $local_varsが設定されている場合、
     * 値はテンプレート内部で使用できるように展開されます。
     * $convert_encodingがtrueで出力エンコーディングが指定されている場合、
     * 出力内容を内部エンコーディングから出力エンコーディングに変換します。
     * 出力エンコーディングはMoony::setOutputEncoding()で設定された値です。
     * 出力エンコーディングを変更する場合、
     * $this->moony->setOutputEncoding('foo')を実行してください。
     *
     * @access public
     * @param bool|object $v 検証結果、またはMoony_Validatorのインスタンス
     * @param string $template テンプレートファイル名
     * @param array $local_vars テンプレートに展開する値の連想配列
     * @param bool $convert_encoding 出力時にエンコーディングを変換するか
     */
    function renderOnError($v, $template, $local_vars = null)
    {
        if ($v === false || $v->hasError()) {
            $this->render($template, $local_vars);
            exit;
        }
    }

    /**
     * リダイレクト処理を行います。
     *
     * @access public
     * @param string $url リダイレクト先のURL
     */
    function redirect($url)
    {
        header("Location: {$url}");
    }

    /**
     * 検証エラーが存在する場合、
     * 指定されたURLにリダイレクトして処理を終了します。
     *
     * @access public
     * @param bool|object $v 検証結果、またはMoony_Validatorのインスタンス
     * @param string $url リダイレクト先のURL
     */
    function redirectOnError($v, $url)
    {
        if ($v === false || $v->hasError()) {
            $this->redirect($url);
            exit;
        }
    }

    /**
     * テキストを出力します。
     * $convert_encodingがtrueで出力エンコーディングが指定されている場合、
     * 出力内容を内部エンコーディングから出力エンコーディングに変換します。
     * 出力エンコーディングはMoony::setOutputEncoding()で設定された値です。
     * 出力エンコーディングを変更する場合、
     * $this->moony->setOutputEncoding('foo')を実行してください。
     *
     * @access public
     * @param string $content 出力するテキスト
     * @param bool $convert_encoding エンコーディング変換を行うかどうか
     */
    function renderText($text, $convert_encoding = true)
    {
        if ($convert_encoding && !is_null($this->moony->output_encoding)) {
            $text = mb_convert_encoding($text, $this->moony->output_encoding,
                $this->moony->internal_encoding);
        }
        echo $text;
    }

    /**
     * 任意のファイルを出力します。
     * 必要に応じたHTTPヘッダを送出してください。
     * 指定されたファイルが存在しないか、
     * ファイルを読み込むことができない場合にfalseを返します。
     *
     * @access public
     * @param string $path 表示するファイル
     * @param int $buffer_size ファイル読み込みサイズ
     * @return null|bool 表示できない場合false
     */
    function renderFile($path, $buffer_size = 4096)
    {
        $path = realpath($path);
        if (!file_exists($path) || !is_file($path)) {
            return false;
        }
        $fp = fopen($path, 'rb');
        if ($fp === false) {
            return false;
        }
        while (!feof($fp)) {
            echo fread($fp, $buffer_size);
        }
        fclose($fp);
    }

    /**
     * 任意の画像を出力します。
     * 指定されたファイルが存在しないか、
     * ファイルを読み込むことができない場合にfalseを返します。
     * $mime_typeはHTTPヘッダのContent-Typeの値として送出されます。
     * （使用例: $this->renderImage('/path/to/file', 'image/jpeg');）
     *
     * @access public
     * @param string $path 描画するファイル
     * @param string $mime_type MIMEタイプ（image/jpegなど）
     * @param int $buffer_size ファイル読み込みサイズ
     * @return null|bool ファイルが存在しない場合false
     */
    function renderImage($path, $mime_type, $buffer_size = 4096)
    {
        $path = realpath($path);
        if (!file_exists($path) || !is_file($path)) {
            return false;
        }
        $fp = fopen($path, 'rb');
        if ($fp === false) {
            return false;
        }

        // $mime_typeをそのままContent-Typeとして送出
        header("Content-Type: {$mime_type}");
        header('Content-Disposition: inline');
        header('Content-Length: ' . filesize($path));

        while (!feof($fp)) {
            echo fread($fp, $buffer_size);
        }
        fclose($fp);
    }

    /**
     * トランザクショントークン文字列の妥当性検査を行います。
     * リクエストパラメータとして受け取った値と、
     * セッションに保存していた値が一致するかどうかを調べます。
     * セッションが開始されていない場合、無条件にfalseを返します。
     * リクエストパラメータとして送信されてこない場合、
     * またセッションにトークンが保存されていない場合もfalseを返します。
     *
     * @access public
     * @return bool 妥当なトランザクショントークンかどうか
     */
    function checkToken()
    {
        if (!$this->session->isStarted()) {
            // セッションが開始されていない
            return false;
        }

        // 送信されてきた値とセッションに保存されていた値を取得
        $received = $this->request->get(MOONY_TOKEN_KEY);
        $saved = $this->session->get(MOONY_TOKEN_KEY);

        // セッションに新たな値を保存
        $this->session->set(MOONY_TOKEN_KEY, md5(uniqid(rand(), true)));

        if (is_null($received) || is_null($saved)) {
            // 取得できない場合はfalse
            return false;
        }

        return $received === $saved;
    }

    /**
     * アップロードされたファイルの保存を行います。
     * 保存先のディレクトリは前もって用意されている必要があります。
     * なお、配列を用いた複数ファイルのアップロードには対応していません。
     *
     * @access public
     * @param string $name アップロードファイル名
     * @param string $path 保存先のパス
     * @param int $mode 保存先のファイルのモード
     * @return bool アップロードに成功したかどうか
     */
    function moveFile($name, $path, $mode = 0666)
    {
        $file_info = $this->request->getFile($name);
        if (is_null($file_info)) {
            return false;
        }
        $dir = dirname($path);
        if (!file_exists($dir) || !is_dir($dir)) {
            // ディレクトリが存在しない
            return false;
        }
        if ($file_info['error'] != UPLOAD_ERR_OK) {
            // アップロード時エラー
            return false;
        }
        if (move_uploaded_file($file_info['tmp_name'], $path)) {
            // アップロードファイルの移動、モード変更
            chmod($path, $mode);
            return true;
        }
        return false;
    }

    /**
     * 任意のファイルをダウンロードさせます。
     * $nameが設定されていない場合、元のファイル名を使用します。
     * 指定されたファイルが存在しないか、
     * ファイルを読み込むことができない場合にfalseを返します。
     * 使用例: $this->sendFile('/path/to/foo.zip');
     *
     * @access public
     * @param string $path ダウンロードするファイルのパス
     * @param string $name ダウンロードするファイルにつける名前
     * @param int $buffer_size ファイル読み込み時のバッファサイズ
     * @return null|bool ファイルが存在しない場合false
     */
    function sendFile($path, $name = null, $buffer_size = 4096)
    {
        $path = realpath($path);
        if (!file_exists($path) || !is_file($path)) {
            return false;
        }
        if (is_null($name)) {
            // 名前が指定されていない場合、元のファイル名を使用
            $name = basename($path);
        }
        $fp = fopen($path, 'rb');
        if ($fp === false) {
            return false;
        }

        header('Content-Type: application/octet-stream');
        header("Content-Disposition: attachment; filename=\"{$name}\"");
        header('Content-Length: ' . filesize($path));

        while (!feof($fp)) {
            echo fread($fp, $buffer_size);
        }
        fclose($fp);
    }
}
?>
