<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

define('DS', DIRECTORY_SEPARATOR);

// チェック
error_reporting(E_ALL);
if (php_sapi_name() != 'cli') {
    echo "   [error] Please call this script via CLI\n";
    exit;
}

require_once 'Console/Getopt.php';

$cg =& new Console_Getopt();
$args = $cg->readPHPArgv();
$results = $cg->getopt($args, 'd:feh', array('dir=', 'entrypoint', 'force', 'help'));
if (PEAR::isError($results)) {
    echo '  [error] ' . $results->getMessage() . "\n";
    exit;
}

$opts = $results[0];
$action_names = $results[1];

$base_dir = getcwd();
$force_create = false;
$create_entry_point = false;
$show_help = false;

foreach ($opts as $opt) {
    switch ($opt[0]) {
        case 'd':
        case '--dir':
            // ベースディレクトリ指定
            $base_dir = $opt[1];
            break;
        case 'e':
        case '--entrypoint':
            // エントリポイントを生成
            $create_entry_point = true;
            break;
        case 'f':
        case '--force':
            // 上書き生成
            $force_create = true;
            break;
        case 'h':
        case '--help':
            $show_help = true;
            break;
    }
}

$no_generates = true;

$gen =& new Moony_Generator($base_dir, $force_create);

if ($show_help) {
    // ヘルプを表示して終了
    $gen->showHelp();
    exit;
}

if ($create_entry_point) {
    // エントリポイント生成
    $gen->createEntryPoint();
    $no_generates = false;
}

foreach ($action_names as $action_name) {
    // アクション生成
    $gen->createAction($action_name);
    $no_generates = false;
}

if ($no_generates) {
    // 何も生成しなかったらヘルプを表示
    $gen->showHelp();
}

/**
 * コントローラを自動生成するためのクラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Generator
{
    /**
     * 自動生成の基準となるディレクトリ
     * @var string
     */
    var $_base_dir;

    /**
     * アクションクラスを格納するディレクトリ
     * @var string
     */
    var $_action_dir;

    /**
     * テンプレートファイルを格納するディレクトリ
     * @var string
     */
    var $_template_dir;

    /**
     * エントリポイント
     * @var string
     */
    var $_entry_point;

    /**
     * 上書き生成するかどうか
     * @var bool
     */
    var $_force_create = false;

    /**
     * コンストラクタです。
     *
     * @access public
     * @param string $base_dir 自動生成の基準となるディレクトリ
     * @param bool $force_create 上書き生成するかどうか
     */
    function Moony_Generator($base_dir, $force_create)
    {
        $this->_base_dir = rtrim($base_dir, DS);
        $this->_action_dir = $this->_base_dir . DS . 'actions';
        $this->_template_dir = $this->_base_dir . DS . 'templates';
        $this->_entry_point = $this->_base_dir . DS . 'index.php';
        $this->_force_create = $force_create; 
    }

    /**
     * Moonyを実行するエントリポイントを生成します。
     *
     * @access public
     */
    function createEntryPoint()
    {
        $skeleton = $this->_getEntryPointSkeleton();
        $this->_createFile($this->_entry_point, $skeleton);
    }

    /**
     * アクションクラスとテンプレートファイルを生成します。
     *
     * @access public
     * @param string $action_name アクション名称
     */
    function createAction($action_name)
    {
        $file_name = str_replace('_', DS, $action_name);
        $template_file = strtolower($file_name) . '.php';

        $action_path =
            $this->_action_dir . DS . $file_name . '.php';
        $action_skeleton = $this->_getActionSkeleton();
        $action_skeleton = sprintf($action_skeleton, $action_name,
            str_replace(DS, '/', $template_file));
        $this->_createFile($action_path, $action_skeleton);

        $template_path =
            $this->_template_dir . DS . $template_file;
        $template_skeleton = $this->_getTemplateSkeleton();
        $template_skeleton = sprintf(
            $template_skeleton, $action_name, $action_path, $template_path);
        $this->_createFile($template_path, $template_skeleton);
    }

    /**
     * ヘルプメッセージを表示します。
     *
     * @access public
     */
    function showHelp()
    {
        echo "\n";
        echo "Usage: moony [options] <action_name> ...\n";
        echo "\n";
        echo "'Foo' generates:\n";
        echo "    - actions/Foo.php ......... the action class\n";
        echo "    - templates/foo.php ....... the template file\n";
        echo "'Foo_Bar' generates:\n";
        echo "    - actions/Foo/Bar.php ..... the action class\n";
        echo "    - templates/foo/bar.php ... the template file\n";
        echo "\n";
        echo "More than one action may be specified at once.\n";
        echo "\n";
        echo "Options:\n";
        echo "  -d DIR, --dir=DIR\n";
        echo "      root directory used when generating files\n";
        echo "  -e, --entrypoint\n";
        echo "      generate an entry point (index.php)\n";
        echo "  -f, --force\n";
        echo "      will overwrite newer files\n";
        echo "  -h, --help\n";
        echo "      show this help\n";
        echo "\n";
    }

    /**
     * エントリポイントの雛形を取得します。
     *
     * @access protected
     * @return string エントリポイントの雛形
     */
    function _getEntryPointSkeleton()
    {
        $skeleton = "<?php\n"
                  . "require_once 'Moony.php';\n"
                  . "\n"
                  . "\$moony = new Moony();\n"
                  . "\$moony->setActionDir('actions');\n"
                  . "\$moony->setTemplateDir('templates');\n"
                  . "// \$moony->setInputEncoding('SJIS-win');\n"
                  . "// \$moony->setOutputEncoding('SJIS-win');\n"
                  . "// \$moony->setInternalEncoding('UTF-8');\n"
                  . "// \$moony->setDefaultAction('Index');\n"
                  . "// \$moony->setDefaultRedirect('http://example.com/');\n"
                  . "// \$moony->setSessionAutoStart(true);\n"
                  . "\$moony->start();\n"
                  .  "?>\n";
        return $skeleton;
    }

    /**
     * アクションクラスの雛形を取得します。
     *
     * @access protected
     * @return string コントローラクラスの雛形
     */
    function _getActionSkeleton()
    {
        $skeleton = "<?php\n"
                  . "class %s extends Moony_Action\n"
                  . "{\n"
                  . "    function execute()\n"
                  . "    {\n"
                  . "        \$this->render('%s');\n"
                  . "    }\n"
                  . "}\n"
                  . "?>\n";
        return $skeleton;
    }

    /**
     * テンプレートファイルの雛形を取得します。
     *
     * @access protected
     * @return string テンプレートファイルの雛形
     */
    function _getTemplateSkeleton()
    {
        $skeleton = "<html>\n"
                  . "<head>\n"
                  . "    <title>It worked!</title>\n"
                  . "</head>\n"
                  . "<body>\n"
                  . "<h1>It worked!</h1>\n"
                  . "<p>Congratulations on your action page of Moony, '%s'.</p>\n"
                  . "<h2>working files</h2>\n"
                  . "<ul>\n"
                  . "    <li>action class: %s</li>\n"
                  . "    <li>template file: %s</li>\n"
                  . "</ul>\n"
                  . "</body>\n"
                  . "</html>\n";
        return $skeleton;
    }

    /**
     * ディレクトリを生成します。
     *
     * @access protected
     * @param string $dir ディレクトリ名称
     */
    function _createDir($dir)
    {
        if (!$this->_force_create && file_exists($dir)) {
            echo "   [error] already exists: {$dir}\n";
            return;
        }
        $parent_dir = dirname($dir);
        if (!file_exists($parent_dir)) {
            $this->_createDir($parent_dir);
        }
        mkdir($dir);
        echo "  [create] {$dir}\n";
    }

    /**
     * プレーンテキストファイルを生成します。
     *
     * @access protected
     * @param string $file ファイル名称
     * @param string $content ファイルの内容
     */
    function _createFile($file, $content = null)
    {
        if (!$this->_force_create && file_exists($file)) {
            echo "   [error] already exists: {$file}\n";
            return;
        }

        $dir = dirname($file);
        if (!file_exists($dir)) {
            $this->_createDir($dir);
        }

        $fp = fopen($file, 'w');
        if (!is_null($content)) {
            fwrite($fp, $content);
        }
        fclose($fp);

        echo "  [create] {$file}\n";
    }
}
?>
