<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

require_once MOONY_ROOT_DIR . '/Moony/Action.php';
require_once MOONY_ROOT_DIR . '/Moony/Helpers.php';
require_once MOONY_ROOT_DIR . '/Moony/Request.php';
require_once MOONY_ROOT_DIR . '/Moony/Router.php';
require_once MOONY_ROOT_DIR . '/Moony/Session.php';

/**
 * 処理を遂行するクラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Processor
{
    /**
     * 処理を実行します。
     *
     * @access public
     * @param object $moony Moonyのインスタンス（設定）
     */
    function process(&$moony)
    {
        ob_start();
        header('X-Framework: Moony/' . MOONY_VERSION);

        $request =& new Moony_Request();
        if (!is_null($moony->input_encoding)) {
            // エンコーディング変換
            $request->convertEncoding(
                $moony->input_encoding, $moony->internal_encoding);
        }

        $session =& new Moony_Session();
        if ($moony->session_auto_start) {
            // セッションの開始
            $session->start();
        }

        // ルーティング
        $path_info = $request->getPathInfo();
        $router =& new Moony_Router();
        $mapping = $router->route($moony, $path_info);

        if ($mapping === false) {
            // アクションクラスが見つからない
            if (!is_null($moony->default_redirect_url)) {
                // デフォルトのリダイレクト先
                header('Location: ' . $moony->default_redirect_url);
                exit;
            }
            header('HTTP/1.1 404 Not Found');
            exit;
        }

        // アクション
        $action =& $mapping['action'];
        $GLOBALS[MOONY_ACTION_KEY] =& $action;

        // アクションの設定
        $action->moony =& $moony;
        $action->request =& $request;
        $action->session =& $session;
        $action->args = $mapping['args'];
        $this->_extractParams($action, $request);

        // アクションの実行
        $this->_invokeAction($action, $moony);

        ob_end_flush();
    }

    /**
     * リクエストパラメータを展開、
     * アクションのプロパティとして設定します。
     * 設定されるのは予めプロパティが用意されていた場合だけです。
     * '_'で始まるパラメータは設定されません。
     * また、次の名称のパラメータは設定されません（Moonyで使用するため）。
     *
     * @access protected
     * @param object $action アクションクラスのインスタンス
     * @param object $request Moony_Request
     */
    function _extractParams(&$action, &$request)
    {
        $params = $request->get();
        $prop_keys = array_keys(get_object_vars($action));
        foreach ($params as $name => $value) {
            if ($name[0] != '_' && $name != 'moony' && $name != 'request' &&
                $name != 'session' && $name != 'args' &&
                in_array($name, $prop_keys)) {
                $action->$name = $value;
            }
        }
    }

    /**
     * アクションクラスの処理を実行します。
     *
     * @access protected
     * @param object $action アクションクラスのインスタンス
     * @param object $moony Moonyのインスタンス（設定）
     */
    function _invokeAction(&$action, &$moony)
    {
        // action::prepare()
        if (method_exists($action, 'prepare')) {
            $action->prepare();
        }

        // action::convert()
        if (method_exists($action, 'convert')) {
            require_once MOONY_ROOT_DIR . '/Moony/Converter.php';
            $converter =& new Moony_Converter($moony->internal_encoding);
            $action->convert($converter);
        }

        // action::validate()
        if (method_exists($action, 'validate')) {
            require_once MOONY_ROOT_DIR . '/Moony/Validator.php';
            $validator =& new Moony_Validator($moony->internal_encoding);
            $GLOBALS[MOONY_VALIDATOR_KEY] =& $validator;
            $action->validate($validator);
        }

        // action::execute()
        if (method_exists($action, 'execute')) {
            $action->execute();
        }
    }
}
?>
