<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

/**
 * リクエストパラメータを取り扱うクラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Request
{
    /**
     * リクエストパラメータの連想配列
     * @var array
     */
    var $_params;

    /**
     * アップロードファイルに関する情報の連想配列
     * @var array
     */
    var $_files;

    /**
     * クエリーストリング
     * @var string
     */
    var $_query_string;

    /**
     * PATH_INFO文字列
     * @var string
     */
    var $_path_info;

    /**
     * リクエストメソッド
     * @var string
     */
    var $_method;

    /**
     * コンストラクタです。
     *
     * @access public
     */
    function Moony_Request()
    {
        $this->_params = $this->_filter($_POST + $_GET);
        $this->_files = $this->_filter($_FILES);

        $query_string = $this->getEnv('QUERY_STRING');
        if ($query_string !== false) {
            $this->_query_string = rawurldecode($query_string);
        }

        $path_info = $this->getEnv('PATH_INFO');
        if ($path_info !== false) {
            $this->_path_info = $path_info;
        }

        $request_method = $this->getEnv('REQUEST_METHOD');
        if ($request_method !== false) {
            $this->_method = $request_method;
        }
    }

    /**
     * リクエストパラメータの値を取得します。
     * 該当する値が存在しない場合、$defaultを返します。
     * $nameが設定されていない場合、全ての値を連想配列で返します。
     *
     * @access public
     * @param string $name リクエストパラメータの名称
     * @param mixed $default 値が存在しない場合のデフォルト値
     * @rerurn string|array|mixed リクエストパラメータの値
     */
    function get($name = null, $default = null)
    {
        if (is_null($name)) {
            return $this->_params;
        }
        if ($this->exists($name)) {
            return $this->_params[$name];
        }
        return $default;
    }

    /**
     * 指定された名称のリクエストパラメータが
     * 存在するかどうか調べます。
     *
     * @access public
     * @param string $name リクエストパラメータの名称
     * @return bool 存在するかどうか
     */
    function exists($name)
    {
        return isset($this->_params[$name]);
    }

    /**
     * アップロードファイルに関する情報を取得します。
     * $_FILESの該当する要素が返されます。
     *
     * @access public
     * @param string $name アップロードファイルの名称
     * @return array アップロードファイルの情報
     */
    function getFile($name)
    {
        if (isset($this->_files[$name])) {
            return $this->_files[$name];
        }
    }

    /**
     * クエリーストリングを取得します。
     * 存在しない場合、nullが返されます。
     *
     * @access public
     * @return string|null クエリーストリング値
     */
    function getQueryString()
    {
        return $this->_query_string;
    }

    /**
     * PATH_INFO文字列を取得します。
     * 存在しない場合、nullが返されます。
     *
     * @access public
     * @return string|null クエリーストリング値
     */
    function getPathInfo()
    {
        return $this->_path_info;
    }

    /**
     * 環境変数を取得します。
     * NULL文字が混入していた場合、除去されます。
     * 該当する環境変数が存在しない場合、falseを返します。
     *
     * @access public
     * @param string $name 環境変数の名称
     * @return string|bool 環境変数の値
     */
    function getEnv($name)
    {
        if (isset($_SERVER[$name])) {
            return str_replace("\0", '', $_SERVER[$name]);
        }
        return false;
    }

    /**
     * リクエストヘッダを取得します。
     * $nameにX-Requested-Withが指定された場合、
     * $_SERVER['HTTP_X_REQUESTED_WITH']を参照します。
     * NULL文字が混入していた場合、除去されます。
     * 該当するヘッダが存在しない場合、falseを返します。
     *
     * @access public
     * @param string $name リクエストヘッダの名称
     * @return string|bool リクエストヘッダの値
     */
    function getHeader($name)
    {
        $key = 'HTTP_' . str_replace('-', '_', strtoupper($name));
        $header = $this->getEnv($key);
        if ($header !== false) {
            return $header;
        }
        return false;
    }

    /**
     * POSTで送信されたかどうかを返します。
     *
     * @access public
     * @return bool POSTで送信されたかどうか
     */
    function isPost()
    {
        return (strtolower($this->_method) == 'post');
    }

    /**
     * GETで送信されたかどうかを返します。
     *
     * @access public
     * @return bool GETで送信されたかどうか
     */
    function isGet()
    {
        return (strtolower($this->_method) == 'get');
    }

    /**
     * PUTで送信されたかどうかを返します。
     *
     * @access public
     * @return bool PUTで送信されたかどうか
     */
    function isPut()
    {
        return (strtolower($this->_method) == 'put');
    }

    /**
     * DELETEで送信されたかどうかを返します。
     *
     * @access public
     * @return bool DELETEで送信されたかどうか
     */
    function isDelete()
    {
        return (strtolower($this->_method) == 'delete');
    }

    /**
     * HEADで送信されたかどうかを返します。
     *
     * @access public
     * @return bool HEADで送信されたかどうか
     */
    function isHead()
    {
        return (strtolower($this->_method) == 'head');
    }

    /**
     * SSLで通信がなされたかどうかを返します。
     *
     * @access public
     * @return bool SSLで通信がなされたかどうか
     */
    function isSsl()
    {
        return isset($_SERVER['HTTPS']);
    }

    /**
     * Prototype.jsで設定されるX-Requested-Withヘッダを参照、
     * XMLHttpRequestでアクセスされたかどうかを返します。
     *
     * @access public
     * @return bool XMLHttpRequestでアクセスされたかどうか
     */
    function isXmlHttpRequest()
    {
        $x_requested_with = $this->getHeader('X-Requested-With');
        if ($x_requested_with === 'XMLHttpRequest') {
            return true;
        }
        return false;
    }

    /**
     * isXmlHttpRequestメソッドのエイリアスです。
     *
     * @access public
     * @return bool XMLHttpRequestでアクセスされたかどうか
     */
    function isXhr()
    {
        return $this->isXmlHttpRequest();
    }

    /**
     * リクエストパラメータのエンコーディングを変換します。
     *
     * @access public
     * @param string $from_encoding 変換元のエンコーディング
     * @param string $to_encoding 変換先のエンコーディング
     */
    function convertEncoding($from_encoding, $to_encoding)
    {
        mb_convert_variables($to_encoding, $from_encoding,
            $this->_params, $this->_files, $this->_query_string);
    }

    /**
     * リクエストパラメータのフィルタ処理を行います。
     * NULLバイト文字列を除去し、
     * 「magic_quotes_gpc」がONになっていた場合は
     * 「stripslashes」関数を用いて不要なスラッシュを除去します。
     *
     * @access protected
     * @param string|array 処理対象値
     * @return string|array フィルタされた値
     */
    function _filter($var)
    {
        static $magic_quotes_gpc;
        if (is_null($magic_quotes_gpc)) {
            $magic_quotes_gpc = get_magic_quotes_gpc();
        }
        if (is_array($var)) {
            return array_map(array($this, '_filter'), $var);
        }
        $var = str_replace("\0", '', $var);
        if ($magic_quotes_gpc) {
            $var = stripslashes($var);
        }
        return $var;
    }
}
?>
