<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

/**
 * アクションのルーティングを行うクラスです。
 * 下記例のような順序でアクションクラスを探索、処理を起動します。
 *   - foo => Foo
 *   - foo/bar => Foo_Bar, Foo
 *   - foo/bar/baz => Foo_Bar_Baz, Foo_Bar, Foo
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Router
{
    /**
     * ルーティングを行います。
     * アクションクラスが見つからない場合、
     * Moony::setDefaultAction()で設定された
     * デフォルトのアクションを起動アクションとします。
     * 最後まで見つからなかった場合、falseを返します。
     *
     * @access public
     * @param object $moony Moonyのインスタンス（設定）
     * @param string $path_info PATH_INFO文字列
     * @return array|bool 起動するアクションの情報
     */
    function route(&$moony, $path_info)
    {
        if (is_null($moony->action_dir)) {
            // ディレクトリ未設定: action_dir
            trigger_error('Not defined: action_dir', E_USER_ERROR);
            header('HTTP/1.1 500 Internal Server Error');
            exit;
        }

        // PATH_INFOを整形
        $path_info = $this->_refinePathInfo($path_info);

        // PATH_INFOを分割、CamelCase
        $paths = explode('/', $path_info);
        $paths = array_map('ucfirst', $paths);

        $args = array();
        while (count($paths) > 0) {

            $action_path = implode(DIRECTORY_SEPARATOR, $paths);
            $action_name = implode('_', $paths);

            if ($this->_loadAction($action_path, $action_name, $moony)) {
                $action =& new $action_name;
                return array(
                    'action' => &$action, 'args' => array_reverse($args));
            }

            $args[] = strtolower(array_pop($paths));
        }

        if (!is_null($moony->default_action_name)) {
            // デフォルトアクションが設定されていた場合
            $action_path = str_replace(
                '_', DIRECTORY_SEPARATOR, $moony->default_action_name);
            $action_name = $moony->default_action_name;
            if ($this->_loadAction($action_path, $action_name, $moony)) {
                $action =& new $action_name;
                return array(
                    'action' => &$action, 'args' => array_reverse($args));
            }
        }

        return false;
    }

    /**
     * PATH_INFO文字列を整形します。
     * 先頭の「/」を除去、小文字に変換します。
     * 拡張子は除去し、末尾が「/」の場合は「index」を付加します。
     * 
     * @access protected
     * @param string $path_info PATH_INFO文字列
     * @return string 整形結果
     */
    function _refinePathInfo($path_info)
    {
        $path_info = ltrim($path_info, '/');
        $path_info = strtolower($path_info);
        $pos = strpos($path_info, '.');
        if ($pos !== false) {
            $path_info = substr($path_info, 0, $pos);
        }
        if (strlen($path_info) == 0 ||
            substr($path_info, -1, 1) == '/') {
            $path_info .= 'index';
        }
        return $path_info;
    }

    /**
     * アクションクラスをロードします。
     *
     * @access protected
     * @param string $action_path アクションパス（例: Foo/Bar）
     * @param string $action_name アクション名（例: Foo_Bar）
     * @param object $moony Moonyのインスタンス（設定）
     * @return bool ロードに成功したか
     */
    function _loadAction($action_path, $action_name, &$moony)
    {
        $path = $moony->action_dir .
            DIRECTORY_SEPARATOR . $action_path . '.php';
        if (file_exists($path)) {
            include_once $path;
            if (class_exists($action_name)) {
                return true;
            }
        }
        return false;
    }
}
?>
