<?php
/**
 * Moony - the tiny web application framework
 *
 * @package   Moony
 * @author    YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @link      http://moony.sourceforge.jp/
 * @copyright 2005-2006 YAMAOKA Hiroyuki
 * @license   http://opensource.org/licenses/bsd-license.php The BSD License
 */

define('MOONY_SESSION_START_TIME_KEY', 'moony_session_start_time');
define('MOONY_FLASH_VARS_KEY', 'moony_flash_vars');

/**
 * セッションを扱うクラスです。
 *
 * @package Moony
 * @author YAMAOKA Hiroyuki <yamaoka@catwalker.jp>
 * @access public
 */
class Moony_Session
{
    /**
     * セッションが開始しているかどうか
     * @var bool
     */
    var $_is_started = false;

    /**
     * セッションの値を取得します。
     * 該当する値が存在しない場合、$defaultを返します。
     * セッションが開始されていない場合、falseを返します。
     *
     * @access public
     * @param string $name セッションの値の名称
     * @param mixed $default 値が存在しない場合のデフォルト値
     * @return mixed|bool セッションの値
     */
    function get($name, $default = null)
    {
        if (!$this->isStarted()) {
            return false;
        }
        if ($this->exists($name)) {
            return $_SESSION[$name];
        }
        return $default;
    }

    /**
     * セッションに値を設定します。
     * $varが配列の場合、展開して全ての値を設定します。
     * セッションが開始されていない場合、falseを返します。
     *
     * @access public
     * @param string|array $var 値の名称、または値の連想配列
     * @param mixed|bool $value 設定する値
     * @return null|bool セッションが開始されていない場合、false
     */
    function set($var, $value = null)
    {
        if (!$this->isStarted()) {
            return false;
        }
        if (is_array($var)) {
            foreach ($var as $name => $val) {
                $_SESSION[$name] = $val;
            }
        } else {
            $_SESSION[$var] = $value;
        }
    }

    /**
     * 一時的に保存できる領域を扱います。
     * $valueが指定されていない場合、値の取得を行います。
     * 取得した値はセッションから除去されます。
     * $valueが指定されている場合、値の保存を行います。
     * 保存された値は次に取得されるまでセッションに保存されます。
     * セッションが開始されていない場合、falseを返します。
     *
     * @access public
     * @param string $name 値の名称
     * @param mixed $value 保存する値
     * @return mixed|bool 保存されていた値
     */
    function flash($name, $value = null)
    {
        if (!$this->isStarted()) {
            return false;
        }
        if (is_null($value)) {
            if (isset($_SESSION[MOONY_FLASH_VARS_KEY][$name])) {
                $value = $_SESSION[MOONY_FLASH_VARS_KEY][$name];
                unset($_SESSION[MOONY_FLASH_VARS_KEY][$name]);
                return $value;
            }
        } else {
            $_SESSION[MOONY_FLASH_VARS_KEY][$name] = $value;
        }
    }

    /**
     * 指定された名称の値がセッションに
     * 保存されているかどうか調べます。
     * セッションが開始されていない場合、常にfalseを返します。
     *
     * @access public
     * @param string $name セッションの値の名称
     * @return bool セッションに保存されているかどうか
     */
    function exists($name)
    {
        if (!$this->isStarted()) {
            return false;
        }
        return array_key_exists($name, $_SESSION);
    }

    /**
     * セッションから値を除去します。
     * セッションが開始されていない場合、falseを返します。
     *
     * @access public
     * @param string $name 除去する値の名称
     * @return mixed|bool 除去された値
     */
    function remove($name)
    {
        if (!$this->isStarted()) {
            return false;
        }
        if ($this->exists($name)) {
            $value = $_SESSION[$name];
            unset($_SESSION[$name]);
            return $value;
        }
    }

    /**
     * セッションに保存された全ての値を消去します。
     * セッションが開始されていない場合、falseを返します。
     *
     * @access public
     * @return null|bool セッションが開始されていない場合、false
     */
    function clean()
    {
        if (!$this->isStarted()) {
            return false;
        }
        $_SESSION = array();
    }

    /**
     * セッションを開始します。
     *
     * @access public
     */
    function start()
    {
        session_start();
        if (array_key_exists(MOONY_SESSION_START_TIME_KEY, $_SESSION)) {
            // 開始時刻が保存されていない場合、
            // 不正なセッションなので再スタート
            $this->clean();
            $this->regenerateId();
            // 開始時刻を保存
            $this->set(MOONY_SESSION_START_TIME_KEY, time());
        }
        $this->_is_started = true;
    }

    /**
     * セッションが開始しているかどうかを返します。
     *
     * @access public
     * @return bool セッションが開始しているかどうか
     */
    function isStarted()
    {
        return $this->_is_started;
    }

    /**
     * セッションを破棄します。
     * セッションに保存された全ての値は破棄されます。
     * セッションが開始されていない場合、falseを返します。
     *
     * @access public
     * @return null|bool セッションが開始されていない場合、false
     */
    function destroy()
    {
        if (!$this->isStarted()) {
            return false;
        }
        $this->clean();
        session_destroy();
        $this->_is_started = false;
    }

    /**
     * セッションIDを再生成します。
     * セッションが開始されていない場合、falseを返します。
     *
     * @access public
     * @return null|bool セッションが開始されていない場合、false
     */
    function regenerateId()
    {
        if (!$this->isStarted()) {
            return false;
        }

        if (version_compare(PHP_VERSION, '5.1.0') >= 0) {
            // PHP 5.1.0以上の場合はsession_regenerate_id(true)
            session_regenerate_id(true);
            return;
        }

        $saved = $_SESSION;

        $this->destory();
        session_id(md5(uniqid(rand(), true)));
        $this->start();
        $this->set($saved);
    }
}
?>
