/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.platform.bean.workflow.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.constant.MospConst;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.base.PlatformDtoInterface;
import jp.mosp.platform.bean.system.PlatformMasterCheckBeanInterface;
import jp.mosp.platform.bean.workflow.ApprovalUnitReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.ApprovalUnitRegistBeanInterface;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dao.workflow.ApprovalRouteUnitDaoInterface;
import jp.mosp.platform.dao.workflow.ApprovalUnitDaoInterface;
import jp.mosp.platform.dto.workflow.ApprovalRouteUnitDtoInterface;
import jp.mosp.platform.dto.workflow.ApprovalUnitDtoInterface;
import jp.mosp.platform.dto.workflow.impl.PfmApprovalUnitDto;

/**
 * 承認ユニットマスタ登録クラス。
 */
public class ApprovalUnitRegistBean extends PlatformBean implements ApprovalUnitRegistBeanInterface {
	
	/**
	 * 承認ユニットマスタDAO
	 */
	private ApprovalUnitDaoInterface				dao;
	
	/**
	 * 承認ユニットマスタ参照インターフェース。
	 */
	protected ApprovalUnitReferenceBeanInterface	unitReference;
	
	/**
	 * 所属・雇用契約・職位・勤務地マスタに関連する整合性確認インターフェース。
	 */
	protected PlatformMasterCheckBeanInterface		masterCheck;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public ApprovalUnitRegistBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス。
	 * @param connection DBコネクション
	 */
	public ApprovalUnitRegistBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (ApprovalUnitDaoInterface)createDao(ApprovalUnitDaoInterface.class);
		unitReference = (ApprovalUnitReferenceBeanInterface)createBean(ApprovalUnitReferenceBeanInterface.class);
		masterCheck = (PlatformMasterCheckBeanInterface)createBean(PlatformMasterCheckBeanInterface.class);
	}
	
	@Override
	public ApprovalUnitDtoInterface getInitDto() {
		return new PfmApprovalUnitDto();
	}
	
	@Override
	public void add(ApprovalUnitDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 履歴追加情報の検証
		checkAdd(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setPfmApprovalUnitId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
		
	}
	
	@Override
	public void delete(ApprovalUnitDtoInterface dto) throws MospException {
		// レコード識別IDを取得。
		dto.setPfmApprovalUnitId(getRecordID(dto));
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 削除対象ユニット情報が使用されていないかを確認
		checkDelete(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// 論理削除
		logicalDelete(dao, dto.getPfmApprovalUnitId());
		// ロック解除
		unLockTable();
		
	}
	
	@Override
	public void insert(ApprovalUnitDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 新規登録情報の検証
		checkInsert(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setPfmApprovalUnitId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
		
	}
	
	@Override
	public void update(ApprovalUnitDtoInterface dto) throws MospException {
		// レコード識別IDを取得。
		dto.setPfmApprovalUnitId(getRecordID(dto));
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 履歴更新情報の検証
		checkUpdate(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// 論理削除
		logicalDelete(dao, dto.getPfmApprovalUnitId());
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setPfmApprovalUnitId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
		
	}
	
	@Override
	public void update(long[] idArray, Date activateDate, int inactivateFlag) throws MospException {
		// レコード識別ID配列の妥当性確認
		validateAryId(idArray);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 一括更新処理
		for (String code : getCodeList(idArray)) {
			// 対象ユニット情報における有効日の情報を取得
			ApprovalUnitDtoInterface dto = dao.findForKey(code, activateDate);
			// 存在確認(存在しなければ履歴追加、存在すれば履歴更新)
			if (dto == null) {
				// 対象ユニット情報における有効日以前で最新の情報を取得
				dto = dao.findForInfo(code, activateDate);
				// 対象ユニット情報確認
				if (dto == null) {
					// 有効日以前に情報が存在しない場合
					addNoCodeBeforeActivateDateMessage(code);
					continue;
				}
				// DTOに有効日、無効フラグを設定
				dto.setActivateDate(activateDate);
				dto.setInactivateFlag(inactivateFlag);
				// DTOの妥当性確認
				validate(dto);
				// 履歴追加情報の検証
				checkAdd(dto);
				if (mospParams.hasErrorMessage()) {
					// エラーが存在したら履歴追加処理をしない
					continue;
				}
				// レコード識別ID最大値をインクリメントしてDTOに設定
				dto.setPfmApprovalUnitId(findForMaxId(dao) + 1);
				// 登録処理
				dao.insert(dto);
			} else {
				// DTOに無効フラグを設定
				dto.setInactivateFlag(inactivateFlag);
				// DTOの妥当性確認
				validate(dto);
				// 履歴更新情報の検証
				checkUpdate(dto);
				if (mospParams.hasErrorMessage()) {
					// エラーが存在したら履歴更新処理をしない
					continue;
				}
				// 論理削除
				logicalDelete(dao, dto.getPfmApprovalUnitId());
				// レコード識別ID最大値をインクリメントしてDTOに設定
				dto.setPfmApprovalUnitId(findForMaxId(dao) + 1);
				// 登録処理
				dao.insert(dto);
			}
		}
		// ロック解除
		unLockTable();
		
	}
	
	/**
	 * ユニットコードリストを取得する。<br>
	 * 同時に排他確認を行う。<br>
	 * @param idArray レコード識別ID配列
	 * @return ユニットコードリスト
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected List<String> getCodeList(long[] idArray) throws MospException {
		// リスト準備
		List<String> list = new ArrayList<String>();
		// レコード識別IDからDTOを取得し、コードをリストへ追加
		for (long id : idArray) {
			// レコード識別IDから対象DTOを取得
			ApprovalUnitDtoInterface dto = (ApprovalUnitDtoInterface)dao.findForKey(id, false);
			// 排他確認
			checkExclusive(dto);
			// 対象コードをリストへ追加
			list.add(dto.getUnitCode());
		}
		return list;
		
	}
	
	/**
	 * 履歴追加時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	private void checkAdd(ApprovalUnitDtoInterface dto) throws MospException {
		// 対象レコードの有効日が重複していないかを確認
		checkDuplicateAdd(dao.findForKey(dto.getUnitCode(), dto.getActivateDate()));
		// 無効フラグ確認
		if (isDtoActivate(dto)) {
			return;
		}
		// 履歴追加対象コードの履歴情報を取得
		List<ApprovalUnitDtoInterface> list = dao.findForHistory(dto.getUnitCode());
		// 生じる無効期間による履歴追加確認要否を取得
		if (needCheckTermForAdd(dto, list) == false) {
			// 無効期間は発生しない
			return;
		}
		// 確認するべき承認ルートユニットマスタリストを取得
		List<ApprovalRouteUnitDtoInterface> routeList = getApprovalRouteListForCheck(dto, list);
		// コード使用確認
		checkCodeIsUsed(dto.getUnitCode(), routeList);
		
	}
	
	/**
	 * 登録情報の妥当性を確認する。
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	private void validate(ApprovalUnitDtoInterface dto) throws MospException {
		// 無効の場合
		if (dto.getInactivateFlag() == MospConst.INACTIVATE_FLAG_ON) {
			// 確認なし
			return;
		}
		// 履歴一覧取得
		List<ApprovalUnitDtoInterface> list = unitReference.getApprovalUnitHistory(dto.getUnitCode());
		// 期間取得
		Date startDate = dto.getActivateDate();
		Date endDate = getEffectiveLastDate(dto.getActivateDate(), list);
		// 所属存在確認
		masterCheck.isCheckSection(dto.getApproverSectionCode(), startDate, endDate);
		// 職位存在確認
		masterCheck.isCheckPosition(dto.getApproverPositionCode(), startDate, endDate);
	}
	
	/**
	 * 削除時の確認処理を行う。<br>
	 * 削除対象ユニット情報を設定している設定適用管理情報がないかの確認を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	private void checkDelete(ApprovalUnitDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getPfmApprovalUnitId());
		// 削除時の所属・職位での整合性確認
		checkMasterDelete(dto);
		// 削除元データの無効フラグ確認
		// 画面上の無効フラグは変更可能であるため確認しない。
		if (isDtoActivate(dao.findForKey(dto.getPfmApprovalUnitId(), true)) == false) {
			// 削除元データが無効であれば無効期間は発生しない
			return;
		}
		// 削除対象コードの履歴情報を取得
		List<ApprovalUnitDtoInterface> list = dao.findForHistory(dto.getUnitCode());
		// 生じる無効期間による削除確認要否を取得
		if (needCheckTermForDelete(dto, list) == false) {
			// 無効期間は発生しない
			return;
		}
		// 確認するべき承認ルートユニットマスタリストを取得
		List<ApprovalRouteUnitDtoInterface> routeUnitList = getApprovalRouteListForCheck(dto, list);
		// コード使用確認
		checkCodeIsUsed(dto.getUnitCode(), routeUnitList);
		
	}
	
	/**
	 * 削除時の所属・職位整合性を確認する。<br>
	 * 削除により一つ前の履歴が活きてくる場合、
	 * 影響が及ぶ期間に対して、その所属～勤務地の存在(有効)確認を行う。<br>
	 * @param dto 削除対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	private void checkMasterDelete(ApprovalUnitDtoInterface dto) throws MospException {
		// 一つ前の履歴取得
		ApprovalUnitDtoInterface beforeDto = unitReference.getApprovalUnitInfo(dto.getUnitCode(), DateUtility.addDay(
				dto.getActivateDate(), -1));
		// 一つ前の履歴が存在しないか無効である場合
		if (beforeDto == null || beforeDto.getInactivateFlag() == MospConst.INACTIVATE_FLAG_ON) {
			// 確認不要
			return;
		}
		// 対象コードの履歴を取得
		List<ApprovalUnitDtoInterface> list = unitReference.getApprovalUnitHistory(dto.getUnitCode());
		// 情報が影響を及ぼす期間を取得
		Date startDate = beforeDto.getActivateDate();
		Date endDate = getEffectiveLastDate(dto.getActivateDate(), list);
		// 所属存在確認
		masterCheck.isCheckSection(beforeDto.getApproverSectionCode(), startDate, endDate);
		// 職位存在確認
		masterCheck.isCheckPosition(beforeDto.getApproverPositionCode(), startDate, endDate);
		
	}
	
	/**
	 * 新規登録時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	private void checkInsert(ApprovalUnitDtoInterface dto) throws MospException {
		// 対象レコードの有効日が重複していないかを確認
		checkDuplicateInsert(dao.findForHistory(dto.getUnitCode()));
		
	}
	
	/**
	 * 履歴更新時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	private void checkUpdate(ApprovalUnitDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getPfmApprovalUnitId());
		// 無効フラグ確認
		if (isDtoActivate(dto)) {
			return;
		}
		// 更新元データの無効フラグ確認
		if (isDtoActivate(dao.findForKey(dto.getPfmApprovalUnitId(), true)) == false) {
			// 更新元データが更新前から無効であれば無効期間は発生しない
			return;
		}
		// 履歴追加対象コードの履歴情報を取得
		List<ApprovalUnitDtoInterface> list = dao.findForHistory(dto.getUnitCode());
		// 確認するべき承認ルートユニットマスタリストを取得
		List<ApprovalRouteUnitDtoInterface> routeUnitList = getApprovalRouteListForCheck(dto, list);
		// コード使用確認
		checkCodeIsUsed(dto.getUnitCode(), routeUnitList);
		
	}
	
	/**
	 * レコード識別IDを取得する。<br>
	 * @param dto 	対象DTO
	 * @return レコード識別ID
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	private long getRecordID(ApprovalUnitDtoInterface dto) throws MospException {
		// ユニット情報を取得する。
		ApprovalUnitDtoInterface subDto = dao.findForKey(dto.getUnitCode(), dto.getActivateDate());
		// レコード識別IDを返す。
		return subDto.getPfmApprovalUnitId();
		
	}
	
	/**
	 * 承認ルートユニットマスタリスト内に対象コードが使用されている情報がないかの確認を行う。<br>
	 * @param code 対象コード
	 * @param list 承認ルートユニットマスタリスト
	 */
	protected void checkCodeIsUsed(String code, List<ApprovalRouteUnitDtoInterface> list) {
		// 同一のルートコードのメッセージは出力しない。
		String codeAdded = "";
		// 承認ルートユニットマスタリストの中身を確認
		for (ApprovalRouteUnitDtoInterface dto : list) {
			// 対象コード確認
			if ((code.equals(dto.getUnitCode())) && (isDtoActivate(dto))) {
				// 同一のルートコードのメッセージは出力しない。
				if (!codeAdded.equals(dto.getRouteCode())) {
					// メッセージ設定
					addUnitCodeIsUsedMessage(code, dto);
					// メッセージに設定したルートコードを保持
					codeAdded = dto.getRouteCode();
				}
			}
		}
	}
	
	/**
	 * 確認すべき承認ルートユニットマスタリストを取得する。<br>
	 * 対象DTOの有効日以前で最新の承認ルートユニットマスタリストと
	 * 対象DTOの有効日～対象DTOの次の履歴の有効日に有効日が含まれる
	 * 承認ルートユニットマスタリストを合わせたリストを取得する。<br>
	 * 対象コード履歴リストは、有効日の昇順で並んでいるものとする。<br>
	 * 各種マスタ操作時に生じる無効期間におけるコード使用確認等で用いられる。<br>
	 * @param dto 	対象DTO
	 * @param list 対象コード履歴リスト
	 * @return 承認ルートマスタリスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected List<ApprovalRouteUnitDtoInterface> getApprovalRouteListForCheck(PlatformDtoInterface dto,
			List<? extends PlatformDtoInterface> list) throws MospException {
		// 承認ルートユニットマスタDAO取得
		ApprovalRouteUnitDaoInterface routeUnitDao;
		routeUnitDao = (ApprovalRouteUnitDaoInterface)createDao(ApprovalRouteUnitDaoInterface.class);
		// 削除対象の有効日以前で最新の承認ルートユニットマスタ情報を取得
		List<ApprovalRouteUnitDtoInterface> routeUnitList = routeUnitDao.findForActivateDate(dto.getActivateDate());
		// 無効期間で承認ルートユニットマスタ履歴情報を取得(対象DTOの有効日～次の履歴の有効日)
		routeUnitList.addAll(routeUnitDao.findForTerm(dto.getActivateDate(), getNextActivateDate(dto.getActivateDate(),
				list)));
		return routeUnitList;
	}
	
	/**
	 * 該当コードが使用されていた場合の警告メッセージを追加する。
	 * {@link #mospParams}に追加する。<br>
	 * @param code コード
	 * @param dto  承認ルートユニットマスタDTO
	 */
	protected void addUnitCodeIsUsedMessage(String code, ApprovalRouteUnitDtoInterface dto) {
		mospParams.addErrorMessage(PlatformMessageConst.MSG_UNIT_CODE_IS_USED, code, dto.getRouteCode());
	}
	
}
