/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.ApplicationReferenceBeanInterface;
import jp.mosp.time.bean.TimeSettingReferenceBeanInterface;
import jp.mosp.time.constant.TimeMessageConst;
import jp.mosp.time.dao.settings.TimeSettingDaoInterface;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;

/**
 * 勤怠設定参照クラス。
 */
public class TimeSettingReferenceBean extends TimeBean implements TimeSettingReferenceBeanInterface {
	
	/**
	 * 勤怠設定管理DAO
	 */
	private TimeSettingDaoInterface				dao;
	
	private ApplicationReferenceBeanInterface	applicationReference;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public TimeSettingReferenceBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosP処理情報
	 * @param connection DBコネクション
	 */
	public TimeSettingReferenceBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (TimeSettingDaoInterface)createDao(TimeSettingDaoInterface.class);
		applicationReference = (ApplicationReferenceBeanInterface)createBean(ApplicationReferenceBeanInterface.class);
	}
	
	@Override
	public TimeSettingDtoInterface getTimeSettingInfo(String workSettingCode, Date targetDate) throws MospException {
		return dao.findForInfo(workSettingCode, targetDate);
	}
	
	@Override
	public List<TimeSettingDtoInterface> getTimeSettingHistory(String timeSettingCode) throws MospException {
		return dao.findForHistory(timeSettingCode);
	}
	
	@Override
	public String getTimeSettingAbbr(String workSettingCode, Date targetDate) throws MospException {
		TimeSettingDtoInterface dto = getTimeSettingInfo(workSettingCode, targetDate);
		if (dto == null) {
			return workSettingCode;
		}
		return dto.getWorkSettingAbbr();
	}
	
	@Override
	public String[][] getCodedSelectArray(Date targetDate, boolean needBlank) throws MospException {
		// プルダウン用配列取得(コード+名称)
		return getSelectArray(targetDate, needBlank, true, true);
	}
	
	@Override
	public TimeSettingDtoInterface findForKey(String workSettingCode, Date activateDate) throws MospException {
		return dao.findForKey(workSettingCode, activateDate);
	}
	
	/**
	 * プルダウン用配列を取得する。<br>
	 * @param targetDate 対象年月日
	 * @param needBlank 空白行要否(true：空白行要、false：空白行不要)
	 * @param isName 名称表示(true：名称表示、false：略称表示)
	 * @param viewCode コード表示(true：コード表示、false：コード非表示)
	 * @return プルダウン用配列
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected String[][] getSelectArray(Date targetDate, boolean needBlank, boolean isName, boolean viewCode)
			throws MospException {
		// 一覧取得
		List<TimeSettingDtoInterface> list = dao.findForActivateDate(targetDate);
		// 一覧件数確認
		if (list.size() == 0) {
			// 対象データ無し
			return getNoObjectDataPulldown();
		}
		// コード最大長取得
		int length = getMaxCodeLength(list, viewCode);
		// プルダウン用配列及びインデックス準備
		String[][] array = prepareSelectArray(list.size(), needBlank);
		int idx = needBlank ? 1 : 0;
		// プルダウン用配列作成
		for (TimeSettingDtoInterface dto : list) {
			// コード設定
			array[idx][0] = dto.getWorkSettingCode();
			// 表示内容設定
			if (isName && viewCode) {
				// コード+名称
				array[idx++][1] = getCodedName(dto.getWorkSettingCode(), dto.getWorkSettingName(), length);
			} else if (isName) {
				// 名称
				array[idx++][1] = dto.getWorkSettingName();
			} else if (viewCode) {
				// コード+略称
				array[idx++][1] = getCodedName(dto.getWorkSettingCode(), dto.getWorkSettingAbbr(), length);
			} else {
				// 略称
				array[idx++][1] = dto.getWorkSettingAbbr();
			}
		}
		return array;
	}
	
	/**
	 * リスト中のDTOにおけるコード最大文字数を取得する。<br>
	 * @param list     対象リスト
	 * @param viewCode コード表示(true：コード表示、false：コード非表示)
	 * @return リスト中のDTOにおけるコード最大文字数
	 */
	protected int getMaxCodeLength(List<TimeSettingDtoInterface> list, boolean viewCode) {
		// コード表示確認
		if (viewCode == false) {
			return 0;
		}
		// コード最大文字数
		int length = 0;
		// コード最大文字数確認
		for (TimeSettingDtoInterface dto : list) {
			if (dto.getWorkSettingCode().length() > length) {
				length = dto.getWorkSettingCode().length();
			}
		}
		return length;
	}
	
	@Override
	public List<String> getWorkSettingCode(String cutoffcode, Date activateDate) throws MospException {
		// 勤怠設定コードリスト準備
		List<String> workSettingCodeList = new ArrayList<String>();
		// 対象の勤怠設定管理DTOを取得
		List<TimeSettingDtoInterface> list = dao.findForInfoList(cutoffcode, activateDate);
		// データ存在チェック
		if (list.isEmpty()) {
			return workSettingCodeList;
		}
		// 勤怠設定コードを取得する
		for (int i = 0; i < list.size(); i++) {
			workSettingCodeList.add(list.get(i).getWorkSettingCode());
		}
		return workSettingCodeList;
	}
	
	@Override
	public int getGeneralWorkTime(String workSettingCode, Date targetDate) throws MospException {
		TimeSettingDtoInterface dto = getTimeSettingInfo(workSettingCode, targetDate);
		if (dto == null) {
			return 8 * 60;
		}
		return DateUtility.getHour(dto.getGeneralWorkTime()) * 60 + DateUtility.getMinute(dto.getGeneralWorkTime());
	}
	
	@Override
	public int getGeneralWorkHour(String workSettingCode, Date targetDate) throws MospException {
		TimeSettingDtoInterface dto = getTimeSettingInfo(workSettingCode, targetDate);
		if (dto == null) {
			return 8;
		}
		int hour = DateUtility.getHour(dto.getGeneralWorkTime());
		if (DateUtility.getMinute(dto.getGeneralWorkTime()) == 0) {
			return hour;
		}
		return hour + 1;
	}
	
	@Override
	public void chkExistTimeSetting(TimeSettingDtoInterface dto, Date targetDate) {
		if (dto == null) {
			String errorMes1 = "";
			if (targetDate == null) {
				errorMes1 = DateUtility.getStringDate(DateUtility.getSystemDate());
			} else {
				errorMes1 = DateUtility.getStringDate(targetDate);
			}
			String errorMes2 = mospParams.getName("WorkManage") + mospParams.getName("Set")
					+ mospParams.getName("Information");
			mospParams.addErrorMessage(TimeMessageConst.MSG_SETTING_APPLICATION_DEFECT, errorMes1, errorMes2);
		}
	}
	
	@Override
	public Integer getBeforeOvertimeFlag(String personalId, Date targetDate) throws MospException {
		ApplicationDtoInterface applicationDto = applicationReference.findForPerson(personalId, targetDate);
		if (applicationDto == null) {
			addApplicationNotExistErrorMessage(targetDate);
			return null;
		}
		TimeSettingDtoInterface dto = dao.findForInfo(applicationDto.getWorkSettingCode(), targetDate);
		if (dto == null) {
			addTimeSettingNotExistErrorMessage(targetDate);
			return null;
		}
		return dto.getBeforeOvertimeFlag();
	}
	
	@Override
	public String[][] getSelectArray(Date targetDate, boolean needBlank) throws MospException {
		// プルダウン用配列取得(略称)
		return getSelectArray(targetDate, needBlank, false, false);
	}
}
