/*
 * Copyright (C) 2010-2011 Mtzky.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.io;

import static org.mtzky.log.GenericMarker.*;
import static org.mtzky.log.ThreadUtils.*;
import static org.slf4j.LoggerFactory.*;

import java.io.Closeable;
import java.io.IOException;
import java.io.PrintStream;

import org.mtzky.UtilConstractedAssertionError;
import org.slf4j.Logger;

/**
 * <p>
 * I/O utilities.
 * </p>
 * 
 * @see Closable
 * @see Closeable
 * @author mtzky
 */
public class IOUtils {

	private static final Logger LOG = getLogger(IOUtils.class);

	private IOUtils() {
		throw new UtilConstractedAssertionError(IOUtils.class);
	}

	/**
	 * <p>
	 * Unconditionally close a {@link Closable}.
	 * </p>
	 * <p>
	 * Logs {@link Logger#warn(org.slf4j.Marker, String) warning} message to
	 * {@link Logger} if {@link IOException} occurred.
	 * </p>
	 * 
	 * @param closable
	 */
	public static void closeQuietly(final Closable closable) {
		if (closable == null) {
			return;
		}
		if (closable.isClosed()) {
			return;
		}
		try {
			closable.close();
		} catch (final IOException e) {
			final Object[] args = { closable, getCallingStackTraceElement() };
			LOG.warn(FAILED_TO_CLOSE, "FAILED to close {} at {}", args);
		}
	}

	/**
	 * <p>
	 * Unconditionally close a {@link Closeable}.
	 * </p>
	 * <p>
	 * Logs {@link Logger#warn(org.slf4j.Marker, String) warning} message to
	 * {@link Logger} if {@link IOException} occurred.
	 * </p>
	 * 
	 * @param closaable
	 */
	public static void closeQuietly(final Closeable closaable) {
		if (closaable == null) {
			return;
		}
		try {
			closaable.close();
		} catch (final IOException e) {
			final Object[] args = { closaable, getCallingStackTraceElement() };
			LOG.warn(FAILED_TO_CLOSE, "FAILED to close {} at {}", args);
		}
	}

	/**
	 * <p lang="ja">
	 * 引数の {@link Runnable} 実行中の {@link System#out} を {@link String} として返します。
	 * </p>
	 * 
	 * @param runnable
	 * @return {@link System#out} as {@link String}
	 * @throws Throwable
	 */
	public static String trapSysOut(final Runnable runnable) throws Throwable {
		final PrintStream originalOut = System.out;
		final PrintStream out = new SPrintStream();
		try {
			System.setOut(out);
			runnable.run();
		} catch (final Throwable e) {
			out.flush();
			originalOut.println(out);
			throw e;
		} finally {
			out.flush();
			System.setOut(originalOut);
		}
		return out.toString();
	}

}
