/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.reflect;

import static java.lang.String.*;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

/**
 * @author mtzky
 */
public class PropDesc {

	private Method getter;
	private Method setter;
	private Field field;
	private final Map<Class<? extends Annotation>, Annotation> annotations = new HashMap<Class<? extends Annotation>, Annotation>();
	private final String name;

	public PropDesc(final String name) {
		this.name = name;
	}

	/**
	 * <p>
	 * Gets a property value from the object. Returns {@code null} if an
	 * {@link Exception} occurred.
	 * </p>
	 * 
	 * @param <T>
	 * @param obj
	 *            the instance to get property value
	 * @return property value
	 * @throws InvocationTargetRuntimeException
	 *             if failed to get
	 */
	@SuppressWarnings("unchecked")
	public <T> T get(final Object obj) {
		try {
			return (T) (getter != null ? getter.invoke(obj) : field.get(obj));
		} catch (final Exception e) {
			final String fmt = "FAILED to get a value from the property '%s'";
			throw new InvocationTargetRuntimeException(format(fmt, name), e);
		}
	}

	/**
	 * <p>
	 * Sets a property value to the object. Returns {@code false} if an
	 * {@link Exception} occurred.
	 * </p>
	 * 
	 * @param obj
	 *            the instance to set property value
	 * @param value
	 * @throws InvocationTargetRuntimeException
	 *             if failed to set
	 */
	public void set(final Object obj, final Object value) {
		try {
			if (setter != null) {
				setter.invoke(obj, value);
			} else {
				field.set(obj, value);
			}
		} catch (final Exception e) {
			final String fmt = "FAILED to set [%s] to the property '%s'";
			final Object[] args = { value, name };
			throw new InvocationTargetRuntimeException(format(fmt, args), e);
		}
	}

	/**
	 * @return property type
	 */
	public Class<?> getType() {
		if (getter != null) {
			return getter.getReturnType();
		}
		return field.getType();
	}

	public String getName() {
		return name;
	}

	public boolean hasGetter() {
		return getter != null;
	}

	public boolean hasSetter() {
		return getter != null;
	}

	public boolean hasField() {
		return field != null;
	}

	public void setGetter(final Method getter) {
		this.getter = getter;
	}

	public void setSetter(final Method setter) {
		this.setter = setter;
	}

	public void setField(final Field field) {
		this.field = field;
	}

	public Annotation[] getAnnotations() {
		return annotations.values().toArray(new Annotation[annotations.size()]);
	}

	@SuppressWarnings("unchecked")
	public <T extends Annotation> T getAnnotation(final Class<T> annotationClass) {
		return (T) annotations.get(annotationClass);
	}

	public void addAnnotations(final Annotation[] annotations) {
		for (final Annotation annotation : annotations) {
			addAnnotation(annotation);
		}
	}

	public void addAnnotation(final Annotation annotation) {
		annotations.put(annotation.annotationType(), annotation);
	}

}
