/*
 * Copyright (C) 2010-2011 Mtzky.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.log;

import java.util.Iterator;

import org.slf4j.Marker;
import org.slf4j.MarkerFactory;

/**
 * <p>
 * This {@link Enum} defines commonly-used {@link Marker}s for <a
 * href="http://www.slf4j.org/">SLF4J</a>.
 * </p>
 * <p lang="ja">
 * </p>
 * <h3>{@link #ROOT} Marker</h3>
 * <p>
 * This marker named {@code "org.mtzky.log"} is referenced by the following:
 * </p>
 * <dl>
 * <dt>{@link #LIFECYCLE} ({@code "org.mtzky.log.LIFECYCLE"})</dt>
 * <dd lang="ja">オブジェクトライフサイクルを意味するマーカです。主にフレームワークやライブラリで利用されます。</dd>
 * <dt>{@link #PROCESS} ({@code "org.mtzky.log.PROCESS"})</dt>
 * <dd lang="ja">アプリケーションの処理を意味するマーカです。</dd>
 * <dt>{@link #PROFILE} ({@code "org.mtzky.log.PROFILE"})</dt>
 * <dd lang="ja">パフォーマンス計測用のマーカです。</dd>
 * <dt>{@link #FAILED} ({@code "org.mtzky.log.FAILED"})</dt>
 * <dd lang="ja">処理が失敗したことを意味するマーカです。以下を参照のこと。</dd>
 * </dl>
 * <h4>{@link #FAILED} Marker</h4>
 * <p lang="ja">
 * 処理が失敗したことを意味するマーカです。各階層のマーカには、対となる {@link #FAILED} を持ったマーカがあります。例えば
 * </p>
 * <ul>
 * <li>{@link #PROCESS} and {@link #FAILED_PROCESS}</li>
 * <li>{@link #INITIALIZE} and {@link #FAILED_TO_INITIALIZE}</li>
 * </ul>
 * <h3>Logging with Marker</h3>
 * <p lang="ja">
 * もし <a href="http://logback.qos.ch/">Logback</a> を使用している場合、この {@link Enum}
 * に定義されたマーカを使用してログ出力を制御する事ができます。例えば、logback(-test).xml について、
 * ライフサイクルを意味するマーカのうち失敗したものだけをログ出力する設定を加える場合は以下のようになります。
 * </p>
 * 
 * <pre>
 * {@literal <configuration>}
 *     ...
 *     {@literal <turboFilter class="ch.qos.logback.classic.turbo.MarkerFilter">}
 *         {@literal <Marker>org.mtzky.log</Marker>}
 *         {@literal <OnMatch>ACCEPT</OnMatch>}
 *     {@literal </turboFilter>}
 *     {@literal <turboFilter class="ch.qos.logback.classic.turbo.MarkerFilter">}
 *         {@literal <Marker>org.mtzky.log.}{@link #LIFECYCLE}{@literal </Marker>}
 *         {@literal <OnMatch>DENY</OnMatch>}
 *     {@literal </turboFilter>}
 *     {@literal <turboFilter class="ch.qos.logback.classic.turbo.MarkerFilter">}
 *         {@literal <Marker>org.mtzky.log.}{@link #FAILED_LIFECYCLE}{@literal </Marker>}
 *         {@literal <OnMatch>ACCEPT</OnMatch>}
 *     {@literal </turboFilter>}
 *     ...
 * {@literal </configuration>}
 * </pre>
 * 
 * @author mtzky
 */
public enum GenericMarker implements Marker {

	/**
	 * <p>
	 * Root marker
	 * </p>
	 */
	ROOT(),

	/**
	 * <p>
	 * Failed processing marker
	 * </p>
	 * 
	 * @see #ROOT
	 */
	FAILED(ROOT),

	/* ================================================================ */

	/**
	 * <p>
	 * Object life cycle processing marker
	 * </p>
	 * 
	 * @see #ROOT
	 */
	LIFECYCLE(ROOT),

	/**
	 * <p>
	 * Failed object life cycle processing marker
	 * </p>
	 * 
	 * @see #FAILED
	 * @see #LIFECYCLE
	 */
	FAILED_LIFECYCLE(FAILED, LIFECYCLE),

	/**
	 * <p>
	 * Application processing marker
	 * </p>
	 * 
	 * @see #ROOT
	 */
	PROCESS(ROOT),

	/**
	 * <p>
	 * Failed application processing marker
	 * </p>
	 * 
	 * @see #FAILED
	 * @see #PROCESS
	 */
	FAILED_PROCESS(FAILED, PROCESS),

	/**
	 * <p>
	 * Performance profile marker
	 * </p>
	 * 
	 * @see #ROOT
	 */
	PROFILE(ROOT),

	/* ================================================================ */

	/**
	 * <p>
	 * Initializing process marker
	 * </p>
	 * 
	 * @see #LIFECYCLE
	 */
	INITIALIZE(LIFECYCLE),

	/**
	 * <p>
	 * Failed initializing process marker
	 * </p>
	 * 
	 * @see #FAILED_LIFECYCLE
	 * @see #INITIALIZE
	 */
	FAILED_TO_INITIALIZE(FAILED_LIFECYCLE, INITIALIZE),

	/**
	 * <p>
	 * Canceling process marker
	 * </p>
	 * 
	 * @see #LIFECYCLE
	 */
	CANCEL(LIFECYCLE),

	/**
	 * <p>
	 * Failed canceling process marker
	 * </p>
	 * 
	 * @see #FAILED_LIFECYCLE
	 * @see #CANCEL
	 */
	FAILED_TO_CANCEL(FAILED_LIFECYCLE, CANCEL),

	/**
	 * <p>
	 * Opening process marker
	 * </p>
	 * 
	 * @see #LIFECYCLE
	 * @since 0.1.3
	 */
	OPEN(LIFECYCLE),

	/**
	 * <p>
	 * Failed opening process marker
	 * </p>
	 * 
	 * @see #FAILED_LIFECYCLE
	 * @see #OPEN
	 * @since 0.1.3
	 */
	FAILED_TO_OPEN(FAILED_LIFECYCLE, OPEN),

	/**
	 * <p>
	 * Closing process marker
	 * </p>
	 * 
	 * @see #LIFECYCLE
	 */
	CLOSE(LIFECYCLE),

	/**
	 * <p>
	 * Failed closing process marker
	 * </p>
	 * 
	 * @see #FAILED_LIFECYCLE
	 * @see #CLOSE
	 */
	FAILED_TO_CLOSE(FAILED_LIFECYCLE, CLOSE),

	/**
	 * <p>
	 * Destroying process marker
	 * </p>
	 * 
	 * @see #LIFECYCLE
	 */
	DESTROY(LIFECYCLE),

	/**
	 * <p>
	 * Failed destroying process marker
	 * </p>
	 * 
	 * @see #FAILED_LIFECYCLE
	 * @see #DESTROY
	 */
	FAILED_TO_DESTROY(FAILED_LIFECYCLE, DESTROY),

	/* ================================================================ */

	/**
	 * <p>
	 * Adding process marker
	 * </p>
	 * 
	 * @see #PROCESS
	 */
	ADD(PROCESS),

	/**
	 * <p>
	 * Failed adding process marker
	 * </p>
	 * 
	 * @see #FAILED_PROCESS
	 * @see #ADD
	 */
	FAILED_TO_ADD(FAILED_PROCESS, ADD),

	/**
	 * <p>
	 * Reading process marker
	 * </p>
	 * 
	 * @see #PROCESS
	 */
	READ(PROCESS),

	/**
	 * <p>
	 * Failed reading process marker
	 * </p>
	 * 
	 * @see #FAILED_PROCESS
	 * @see #READ
	 */
	FAILED_TO_READ(FAILED_PROCESS, READ),

	/**
	 * <p>
	 * Changing process marker
	 * </p>
	 * 
	 * @see #PROCESS
	 */
	CHANGE(PROCESS),

	/**
	 * <p>
	 * Failed changing process marker
	 * </p>
	 * 
	 * @see #FAILED_PROCESS
	 * @see #CHANGE
	 */
	FAILED_TO_CHANGE(FAILED_PROCESS, CHANGE),

	/**
	 * <p>
	 * Deleting process marker
	 * </p>
	 * 
	 * @see #PROCESS
	 */
	DELETE(PROCESS),

	/**
	 * <p>
	 * Failed deleting process marker
	 * </p>
	 * 
	 * @see #FAILED_PROCESS
	 * @see #DELETE
	 */
	FAILED_TO_DELETE(FAILED_PROCESS, DELETE),

	/* ================================================================ */

	/**
	 * <p>
	 * Parsing process marker
	 * </p>
	 * 
	 * @see #READ
	 */
	PARSE(READ),

	/**
	 * <p>
	 * Failed parsing process marker
	 * </p>
	 * 
	 * @see #FAILED_TO_READ
	 * @see #PARSE
	 */
	FAILED_TO_PARSE(FAILED_TO_READ, PARSE),

	/**
	 * <p>
	 * Showing process marker
	 * </p>
	 * 
	 * @see #READ
	 */
	SHOW(READ),

	/**
	 * <p>
	 * Failed showing process marker
	 * </p>
	 * 
	 * @see #FAILED_TO_READ
	 * @see #SHOW
	 */
	FAILED_TO_SHOW(FAILED_TO_READ, SHOW),

	/**
	 * <p>
	 * Listing process marker
	 * </p>
	 * 
	 * @see #READ
	 */
	LIST(READ),

	/**
	 * <p>
	 * Failed listing process marker
	 * </p>
	 * 
	 * @see #FAILED_TO_READ
	 * @see #LIST
	 */
	FAILED_TO_LIST(FAILED_TO_READ, LIST),

	/* ================================================================ */

	;

	private final Marker marker;

	private GenericMarker(final Marker... parents) {
		final StringBuilder name = new StringBuilder();
		name.append(GenericMarker.class.getPackage().getName());
		if (0 < parents.length) {
			name.append('.').append(name());
		}
		this.marker = MarkerFactory.getMarker(name.toString());
		for (final Marker parent : parents) {
			parent.add(this.marker);
		}
	}

	@Override
	public void add(final Marker reference) {
		marker.add(reference);
	}

	@Override
	public boolean contains(final Marker other) {
		return marker.contains(other);
	}

	@Override
	public boolean contains(final String name) {
		return marker.contains(name);
	}

	@Override
	public String getName() {
		return marker.getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	@Deprecated
	public boolean hasChildren() {
		return marker.hasReferences();
	}

	@Override
	public boolean hasReferences() {
		return marker.hasReferences();
	}

	@Override
	@SuppressWarnings("unchecked")
	public Iterator<Marker> iterator() {
		return marker.iterator();
	}

	@Override
	public boolean remove(final Marker reference) {
		return marker.remove(reference);
	}

}
